/*  -*- c++ -*-  (for Emacs)
 *
 *  splitcollectiondialog.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Sun Apr 23 2006.
 *  Copyright (c) 2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

// TODO: use global vars for table names and field identifiers

#include "splitcollectiondialog.h"

#include <cmath>
#include <cstdlib>

#include <QDebug> // TODO: remove me
#include <QHash>
#include <QList>
#include <QMessageBox>
#include <QSet>
#include <QSqlDatabase>
#include <QSqlQuery>

#include <MvcDigestDb/dcollectionrecord.h>
#include <MvcDigestDb/digestdbcontroller.h>
#include <MvcDigestDb/digestdbmodel.h>


/*!
 * \class SplitCollectionDialog
 *
 * \brief The SplitCollectionDialog class provides a dialog and the DB logic to
 *        take a collection and automatically create training and test sets from
 *        it.
 *
 * The dialog does not yet handle changes to the database while the window is open.
 * Thus, it should only be executed as application-modal, not window-model.
 */


/*!
 * Constructs a new dialog for splitting collections.
 *
 * \b Note: The collectionId must be a positive integer and the
 *          \em controller must be non-null.
 *
 * \b Note: If the collectionId is not valid, the the user will be alerted with
 *          a critical error message, as this never be allowed to occur, but if
 *          it does, we wan't to fail gracefully.
 */
SplitCollectionDialog::SplitCollectionDialog( int collectionId,
					      DigestDbController* controller,
					      QWidget* parent, Qt::WFlags f )
  : QDialog(parent, f),
    AbstractDigestDbView(this),
    m_srcCollectionId(collectionId),
    m_trainCollectionId(-1),
    m_testCollectionId(-1),
    c_totalGestures(0)
{
  Q_ASSERT( collectionId >= 0 );
  Q_ASSERT( controller );
  m_ui.setupUi(this);
  controller->attachView( this );

  // TODO: only show dialog once reset has been executed - so user doesn't see it being setup
}


void SplitCollectionDialog::accept()
{
  if ( digestDbModel() )
    {
      // Request that two new collections be made - one for training, one for testing

      // Create unique collection labels - both sets will ALWAYS share any required
      // numerical suffix - easier to match - even if there are gaps in numbers.
      QString trainLabel = c_label + tr(" - training");
      QString testLabel  = c_label + tr(" - testing");
      int suffixNum = 0; // 0 == not init'd, 1 == special case of no suffix
      QString suffix;
      do {
	// TODO: fix this, it's not preventing non-unqiue labels!
	++suffixNum;
	suffix = (suffixNum>1) ? QString(" %1").arg(suffixNum) : "";
      } while( digestDbModel()->collectionLabelExists(trainLabel + suffix)
	       || digestDbModel()->collectionLabelExists(testLabel + suffix) );

      DCollectionRecord rec; // ID to be generated
      rec.label = trainLabel + suffix;
      postControllerEvent( new CCollectionAddEvent(rec, this) );
      rec.label = testLabel + suffix;
      postControllerEvent( new CCollectionAddEvent(rec, this) );
    }

  // Delay until we're finished - see end of collectionsEvent():
  // QDialog::accept(); // hides dialog

  // TODO: add timeout - failsafe option?
  // TODO: add progress bar - or event just a basic indicator?
}


void SplitCollectionDialog::resetEvent( VEvent* event )
{
  AbstractDigestDbView::resetEvent(event);

  if ( !digestDbModel() ) return;

  /*
   * Try to gather details about the collection.
   */

  QSqlQuery q( database() );

  q.exec( "SELECT label FROM Collection WHERE id="
	  + QString::number(m_srcCollectionId) );
  if ( q.next() ) {
    c_label = q.value(0).toString();
  } else {
    QMessageBox::critical( this, QString(),
			   tr("Internal Error: "
			      "Invalid collection ID requested to be split: %1")
			   .arg(m_srcCollectionId) );
    return;
  }

  q.exec( "SELECT COUNT() FROM CollectionMap WHERE collectionId="
	  + QString::number(m_srcCollectionId) );
  if ( q.next() ) {
    bool intOk = false;
    c_totalGestures = q.value(0).toInt(&intOk);
    Q_ASSERT(intOk);
  }

  q.exec( "SELECT A2.classId, A2.gestureId "
	  "FROM CollectionMap A1 LEFT JOIN ClassMap A2 "
	  "ON A1.gestureId = A2.gestureId "
	  "WHERE A1.collectionId=" + QString::number(m_srcCollectionId) + " "
	  "GROUP BY A2.gestureId" );
  while ( q.next() )
    c_classToGestures[q.value(0).toInt()] += q.value(1).toInt();


  /*
   * Update the UI.
   */

  Q_ASSERT( m_ui.topLabel );
  Q_ASSERT( m_ui.trainCountSpinBox );
  Q_ASSERT( m_ui.testCountSpinBox );
  Q_ASSERT( m_ui.useAllGestures );
  m_ui.topLabel->setText( tr("Split collection \"%1\" into two additional sets %2")
			  .arg(c_label)
			  .arg(QChar(0x2026)) ); // ellipsis - "..."
  m_ui.trainCountSpinBox->setMaximum( c_totalGestures );
  m_ui.testCountSpinBox->setMaximum( c_totalGestures );
  m_ui.useAllGestures->setText( tr("Use all %1 gestures").arg(c_totalGestures) );
}


void SplitCollectionDialog::collectionsEvent( VCollectionsEvent* event )
{
  // TODO: cleanup!

  // Assumption: We receive the event for trainCollectionId before testCollectionId

  Q_ASSERT( event );
  if ( event->originalSender() == this )
    {
      Q_ASSERT( event->idSet().size() == 1 ); // as we add 1 and only 1 at a time
      int id = * event->idSet().constBegin();

      if ( m_trainCollectionId < 0 ) {
	m_trainCollectionId = id;
      }
      else if ( m_testCollectionId < 0 )
	{
	  m_testCollectionId = id;
	  
	  // finished adding collections - now add their gestures

	  Q_ASSERT( m_ui.trainPercentSpinBox );
	  double trainingRatio = m_ui.trainPercentSpinBox->value() / 100.0;
	  
	  // TODO: impl other modes - the following assumes calcPerClass == true
	  // TODO: handle case where a single gesture may belong to multiple classes

	  QSet<int> trainSet;
	  QSet<int> testSet;

	  QHashIterator< int, QList<int> > it( c_classToGestures );
	  while ( it.hasNext() )
	    {
	      it.next();
	      QList<int> list = it.value();
	      int remainingForTraining = (int) rint( trainingRatio * list.size() );
	      while ( remainingForTraining > 0 ) { // don't compare to the multi-class "trainSet"
		trainSet += list.takeAt( random() % list.size() );
		--remainingForTraining;
	      }
	      testSet += list.toSet();
	    }

	  QSet<int> trainCollectionIdSet;
	  trainCollectionIdSet += m_trainCollectionId;
	  postControllerEvent( new CGesturesChangeCollectionsEvent
			       (trainSet, trainCollectionIdSet, IdSet(), this) );

	  QSet<int> testCollectionIdSet;
	  testCollectionIdSet += m_testCollectionId;
	  postControllerEvent( new CGesturesChangeCollectionsEvent
			       (testSet, testCollectionIdSet, IdSet(), this) );

	  QDialog::accept(); // hides dialog - as we're now finished
	}
    }
}


void SplitCollectionDialog::on_trainPercentSpinBox_valueChanged( double value ) {
  Q_ASSERT( m_ui.testPercentSpinBox );
  m_ui.testPercentSpinBox->blockSignals( true );
  m_ui.testPercentSpinBox->setValue( 100.0 - value );
  m_ui.testPercentSpinBox->blockSignals( false );
#if 0 // TODO: must also handle case when calcPerClass == false (use c_classToGestures)
  Q_ASSERT( m_ui.trainCountSpinBox );
  m_ui.trainCountSpinBox->blockSignals( true );
  m_ui.trainCountSpinBox->setValue( (int) rint(value/100.0 * (double)c_totalGestures) );
  m_ui.trainCountSpinBox->blockSignals( false );
#endif
}

void SplitCollectionDialog::on_trainCountSpinBox_valueChanged( int /*value*/ ) {
  // TODO: impl me
}

void SplitCollectionDialog::on_testPercentSpinBox_valueChanged( double value ) {
  Q_ASSERT( m_ui.trainPercentSpinBox );
  m_ui.trainPercentSpinBox->blockSignals( true );
  m_ui.trainPercentSpinBox->setValue( 100.0 - value );
  m_ui.trainPercentSpinBox->blockSignals( false );
#if 0 // TODO: must also handle case when calcPerClass == false (use c_classToGestures)
  Q_ASSERT( m_ui.testCountSpinBox );
  m_ui.testCountSpinBox->blockSignals( true );
  m_ui.testCountSpinBox->setValue( (int) rint(value/100.0 * (double)c_totalGestures) );
  m_ui.testCountSpinBox->blockSignals( false );
#endif
}

void SplitCollectionDialog::on_testCountSpinBox_valueChanged( int /*value*/ ) {
  // TODO: impl me
}
