/*  -*- c++ -*-  (for Emacs)
 *
 *  connectiondialog.h
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jan 05 2006.
 *  Copyright (c) 2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "connectiondialog.h"

#include <QApplication>
#include <QFileDialog>
#include <QSettings>
#include <QSqlDatabase>

#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/digestdbcontroller.h"
#include "MvcSettings/settingsmodel.h"
#include "MvcSettings/settingscontroller.h"

#include "digestsettings.h"


/*!
 * Constructs a database connection dialog with the given \em parent and
 * \em flags and attaches itself to the \em controller.
 */
ConnectionDialog::ConnectionDialog( AbstractController* settingsController,
				    AbstractController* digestDbController,
				    QWidget* parent, Qt::WindowFlags flags )
   : QDialog(parent, flags),
     AbstractSettingsView(this),
     AbstractDigestDbView(this)
{
  m_driverToLabel.insert( "QIBASE",  "Borland InterBase" );
  m_driverToLabel.insert( "QDB2",    "IBM DB2"           );
  m_driverToLabel.insert( "QMYSQL",  "MySQL"             );
  m_driverToLabel.insert( "QODBC",   "ODBC"              );
  m_driverToLabel.insert( "QOCI",    "Oracle (OCI)"      );
  m_driverToLabel.insert( "QPSQL",   "PostgreSQL"        );
  m_driverToLabel.insert( "QSQLITE", "SQLite"            );
  m_driverToLabel.insert( "QTDS",    "Sybase (TDS)"      );

  m_ui.setupUi( this );

  // Allow the port number to consist of zero to four digits (between 0 and 9)
  // Note: The "0" in the mask is an: "ASCII digit permitted but not required."
  Q_ASSERT( m_ui.portEdit );
  m_ui.portEdit->setInputMask( "0000" );

  if ( settingsController )
    settingsController->attachView( static_cast<AbstractSettingsView*>(this) );

  if ( digestDbController )
    digestDbController->attachView( static_cast<AbstractDigestDbView*>(this) );
}


QString ConnectionDialog::password() const {
  Q_ASSERT( m_ui.passwordEdit );
  return m_ui.passwordEdit->text();
}


/*!
 * Calls either AbstractSettingsView::dispatchEvent() or
 * AbstractDigestDbView::dispatchEvent(), depending on the event sender's
 * module ID.
 */
void ConnectionDialog::customEvent( QEvent* e )
{
  VEvent* ve = dynamic_cast<VEvent*>(e);
  if ( ve )
    {
      Q_ASSERT( ve->sender() );
      MvcModuleId_t m = ve->sender()->moduleId();
      if ( m == AbstractSettingsView::classModuleId() )
	AbstractSettingsView::dispatchEvent(ve);
      else if ( m == AbstractDigestDbView::classModuleId() )
	AbstractDigestDbView::dispatchEvent(ve);
    }
}


/*!
 * Resets the settings dialog and resynchronises it with the settings model.
 */
void ConnectionDialog::settingsResetEvent( VEvent* )
{
  Q_ASSERT( settingsController() );
  Q_ASSERT( settingsModel() );

  SettingsController* c = settingsController();

  Q_ASSERT( m_ui.dbTypeComboBox );
  m_ui.dbTypeComboBox->clear();
  QHashIterator<QString, QString> it( m_driverToLabel );
  while ( it.hasNext() ) {
    it.next();
    if ( QSqlDatabase::isDriverAvailable(it.key()) )
      m_ui.dbTypeComboBox->addItem( it.value(), it.key() );
  }

  // Note: As remoteRadioButton's state is the inverse of filenameRadioButton's,
  //       we only need to store one of them.
  c->bind( m_ui.dbTypeComboBox, "currentKey",
	   SIGNAL(currentIndexChanged(int)), DigestSettings::dbTypeKey );
  c->bind( m_ui.remoteRadioButton, "checked",
	   SIGNAL(toggled(bool)), DigestSettings::dbIsRemoteKey );
  c->bind( m_ui.filenameEdit, "text",
	   SIGNAL(editingFinished()), DigestSettings::dbFilenameKey );
  c->bind( m_ui.dbNameEdit, "text",
	   SIGNAL(editingFinished()), DigestSettings::dbNameKey );
  c->bind( m_ui.hostnameEdit, "text",
	   SIGNAL(editingFinished()), DigestSettings::dbHostnameKey );
  c->bind( m_ui.portEdit, "text",
	   SIGNAL(editingFinished()), DigestSettings::dbPortKey );
  c->bind( m_ui.usernameEdit, "text",
	   SIGNAL(editingFinished()), DigestSettings::dbUsernameKey );
  // Warning - don't store the password
}


/*!
 * Resets widgets that use data from the Digest Database.
 */
void ConnectionDialog::digestDbResetEvent( VEvent* )
{}


void ConnectionDialog::on_chooseFileButton_clicked()
{
  QString filename = QFileDialog::getOpenFileName( this, tr("Open") );

  if ( filename.isEmpty() )
    return; // this also covers the case of when the user clicks cancel

  Q_ASSERT( m_ui.filenameEdit );
  m_ui.filenameEdit->setText( filename );
}
