/*  -*- c++ -*-  (for Emacs)
 *
 *  abstractrecogniser.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Mon Jul 14 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "abstractrecogniser.h"

#include <QDir> // TODO: remove the need for me!
#include <QVariant>

#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/dgesturerecord.h"
#include "MvcDigestDb/dtrainedrecogrecord.h"

#include "abstractfeature.h"
#include "featurefactory.h"


/*!
  \class AbstractRecogniser
  
  \brief The AbstractRecogniser class is an abstract base class that custom
  recognisers must extend and implement.
  
  Once an instance of this class has been created, the readModelFile() method
  will be called, followed by classifyGestureImp() a multitude of times.
  
  In order to make a recogniser implementation visible to the user in the
  application, its class must be added to the RecogniserFactory.
  
  See also: AbstractRecogniserTrainer and RecogniserFactory.

TODO: FIX DOC!

 * \brief The AbstractRecogniserTrainer class is an abstract base class that
 *        custom recogniser trainers must extend and implement.
 *
 * Once an instance of this class has been created, the train() method will be
 * called. The training is implemented in the form of a "template method".
 * (By the way, a Template Method is where we "Define the skeleton of an
 * algorithm in an operation, deferring some steps to subclasses.
 * Template Method lets subclasses redefine certain steps of an algorithm
 * without changing the algorithm's structure." - Design Patterns, Gamma et al.)
 * When train() is called, the following pattern of calls will be made:
 * \code
 * initTraining()
 * examineSample()
 * examineSample()
 * examineSample()
 * ...
 * examineSample()
 * finalizeTraining()
 * writeModelFile()
 * \endcode
 *
 * In order to make a recogniser trainer implementation visible to the user in
 * the application, its class must be added to the RecogniserTrainerFactory.
 *
 * See also: AbstractRecogniser and RecogniserTrainerFactory.
 */


/*!
  The \em jvm is required, as it need to instantiate feature extractors, which
  may be written in Java - TODO: remove this - make the factory map the JVM with
  Java features automatically!
 */
AbstractRecogniser::AbstractRecogniser( JavaVM* jvm,
					DigestDbModel* digestDbModel,
					QObject* parent )
  : QObject(parent),
    m_jvm(jvm),
    m_digestDbModel(digestDbModel),
    m_trainingProgress(0)
{}

AbstractRecogniser::~AbstractRecogniser()
{} // Note: the m_featureInstances are children of this object -> auto deleted


/*!
  \fn QByteArray AbstractRecogniser::key() const

  Returns the recogniser's "key". This is used for identifying recognisers for
  purposes such as recording the type of a given trained recogniser and for use
  in "factories".

  This string should only contain lower-case characters. All string matching
  performed with it will be case-insensitive.

  \b Note: This does not return a (const) reference for the sake of "wrappers"
           (e.g. JavaRecogniserWrapper and StlRecogniserWrapper), which may have
	   to convert the string from a different structure, usually with the use
	   of a temporary variable. Qt's implicit sharing will help performance.
*/


/*!
  \fn QString AbstractRecogniser::title() const

  Returns the recogniser's "title". This is used in the application's user
  interface to label the recogniser in a human friendly / readable manner.

  \b Note: This does not return a (const) reference for the sake of "wrappers"
           (e.g. JavaRecogniserWrapper and StlRecogniserWrapper), which may have
	   to convert the string from a different structure, usually with the use
	   of a temporary variable. Qt's implicit sharing will help performance.
*/


/*!
  \fn QString AbstractRecogniser::description() const

  Returns the recogniser's "description". This is used in the application's user
  interface to provide more information to the user about the recogniser.

  A subset of HTML may be used, such as to make the text bold and change its
  colour, size and alignment. Please see the Qt documentation for more details on
  its HTML support.

  \b Note: This does not return a (const) reference for the sake of "wrappers"
           (e.g. JavaRecogniserWrapper and StlRecogniserWrapper), which may have
	   to convert the string from a different structure, usually with the use
	   of a temporary variable. Qt's implicit sharing will help performance.
*/


JavaVM* AbstractRecogniser::jvm() const
{ return m_jvm; }

DigestDbModel* AbstractRecogniser::digestDbModel() const
{ return m_digestDbModel; }

const QSet<int>& AbstractRecogniser::trainingSet() const
{ return m_trainingSet; }

const QList<QByteArray>& AbstractRecogniser::orderedFeatures() const
{ return m_orderedFeatures; }


/*!
  TODO: re-write this doc

  Returns the file name for the output model file, including the path
  (e.g. "/Users/bob/Library/Digest/linear_recog52.model").

  This should be used by the implementation of writeModelFile().

  \b Note: The reason that "Path" is part of the method name is because an
            outputPath() method exists. Thus, if this method were called
            outputFilename(), then it might appear that it did not include the
            path and would need to be concatenated with outputPath().
*/
const QString& AbstractRecogniser::modelFilePath() const
{ return m_modelFilePath; }


const QHash<QString, QVariant>& AbstractRecogniser::params() const
{ return m_params; }


/*!
  Returnns the current training progress - that is, a value between 0 and
  trainingSet().count().
*/
int AbstractRecogniser::currentTrainingProgress() const 
{ return m_trainingProgress; }


bool AbstractRecogniser::loadRecord( const DTrainedRecogRecord& record )
{
  m_trainingSet      = record.trainingSet;
  m_orderedFeatures  = record.orderedFeatures;
  m_modelFilePath    = ( QDir::homePath()
			 + "/Library/Digest/" // TODO: remove static path!
			 + record.modelFile );

  rebuildFeatureInstances(); // as the features changed

  return readModelFile( m_modelFilePath );
}


/*!
  TODO: re-write me

  Trains a new recogniser, using the provided \em gestureIds,
  \em featureKeys and writes a model file out to \em outputFilePath.

  The \em outputFilePath is the file name for the output model file,
  including the path (e.g. "/Users/bob/Library/Digest/linear_recog52.model").
  See also outputFilePath() and outputPath().
 */
bool AbstractRecogniser::train( const QSet<int>& trainingSet,
				const QList<QByteArray>& orderedFeatures,
				const QString& modelFilePath,
				const QHash<QString, QVariant>& params )
{
  m_trainingProgress  = 0;
  m_trainingSet       = trainingSet;
  m_orderedFeatures   = orderedFeatures;
  m_modelFilePath     = modelFilePath;
  m_params            = params;

  rebuildFeatureInstances(); // as the features changed

  if ( ! initTraining(m_orderedFeatures, m_params) ) return false;

  foreach ( int id, m_trainingSet )
    {
      Q_ASSERT( m_digestDbModel != 0 );
      DGestureRecord g = m_digestDbModel->fetchGesture( id );

      if ( ! examineSample(extractFeatures(flatten(g.strokes)), g.classes) )
	return false;
      
      ++m_trainingProgress;
      emit trainingProgressed( m_trainingProgress );
    }

  if ( ! finaliseTraining() ) return false;
  if ( ! writeModelFile(m_modelFilePath) ) return false;

  return true; // all looks well...
}


/*!
  \fn ClassProbabilities AbstractRecogniser::classify( const FeatureVec& featureVec )

  This is called every time a given gesture is to be classified / recognised.

  The \em featureVec is the feature vector that has been extracted from the
  \em gesture. Hence, the \em gesture parameter may not need to be used at
  all.
   
  \b Note: The FeatureResultT typedef is currently a \b double.
   
  The featureVec can be converted to an STL vector:
  \code
  vector<FeatureResultT> v = featureVec.toStdVector()
  \endcode

  \b Note: If you wan't the stoke data for some reason, then store a copy of
           what you return from flatten().
   
  The ClassProbabilities results to be returned is simply a hash-table that maps
  gesture classes to the recogniser's "confidence" that the given \em gesture
  belongs to the given class. The sum of the confidence values across all the
  classes (those known and trained-on) should be equal to 1.0. Thus, scaling
  may need to be performed before returning the results.
   
  \b Note: The ClassProbabilities typedef is a QHash (hash-table - see the Qt doc),
  mapping the class ID ClassIndexT (currently an \b int) to
  the confidence ClassProbabilityT (currently a \b double).
   
  It does not matter if there are fewer classes in the returned ClassProbabilities
  than there exists in the current database (new classes may be added to or
  removed from the database at any given time). Thus, a recogniser could
  simply return a hash that only has one item in it, which would have to have
  a confidence of 1.0.
   
  Example - A recogniser has trained on three classes: X, Y and Z.
  If the recogniser has a confidence of 70% that the given gesture was an X,
  20% that it was a Y and 10% that it was a Z, then the following would return
  the corresponding ClassProbabilities:
  \code
  ClassProbabilities results;
  results[X] = 0.7;
  results[Y] = 0.2;
  results[Z] = 0.1;
  return results;
  \endcode
  \b Warning: The above is only an example of how the results are represented
  by ClassProbabilities. A real recogniser would obviously not be hard
  coded and would get the class IDs from its own model.
*/



/*!
  \fn bool AbstractRecogniser::initTraining( const QList<QByteArray>& featureKeys,
                                             const QHash<QString, QVariant>& params )

  This is used primarily for ensuring that the trainer's state is ready for
  training, as train() can be called multiple times.

  The \em featureKeys are provided here, as they may be required to initialise
  one or more data structures.
*/


/*!
  \fn AbstractRecogniser::examineSample( const FeatureVec& featureVec,
                                         const QSet<int>& classes ) 

  This is called every time a sample/example is to be processed (or logged,
  where processing could be delayed until finaliseTraining() is called).
  
  The \em featureVec is the feature vector that has been extracted from the
  \em sample.
  
  \b Note: The FeatureResultT typedef is currently a \b double.
  
  The featureVec can be converted to an STL vector:
  \code
  vector<FeatureResultT> v = featureVec.toStdVector()
  \endcode

  \b Note: If you wan't the stoke data for some reason, then store a copy of
           what you return from flatten().
*/


/*!
 \fn bool AbstractRecogniser::finaliseTraining()

  This is used primarily for building the model if that wasn't (completely)
  done by examineSample().
 
  This should also be used to free any allocated memory and to clear the
  state, remembering that train() may be called multiple times.
*/


/*!
 \fn bool AbstractRecogniser::writeModelFile( const QString& fileName )

 This must be implemented and must write a file to \em fileName.
 The \em fileName is the file name for the output model file, including the
 path (e.g. "/Users/bob/Library/Digest/linear_recog52.model").
 If the model must be written as multiple files, then \em fileName
 should be the master/index/reference file (which will be passed to
 the corresponding AbstractRecogniser::readModelFile() method) and any
 support files should be placed along side it in the same path.

 There are no restrictions on the file being binary or ascii, nor on its
 contents.

 See also fileName()
*/


/*!
  \fn bool AbstractRecogniser::readModelFile( const QString& fileName )

  This method is called during initialisation, as to load in the recogniser
  specific model file that was created by the corresponding
  AbstractRecogniserTrainer::writeModelFile() method. Thus, this method will
  be called only once.
 
  The \em fileName is the file name of the model file, including the path
  (e.g. "/Users/bob/Library/Digest/linear_recog52.model").
 
  Because some recognisers may not require a training phase, or may not use a
  GestureLab managed model file for some other reason, this method could be
  implemented to just return \em true.
 
 
  There are no restrictions on the file being binary or ascii, nor on its
  contents.
*/


/*!
  This is used to merge/join/flattern a (potentially) multi-stroke gesture before
  feature extraction is performed (and thus examination or classification).

  If any filtering needs to be performed on the stroke(s), then it should be done
  by this method. This includes smoothing the stroke(s) or even reducing the
  number of points in them.

  This default implementation simply joins all of the strokes together.
  It does not join strokes together based on the location of their start and end
  points.

  \b Note: This is non-const, as the recogniser may need to examine the
           \em stroke's structure and record information about it, ready
           for use by either classify() or examineSample().
*/
Stroke AbstractRecogniser::flatten( const StrokeList& strokes ) {
  Stroke stroke;
  foreach ( const Stroke& s, strokes ) stroke += s;
  return stroke;
}


// Note: Using hash allows for differences in known shape classes
ClassProbabilities AbstractRecogniser::classify( const StrokeList& strokes )
{ return classify( extractFeatures(flatten(strokes)) ); }


void AbstractRecogniser::rebuildFeatureInstances()
{
  // Note: Use QVector (vs. QList), as we know exactly how much to allocate.
  qDeleteAll( m_featureInstances );
  m_featureInstances.resize( m_orderedFeatures.size() ); // no need to clear()
  Q_ASSERT( m_featureInstances.size() == m_orderedFeatures.size() );
  for ( int i=0; i < m_orderedFeatures.size(); ++i ) {
    m_featureInstances[i]
      = FeatureFactory::create( m_orderedFeatures.at(i), m_jvm, this );
    Q_ASSERT( m_featureInstances.at(i) != 0 );
  }
}


/*!
 * \b Warning: You shouldn't need to call this directly, nor re-implement it.
 *             However, for some reason you want to - hence, it's allowed.
 */
FeatureVec AbstractRecogniser::extractFeatures( const Stroke& stroke )
{
  FeatureVec featureVec( m_featureInstances.size() );
  for ( int i=0; i < m_featureInstances.size(); ++i ) {
    Q_ASSERT( m_featureInstances.at(i) != 0 );
    featureVec[i]
      = m_featureInstances.at(i)->calcValue( stroke );
  }
  return featureVec;
}
