/*  -*- c++ -*-  (for Emacs)
 *
 *  abstractrecogniser.h
 *  Digest
 * 
 *  Created by Aidan Lane on Mon Jul 11 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef ABSTRACTRECOGNISER_H
#define ABSTRACTRECOGNISER_H


#include <QObject>

#include <QHash>
#include <QList>
#include <QPointer>
#include <QSet>
#include <QString>
#include <QVector>

#include <jni.h>

#include "Gesture/strokelist.h"
#include "MvcDigestDb/dexperimentresultrecord.h" // for ClassProbabilityT

#include "abstractfeature.h"

class QByteArray;
class QVariant;

class DigestDbModel;
class DTrainedRecogRecord;

// Note: FeatureResultT is defined in abstractfeature.h
typedef QVector<FeatureResultT>                       FeatureVec;
typedef QVectorIterator<FeatureResultT>               FeatureVecIterator;
typedef QMutableVectorIterator<FeatureResultT>        MutableFeatureVecIterator;


class AbstractRecogniser : public QObject {

  Q_OBJECT

public:
  AbstractRecogniser( JavaVM* jvm,
		      DigestDbModel* digestDbModel,
		      QObject* parent = 0 );

  virtual ~AbstractRecogniser();

  virtual QByteArray  key()         const = 0;
  virtual QString     title()       const = 0;
  virtual QString     description() const = 0;

  JavaVM*             jvm()             const;
  DigestDbModel*      digestDbModel()   const;

  const QSet<int>&                 trainingSet()     const;
  const QList<QByteArray>&         orderedFeatures() const;
  const QString&                   modelFilePath()   const;
  const QHash<QString, QVariant>&  params()          const;

  virtual QHash<QString, QVariant> defaultParams() const = 0;

  int currentTrainingProgress() const;

  virtual bool loadRecord( const DTrainedRecogRecord& record );

  virtual bool train( const QSet<int>& trainingSet,
		      const QList<QByteArray>& orderedFeatures,
		      const QString& modelFilePath,
		      const QHash<QString, QVariant>& params );

  virtual ClassProbabilities classify( const StrokeList& strokes );


signals:
  void trainingProgressed( int progress );


protected:
  virtual bool initTraining( const QList<QByteArray>& featureKeys,
			     const QHash<QString, QVariant>& params ) = 0;
  virtual bool examineSample( const FeatureVec& featureVec,
			      const QSet<int>& classes ) = 0;
  virtual bool finaliseTraining() = 0;

  virtual bool writeModelFile( const QString& fileName ) = 0;
  virtual bool readModelFile( const QString& fileName ) = 0;

  virtual Stroke flatten( const StrokeList& strokes );
  virtual ClassProbabilities classify( const FeatureVec& featureVec ) = 0;

  virtual FeatureVec extractFeatures( const Stroke& stroke );


private:
  void rebuildFeatureInstances();

  JavaVM*                    m_jvm;
  QPointer<DigestDbModel>    m_digestDbModel;

  QSet<int>                  m_trainingSet;
  QList<QByteArray>          m_orderedFeatures;
  QString                    m_modelFilePath;
  QHash<QString, QVariant>   m_params;

  int                        m_trainingProgress;

  QVector<AbstractFeature*>  m_featureInstances;
};


#endif  // ! ABSTRACTRECOGNISER_H
