/*  -*- c++ -*-  (for Emacs)
 *
 *  javarecogniserwrapper.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Fri Oct 28 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "javarecogniserwrapper.h"

#include <QDebug> // TODO: remove me!
#include <QVariant>


/*!
 * The \em javaClassName is the  fully-qualified class name
 * (that is, a package name, delimited by "/", followed by the class name).
 * If the name begins with "[" (the array signature character),
 * it returns an array class. The string is encoded in modified UTF-8.
 * Example: "GestureRecognition/JLinearRecogniserTrainer".
 */
JavaRecogniserWrapper::JavaRecogniserWrapper( const QString& javaClassName,
					      JavaVM* jvm,
					      DigestDbModel* digestDbModel,
					      QObject* parent )
  : AbstractRecogniser(jvm, digestDbModel, parent),
    m_javaClassName(javaClassName),
    m_env(0),
    m_recog_cls(0),
    m_recog_gref(0),
    m_recog_constructor_mid(0),
    m_recog_key_mid(0),
    m_recog_title_mid(0),
    m_recog_description_mid(0),
    m_recog_initTraining_mid(0),
    m_recog_examineSample_mid(0),
    m_recog_finaliseTraining_mid(0),
    m_recog_writeModelFile_mid(0),
    m_recog_readModelFile_mid(0),
    m_recog_flatten_mid(0),
    m_recog_classify_mid(0),
    m_vector_cls(0),
    m_vector_constructor_mid(0),
    m_vector_add_mid(0),
    m_double_cls(0),
    m_double_constructor_mid(0)
{
  // TODO: cleanup code!

  jint res = 0;

  // Must pass NULL as the third argument
  Q_ASSERT( jvm );
  res = jvm->AttachCurrentThread( (void**)&m_env, 0 );
  if ( res < 0 ) {
    qDebug( "Thread: attach failed" ); // TODO: handle me properly!
    return;
  }

  /*
   * Recogniser class
   */

  // WARNING!!! Yes, the following is meant to use "/" to separate the package
  //            name from the class name - this took a while to figure out :-/
  m_recog_cls = m_env->FindClass( m_javaClassName.toUtf8().constData() );
  if ( m_recog_cls == 0 ) {
    qDebug( "Thread: Can't find the %s class", m_javaClassName.toUtf8().constData() ); // TODO: handle me properly!
    cleanup();
    return;
  }

  m_recog_constructor_mid       = tryGetMethodID( m_recog_cls,
						    "<init>", "()V" );
  m_recog_key_mid               = tryGetMethodID( m_recog_cls,
						    "key",
						    "()Ljava/lang/String;" );
  m_recog_title_mid             = tryGetMethodID( m_recog_cls,
						    "title",
						    "()Ljava/lang/String;" );
  m_recog_description_mid       = tryGetMethodID( m_recog_cls,
						    "description",
						    "()Ljava/lang/String;" );
  m_recog_initTraining_mid      = tryGetMethodID( m_recog_cls,
						    "initTraining", "()Z" );
  m_recog_examineSample_mid     = tryGetMethodID( m_recog_cls,
						    "examineSample",
						    "(Ljava/util/Vector;)Z" );
  m_recog_finaliseTraining_mid  = tryGetMethodID( m_recog_cls,
						    "finaliseTraining", "()Z" );
  m_recog_writeModelFile_mid    = tryGetMethodID( m_recog_cls,
						    "writeModelFile",
						    "(Ljava/lang/String;)Z" );
  m_recog_readModelFile_mid     = tryGetMethodID( m_recog_cls,
						    "readModelFile",
						    "(Ljava/lang/String;)Z" );
  m_recog_flatten_mid       = tryGetMethodID( m_recog_cls,
						    "flatten", "()V" );
  m_recog_classify_mid          = tryGetMethodID( m_recog_cls,
						    "classify", "()V" );


  /*
   * Vector class
   */

  m_vector_cls = m_env->FindClass( "Ljava/util/Vector;" );
  if ( m_vector_cls == 0 ) {
    qDebug( "Thread: Can't find the Vector class" ); // TODO: handle me properly!
    cleanup();
    return;
  }

  m_vector_constructor_mid       = tryGetMethodID( m_vector_cls,
						   "<init>", "()V" );
  m_vector_add_mid               = tryGetMethodID( m_vector_cls,
						   "add",
						   "(Ljava/lang/Object;)Z" );


  /*
   * Double class
   */

  m_double_cls = m_env->FindClass( "Ljava/lang/Double;" );
  if ( m_double_cls == 0 ) {
    qDebug( "Thread: Can't find the Double class" ); // TODO: handle me properly!
    cleanup();
    return;
  }

  m_double_constructor_mid       = tryGetMethodID( m_double_cls, "<init>", "(D)V" );


  /*
   * Finial initialisation
   */

  jobject trainer_lref
    = m_env->NewObject( m_recog_cls, m_recog_constructor_mid );
  if ( trainer_lref == 0 ) {
    qDebug( "Thread: Can't create instance of %s", m_javaClassName.toUtf8().constData() ); // TODO: handle me properly!
    cleanup();
    return;
  }
  m_recog_gref = m_env->NewGlobalRef( trainer_lref );
  if ( m_recog_gref == 0 ) {
    qDebug( "Thread: Can't create global ref of %s", m_javaClassName.toUtf8().constData() ); // TODO: handle me properly!
    cleanup();
    return;
  }
}


JavaRecogniserWrapper::~JavaRecogniserWrapper()
{
  cleanup();
}


const QString& JavaRecogniserWrapper::javaClassName() const {
  return m_javaClassName;
}


void JavaRecogniserWrapper::cleanup()
{
  m_env->DeleteGlobalRef( m_recog_gref );
  m_env->DeleteGlobalRef( m_recog_cls );
  m_recog_gref = 0;
  m_recog_cls = 0;

  if ( m_env->ExceptionOccurred() )
    m_env->ExceptionDescribe(); // TODO: handle me properly!
  jvm()->DetachCurrentThread();
  m_env = 0;
}


QByteArray JavaRecogniserWrapper::key() const {
  Q_ASSERT( m_env && m_recog_gref && m_recog_key_mid );
  return jstringToQString( (jstring) m_env->CallObjectMethod
			   (m_recog_gref, m_recog_key_mid) ).toLatin1();
}

QString JavaRecogniserWrapper::title() const {
  Q_ASSERT( m_env && m_recog_gref && m_recog_title_mid );
  return jstringToQString( (jstring) m_env->CallObjectMethod
			   (m_recog_gref, m_recog_title_mid) );
}

QString JavaRecogniserWrapper::description() const {
  Q_ASSERT( m_env && m_recog_gref && m_recog_description_mid );
  return jstringToQString( (jstring) m_env->CallObjectMethod
			   (m_recog_gref, m_recog_description_mid) );
}


QHash<QString, QVariant> JavaRecogniserWrapper::defaultParams() const {
  QHash<QString, QVariant> p;
  // TODO: impl me!
  return p;
}


bool JavaRecogniserWrapper::initTraining( const QList<QByteArray>& /*featureKeys*/,
					  const QHash<QString, QVariant>& /*params*/ ) {
  Q_ASSERT( m_env && m_recog_gref && m_recog_initTraining_mid );
  return m_env->CallBooleanMethod( m_recog_gref, m_recog_initTraining_mid );
}


bool JavaRecogniserWrapper::examineSample( const FeatureVec& /*featureVec*/,
					   const QSet<int>& /*classes*/ )
{
  Q_ASSERT( m_env != 0 );
  Q_ASSERT( m_recog_gref != 0 );
  Q_ASSERT( m_recog_examineSample_mid != 0 );

  jobject featureVec_lref
    = m_env->NewObject( m_vector_cls, m_vector_constructor_mid );
  if ( featureVec_lref == 0 ) {
    qDebug( "Thread: Can't create instance of Vector" ); // TODO: handle me properly!
    cleanup();
    return false;
  }

  // TODO: finish me!

  return m_env->CallBooleanMethod( m_recog_gref, m_recog_examineSample_mid );
}


bool JavaRecogniserWrapper::finaliseTraining() {
  Q_ASSERT( m_env && m_recog_gref && m_recog_finaliseTraining_mid );
  return m_env->CallBooleanMethod( m_recog_gref, m_recog_finaliseTraining_mid );
}


bool JavaRecogniserWrapper::writeModelFile( const QString& fileName ) {
  Q_ASSERT( m_env && m_recog_gref && m_recog_writeModelFile_mid );
  return m_env->CallBooleanMethod( m_recog_gref,
				   m_recog_writeModelFile_mid,
				   m_env->NewStringUTF(fileName.toUtf8().constData()) );
}


bool JavaRecogniserWrapper::readModelFile( const QString& fileName ) {
  Q_ASSERT( m_env && m_recog_gref && m_recog_readModelFile_mid );
  return m_env->CallBooleanMethod( m_recog_gref,
				   m_recog_readModelFile_mid,
				   m_env->NewStringUTF(fileName.toUtf8().constData()) );
}


Stroke JavaRecogniserWrapper::flatten( const StrokeList& /*strokes*/ )
{
  Q_ASSERT( m_env && m_recog_gref && m_recog_flatten_mid );
  m_env->CallVoidMethod( m_recog_gref, m_recog_flatten_mid );
  return Stroke();
}


ClassProbabilities JavaRecogniserWrapper::classify( const FeatureVec& /*featureVec*/ )
{
  Q_ASSERT( m_env && m_recog_gref && m_recog_classify_mid );
  m_env->CallVoidMethod( m_recog_gref, m_recog_classify_mid );
  return ClassProbabilities();
}


jmethodID JavaRecogniserWrapper::tryGetMethodID( jclass cls, const char* name, const char* sig )
{
  jmethodID mid = m_env->GetMethodID( cls, name, sig );
  if ( mid == 0 ) {
    qDebug( "Thread: Can't find method: %s", name );
    exit( 1 ); // TODO: handle me properly!
  }
  return mid;
}


QString JavaRecogniserWrapper::jstringToQString( jstring jstr ) const
{
  Q_ASSERT( m_env );
  QString qstr;
  const char* utf8_str = m_env->GetStringUTFChars( jstr, 0 );
  qstr = QString::fromUtf8( utf8_str );
  m_env->ReleaseStringUTFChars( jstr, utf8_str );
  return qstr;
}
