/*  -*- c++ -*-  (for Emacs)
 *
 *  recogniserfactory.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Mon Jul 11 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "recogniserfactory.h"

#include <QByteArray>
#include <QCoreApplication>
#include <QDebug> // TODO: remove me
#include <QDir>
#include <QHash>
#include <QLibrary>
#include <QString>

#include <cstring>

#include "stlrecogniserfactoryplugin.h"
#include "stlrecogniserinterface.h"
#include "stlrecogniserwrapper.h"

using namespace std;


static QHash<QLibrary*, StlRecogniserFactoryPlugin*>   s_stlPlugins;
static QHash<QByteArray, StlRecogniserFactoryPlugin*>  s_lowerKeyToStlPlugin;

static bool s_needsKeysRefresh = true;

bool tryLoadStlLibrary( const QString& fileName );
bool tryLoadJavaLibrary( const QString& fileName, JavaVM* jvm );



/*!
 * \b Warning:
 *    This method is not thread-safe until it returns from the first call.
 */
const QList<QByteArray>& RecogniserFactory::keys()
{
  static QList<QByteArray> list;

  if ( s_needsKeysRefresh ) {
    s_needsKeysRefresh = false;
    list << s_lowerKeyToStlPlugin.keys();
  }

  return list;
}


/*!
 * Creates an instance of a recogniser, specified using the given \em key.
 * The \em jvm, \em digestDbModel and \em parent parameters are all passed
 * directly to the given recogniser 's constructor as needed.
 *
 * \b Note:
 * Not all recogniser s make use of the \em jvm, but some do, such as
 * JLinearRecogniser. Hence, to handle this generically, a valid
 * Java Virtual Machine must always be passed.
 *
 *
 * \b Note: This method uses case in-sensitive key matching.
 */
AbstractRecogniser*
RecogniserFactory::create( const QByteArray& key,
			   JavaVM* jvm,
			   DigestDbModel* digestDbModel,
			   QObject* parent )
{
  const QByteArray lowerKey = key.toLower();

  if ( s_lowerKeyToStlPlugin.contains(lowerKey) ) {
    StlRecogniserFactoryPlugin* p = s_lowerKeyToStlPlugin.value( lowerKey );
    Q_ASSERT( p ); // sanity check for what this class has done (can assert it)
    StlRecogniserInterface* stl_r = p->create( lowerKey.constData() );
    if ( stl_r == 0 ) return 0;  // out of our hands (can't assert it)
    return new StlRecogniserWrapper( stl_r, jvm, digestDbModel, parent );
  }
  
  return 0;
}


QString RecogniserFactory::title( const QByteArray& key )
{
  const QByteArray lowerKey = key.toLower(); // cache it
  
  if ( s_lowerKeyToStlPlugin.contains(lowerKey) ) {
    StlRecogniserFactoryPlugin* p = s_lowerKeyToStlPlugin.value( lowerKey );
    Q_ASSERT( p );
    return QString::fromStdString( p->title(lowerKey.constData()) );
  }

  return QString();
}


QString RecogniserFactory::description( const QByteArray& key )
{
  // Warning: The descriptions may be long and there may be many of them,
  //          hence, its probably best that we don't cache them.
  const QByteArray lowerKey = key.toLower(); // cache it

  if ( s_lowerKeyToStlPlugin.contains(lowerKey) ) {
    StlRecogniserFactoryPlugin* p = s_lowerKeyToStlPlugin.value( lowerKey );
    Q_ASSERT( p );
    return QString::fromStdString( p->description(lowerKey.constData()) );
  }

  return QString();
}


/*!
 * \b Warning: This \b must not be called if any plugin created recognisers are
 *             still in use.
 */
void RecogniserFactory::loadPlugins( JavaVM* jvm )
{
  qDebug("RecogniserFactory::loadPlugins"); // TODO: remove me

  unloadPlugins();
  Q_ASSERT( s_stlPlugins.isEmpty() );      // sanity check
  Q_ASSERT( s_lowerKeyToStlPlugin.isEmpty() );  // sanity check

  QDir pluginsDir = QDir( QCoreApplication::applicationDirPath() );

#if defined(Q_OS_WIN)
  if ( pluginsDir.dirName().toLower() == "debug"
       || pluginsDir.dirName().toLower() == "release" )
    pluginsDir.cdUp();
#elif defined(Q_OS_MAC)
  if (pluginsDir.dirName() == "MacOS")
    pluginsDir.cdUp();
#endif
  // TODO: check if the following is case-insenstive on a case sensitive FS
  pluginsDir.cd( "plugins" );

  // TODO: write code to cleanup plugin objects on app destruction!
  foreach ( const QString& fileName, pluginsDir.entryList(QDir::Files) ) {
    QString absFileName = pluginsDir.absoluteFilePath( fileName );
    if ( ! tryLoadStlLibrary(absFileName) )
      tryLoadJavaLibrary(absFileName, jvm);
  }

  s_needsKeysRefresh = true;
}



/*!
 * \b Warning: This \b must not be called if any plugin created recognisers are
 *             still in use.
 */
void RecogniserFactory::unloadPlugins()
{
  QHashIterator<QLibrary*, StlRecogniserFactoryPlugin*> it( s_stlPlugins );
  while ( it.hasNext() ) {
    it.next();
    delete it.value(); // delete the plugin's instance object first
    Q_ASSERT( it.key() );
    it.key()->unload();
    delete it.key();    // delete the plugin's loader last
  }
  s_stlPlugins.clear();
  s_lowerKeyToStlPlugin.clear();
  s_needsKeysRefresh = true;
}


bool tryLoadStlLibrary( const QString& fileName )
{
  qDebug() << "tryLoadStlLibrary: " << fileName; // TODO: remove me

  StlRecogniserFactoryPluginIdFunction        plugin_id_func        = 0;
  StlRecogniserFactoryPluginInstanceFunction  plugin_instance_func  = 0;
  StlRecogniserFactoryPlugin*                 plugin                = 0;

  QLibrary* loader = new QLibrary( fileName );

  plugin_id_func = ( (StlRecogniserFactoryPluginIdFunction)
		     loader->resolve("plugin_id") );

  if ( !plugin_id_func
       || strcmp(plugin_id_func(), StlRecogniserFactoryPlugin_id) != 0 )
    return false;

  plugin_instance_func = ( (StlRecogniserFactoryPluginInstanceFunction)
			   loader->resolve("plugin_instance") );

  if ( !plugin_instance_func
       || !plugin_instance_func() )
    return false;

  plugin = plugin_instance_func();

  if ( plugin )
    {
      s_stlPlugins.insert( loader, plugin );

      // TODO: warn user of key conflicts!?
      set<string> keys = plugin->keys();
      set<string>::const_iterator it;
      for ( it = keys.begin(); it != keys.end(); ++it )
	s_lowerKeyToStlPlugin.insert( QByteArray((*it).c_str()).toLower(), plugin );
    }

  return true;
}


/*!
 * Alg:
 *
 * 1. Searches inside the 
 *
 */
bool tryLoadJavaLibrary( const QString& fileName, JavaVM* jvm )
{
  qDebug() << "tryLoadJavaLibrary: " << fileName;
  qDebug() << "is java? " << fileName;

  JNIEnv*  m_env;
  jclass   m_recog_cls;
  jobject  m_recog_gref;
  jint res = 0;
  bool ret = false;
#if 0
  // Must pass NULL as the third argument
  Q_ASSERT( jvm );
  res = jvm->AttachCurrentThread( (void**)&m_env, 0 );
  if ( res < 0 ) {
    qDebug( "Thread: attach failed" ); // TODO: handle me properly!
    ret = false;
    goto cleanup; // ;-)
  }

  /*
   * Recogniser class
   */

  // WARNING!!! Yes, the following is meant to use "/" to separate the package
  //            name from the class name - this took a while to figure out :-/
  m_recog_cls = m_env->FindClass( m_javaClassName.toUtf8().constData() );
  if ( m_recog_cls == 0 ) {
    qDebug( "Thread: Can't find the %s class", m_javaClassName.toUtf8().constData() ); // TODO: handle me properly!
    ret = false;
    goto cleanup; // ;-)
  }


  ret = true; // all appears to be ok


 cleanup:
  if ( m_env != 0 )
    {
      m_env->DeleteGlobalRef( m_recog_gref );
      m_env->DeleteGlobalRef( m_recog_cls );
      m_recog_gref = 0;
      m_recog_cls = 0;

      if ( m_env->ExceptionOccurred() )
	m_env->ExceptionDescribe(); // TODO: handle me properly!
      jvm()->DetachCurrentThread();
      m_env = 0;
    }
#endif
  return ret;
}
