/*  -*- c++ -*-  (for Emacs)
 *
 *  stlrecogniserwrapper.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jan 19 2006.
 *  Copyright (c) 2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "stlrecogniserwrapper.h"

#include <QVariant>

#include <set>

#include "stlrecogniserinterface.h"

using namespace std;


/*!
  Constructs a new C++ STL recogniser wrapper.

  \b Note: The new object takes ownership of the \em recogniser and hence becomes
           responsible for the deletion of it.

  \b Note: This asserts that the \em recogniser is non-null. This is also
           asserted by almost all of this class's methods.
*/
StlRecogniserWrapper::StlRecogniserWrapper( StlRecogniserInterface* recogniser,
					    JavaVM* jvm,
					    DigestDbModel* digestDbModel,
					    QObject* parent )
  : AbstractRecogniser(jvm, digestDbModel, parent),
    m_recogniser(recogniser)
{
  Q_ASSERT( m_recogniser );
}

StlRecogniserWrapper::~StlRecogniserWrapper()
{ delete m_recogniser; }


QByteArray StlRecogniserWrapper::key() const {
  Q_ASSERT( m_recogniser );
  return QByteArray( m_recogniser->key().c_str() );
}

QString StlRecogniserWrapper::title() const {
  Q_ASSERT( m_recogniser );
  return QString::fromStdString( m_recogniser->title() );
}

QString StlRecogniserWrapper::description() const {
  Q_ASSERT( m_recogniser );
  return QString::fromStdString( m_recogniser->description() );
}


QHash<QString, QVariant> StlRecogniserWrapper::defaultParams() const {
  QHash<QString, QVariant> p;
  // TODO: impl me!
  return p;
}


StlRecogniserInterface* StlRecogniserWrapper::recogniser() const
{ return m_recogniser; }


bool StlRecogniserWrapper::initTraining( const QList<QByteArray>& featureKeys,
					 const QHash<QString, QVariant>& params )
{
  Q_ASSERT( m_recogniser );

  list<string> stl_featureKeys;
  foreach ( const QByteArray& key, featureKeys )
    stl_featureKeys.push_back( key.constData() );

  map<string, string> stl_params;
  QHashIterator<QString, QVariant> it( params );
  while ( it.hasNext() ) {
    it.next();
    stl_params[ it.key().toStdString() ] = it.value().toString().toStdString();
  }

  return m_recogniser->initTraining( stl_featureKeys, stl_params );
}

bool StlRecogniserWrapper::examineSample( const FeatureVec& featureVec,
					  const QSet<int>& classes ) {
  Q_ASSERT( m_recogniser );
  set<int> cs;
  foreach ( int id, classes ) cs.insert(id);
  return m_recogniser->examineSample( featureVec.toStdVector(), cs );
}

bool StlRecogniserWrapper::finaliseTraining() {
  Q_ASSERT( m_recogniser );
  return m_recogniser->finaliseTraining();
}


bool StlRecogniserWrapper::writeModelFile( const QString& fileName ) {
  Q_ASSERT( m_recogniser );
  return m_recogniser->writeModelFile( fileName.toStdString() );
}

bool StlRecogniserWrapper::readModelFile( const QString& fileName ) {
  Q_ASSERT( m_recogniser );
  return m_recogniser->readModelFile( fileName.toStdString() );
}


Stroke StlRecogniserWrapper::flatten( const StrokeList& strokes )
{
  Q_ASSERT( m_recogniser );

  StlStrokeList stl_strokes;
  foreach ( const Stroke& s, strokes ) {
    StlStroke stl_s;
    foreach ( const StrokePoint& pt, s )
      stl_s.push_back( StlStrokePoint(pt.x(), pt.y(),
				      pt.pressure, pt.milliTime) );
    stl_strokes.push_back( stl_s );
  }

  StlStroke stl_stroke = m_recogniser->flatten( stl_strokes );

  Stroke stroke;
  StlStroke::const_iterator it = stl_stroke.begin();
  while ( it != stl_stroke.end() ) {
    const StlStrokePoint& pt = *it;
    stroke += StrokePoint( pt.x, pt.y, pt.pressure, pt.milliTime );
    ++it;
  }

  return stroke;
}


ClassProbabilities StlRecogniserWrapper::classify( const FeatureVec& featureVec )
{
  Q_ASSERT( m_recogniser );
  StlClassProbabilities stl_probs
    = m_recogniser->classify( featureVec.toStdVector() );

  ClassProbabilities probs;
  StlClassProbabilities::const_iterator it = stl_probs.begin();
  while ( it != stl_probs.end() ) {
    probs.insert( (*it).first, (*it).second );
    ++it;
  }
  return probs;
}
