/*
 *  abstractcontroller.cpp
 *  Digest
 * 
 *  Imported into Digest by Aidan Lane on Thu Jun 9 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "abstractcontroller.h"

#include <QCoreApplication>

#include "abstractmodel.h"
#include "abstractview.h"


/*!
 * Constructs an abstract controller.
 */
AbstractController::AbstractController( QObject* objectPtr )
  : MvcBase(objectPtr),
    m_model(0)
{
}


/*!
 * Destroys the abstract controller.
 *
 * If the model() is non-null, then a MChangeControllerEvent will be posted to it
 * with null (0) as the parameter. Likewise, a VChangeControllerEvent will be
 * posted with null (0) as the parameter to each of the attached views().
 */
AbstractController::~AbstractController()
{
#if 0 // TODO: fixme:
  if ( !m_modelObject.isNull() )
    QCoreApplication::postEvent( m_modelObject,
				 new MChangeControllerEvent(0) );
      
  foreach ( AbstractView* v, m_views ) {
    Q_ASSERT( v != 0 );
    QCoreApplication::postEvent( v->objectPtr(),
				 new VChangeControllerEvent(0, this) );
  }
#endif
}


AbstractModel* AbstractController::model() const
{ // note: m_modelObject is a guarded pointer, m_model isn't
  return ( m_modelObject.isNull()
	   ? 0
	   : m_model );
}


/*!
 * Convenience method.
 *
 * Posts a CChangeModelEvent to itself with the given model.
 */
void AbstractController::setModel( AbstractModel* model )
{
  /* Don't just set m_model directly - sub-classes may re-implement,
   * changeModelEvent. Nor should we call changeModelEvent directly, as
   * the sub-class's vtable won't be ready yet. postEvent() will work, as
   * it queues the event and waits until the main event loop is reached again,
   * at which time the sub-class's vtable will be ready.
   */
  QCoreApplication::postEvent( objectPtr(), new CChangeModelEvent(model) );
}


/*!
 * Convenience method.
 *
 * Posts a CViewAttachEvent to itself with the given view.
 */
void AbstractController::attachView( AbstractView* view )
{
  QCoreApplication::postEvent( objectPtr(), new CViewAttachEvent(view) );
}


/*!
 * Convenience method.
 *
 * Posts a CViewDetachEvent to itself with the given view.
 */
void AbstractController::detachView( AbstractView* view )
{
  QCoreApplication::postEvent( objectPtr(), new CViewDetachEvent(view) );
}


/*!
 * The controller's event dispatcher.
 *
 * Asserts that the event is non-null.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 */
void AbstractController::dispatchEvent( CEvent* event )
{	
  Q_ASSERT( event != 0 );

  if ( event->moduleId() != MvcCore::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case CEvent::ChangeModel:
      changeModelEvent( static_cast<CChangeModelEvent*>(event) ); break;
    case CEvent::ViewAttach:
      viewAttachEvent( static_cast<CViewAttachEvent*>(event) ); break;
    case CEvent::ViewDetach:
      viewDetachEvent( static_cast<CViewDetachEvent*>(event) ); break;
    case CEvent::DetachAllViews:
      detachAllViewsEvent( event); break;
    default:
      break;
    }
}


/*!
 * Changes the model that the controller is to operate on.
 *
 * If the model() is non-null, this will post a MChangeControllerEvent to it
 * and a VModelChangedEvent to all attached views().
 *
 * Asserts that the \em event is non-null.
 */
void AbstractController::changeModelEvent( CChangeModelEvent* event )
{
  Q_ASSERT( event != 0 );

  m_model = event->model();
  m_modelObject = ( (m_model==0)
		    ? 0
		    : m_model->objectPtr() );

  if ( m_model != 0 )
    {
      QCoreApplication::postEvent( m_modelObject,
				   new MChangeControllerEvent(this) );
      
      foreach ( AbstractView* v, m_views ) {
	Q_ASSERT( v != 0 );
	QCoreApplication::postEvent( v->objectPtr(),
				     new VModelChangedEvent(m_model, this) );
      }
    }
}


/*!
 * Attaches a view to the controller.
 *
 * Posts a VChangeControllerEvent to the view.
 * However, a VModelChangedEvent is \em not sent. After receiving the former,
 * the view should post the latter to itself.
 *
 * Duplicates will be implicitly ignored, as the AbstractView pointers are stored
 * in a QSet.
 *
 * Asserts that both the \em event and the view \em event->view() are non-null.
 */
void AbstractController::viewAttachEvent( CViewAttachEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->view() != 0 );

  m_views.insert( event->view() );
  QCoreApplication::postEvent( event->view()->objectPtr(),
			       new VChangeControllerEvent(this, this) );
}


/*!
 * Detaches a view from the controller.
 *
 * Asserts that the \em event is non-null.
 */
void AbstractController::viewDetachEvent( CViewDetachEvent* event )
{
  Q_ASSERT( event != 0 );
  m_views.remove( event->view() );
}


/*!
 * Detaches all views from the controller.
 */
void AbstractController::detachAllViewsEvent( CEvent* )
{
  m_views.clear();
}
