/*
 *  doccontroller.cpp
 *  Digest
 * 
 *  Imported into Digest by Aidan Lane on Thu Jun 9 2005.
 *  Modifications Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  Nodal file:
 *
 *    projectcontroller.cpp
 *    Nodal
 *
 *    Imported into Nodal by Aidan Lane on Thu Feb 24 2005.
 *    Modifications Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *    Original file:
 *
 *      project.cpp
 *      EverGreen
 *
 *      Created by Aidan Lane on Wed Jul 14 2004.
 *      Copyright (c) 2004 Aidan Lane. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "doccontroller.h"

#include <QCoreApplication>
#include <QDomDocument>
#include <QDomElement>
#include <QFileInfo>

// TODO: remove unneded headers!
#include "abstractdocview.h"
#include "docevents.h"
#include "docviewevents.h"

// TODO: make the following properties of the doc controller?!?! - doc ?!?!?
#define CURRENT_FILEFORMAT_TYPE_TAG     "NodalNetwork"
#define CURRENT_FILEFORMAT_VERSION_TAG  "1.0"


/*!
 * Constructs a document controller with the given \em parent.
 */
DocController::DocController( QObject* parent )
  : QObject(parent),
    AbstractController(this)
{
}


/*!
 * Constructs a document controller with the given \em parent and uses \em doc as
 * its model.
 */
DocController::DocController( AbstractModel* doc, QObject* parent )
  : QObject(parent),
    AbstractController(this)
{
  setModel( doc );
}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * Doc* from AbstractModel*.
 *
 * See also AbstractController::model().
 */
Doc* DocController::doc() const {
  return c_doc;
}


/*!
 * The doc event dispatcher.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 *
 * Asserts that the \em event is non-null.
 */
void DocController::dispatchEvent( CEvent* event )
{
  Q_ASSERT( event != 0 );

  AbstractController::dispatchEvent( event );

  if ( event->moduleId() != MvcCore::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case CDocEvent::Rename:
      renameEvent( static_cast<CRenameEvent*>(event) ); break;
    case CFileEvent::FileSaveTo:
      fileSaveToEvent( static_cast<CFileEvent*>(event) ); break;
    case CFileEvent::FileInitFrom:
      fileInitFromEvent( static_cast<CFileEvent*>(event) ); break;
    default:
      break;
    }
}


/* Post a DocViewEvent to our views.
 * Any DocView object that has the same address as the value of "exclude"
 * will be exluded from the updates.
 * Therefore, set "exclude" to 0 if you don't want to exclude anything.
 * It asserts that the view pointer is non-null.
 * BTW: Yes, I know, this is very hacky :-)
 */
#define POST_VIEW_EVENT(EVENT,EXCLUDE)	{AbstractViewSetIterator i(views()); while(i.hasNext()) { AbstractView* v=i.next(); Q_ASSERT(v!=0); Q_ASSERT(v->objectPtr()!=0); if (v->objectPtr() != (EXCLUDE)) QCoreApplication::postEvent( v->objectPtr(), new EVENT ); }}

#define POST_MODIFIED_STATE_CHANGE_EVENTS(STATE) { if ( STATE != c_doc->isModified()) {QCoreApplication::postEvent( c_doc, new MChangeModifiedStateEvent(STATE) ); POST_VIEW_EVENT( VModifiedStateChangedEvent(STATE, this), 0 ); } }


/*!
 * Re-implemented, as to also have the pointer returned by doc() updated.
 *
 * Asserts that the \em event is non-null.
 */
void DocController::changeModelEvent( CChangeModelEvent* event )
{
  Q_ASSERT( event != 0 );
  AbstractController::changeModelEvent( event );
  c_doc = ( (event->model()==0)
	    ? 0
	    : qobject_cast<Doc*>(event->model()->objectPtr()) );
}


/*!
 * Will post VEvent::Renamed after making the change.
 *
 * Asserts that the \em event is non-null.
 */
void DocController::renameEvent( CRenameEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( c_doc != 0 );
  QCoreApplication::postEvent( c_doc, new MRenameEvent(event->name()) ); // change the doc's name
  POST_VIEW_EVENT( VRenamedEvent(event->name(), this),
		   event->updateSender() ? 0 : event->sender() );
  POST_MODIFIED_STATE_CHANGE_EVENTS( true ); // -> Doc has been modified
}


/*!
 * Note: To save a doc to file using the doc's current filename, pass the
 *       value of Doc::filename() to the CFileEvent.
 *
 * Asserts that the \em event is non-null and that the filename that the event
 * contains is non-empty.
 */
void DocController::fileSaveToEvent( CFileEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( c_doc != 0 );
  Q_ASSERT( ! event->filename().isEmpty() ); // we're being to saved for a specifc file, so the filename can't be empty!

  const QString& filename = event->filename();
  
  if ( ! saveDocToFile(filename) )
    return; // error -> can't go any further

  // Set the doc's name to the file's base name. FIXME: this may need to be changed later.
  // Note: the doc's name is used for the document window's title.
  // Warning: Don't use renameEvent(), as it causes problems with the modified state.
  QFileInfo fileInfo( filename );
  QString name = fileInfo.baseName(); // e.g. filename="/Users/bob/Foo.ngn", name=basename="Foo"
  QCoreApplication::postEvent( c_doc, new MRenameEvent(name) ); // change the doc's name
  POST_VIEW_EVENT( VRenamedEvent(name, this),
		   event->updateSender() ? 0 : event->sender() );

  QCoreApplication::postEvent( c_doc, new MChangeFilenameEvent(filename) ); // change the doc's filename
  POST_VIEW_EVENT( VFilenameChangedEvent(filename, this),
		   event->updateSender() ? 0 : event->sender() );

  POST_MODIFIED_STATE_CHANGE_EVENTS( false ); // *** Doc state == file state
}


/*!
 * CURRENT ASSUMPTION: Doc is currently empty!
 *
 * Asserts that the \em event is non-null and that the filename that the event
 * contains is non-empty.
 */
void DocController::fileInitFromEvent( CFileEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( c_doc != 0 );
  Q_ASSERT( ! event->filename().isEmpty() ); // we're being to load for a specifc file, so the filename can't be empty!
 
  if ( ! initDocFromFile(event->filename()) )
    return; // error -> can't go any further

  POST_MODIFIED_STATE_CHANGE_EVENTS( false ); // *** Doc state == file state
}



bool DocController::saveDocToFile( const QString& filename ) const
{

  // Create a DOM doc that describes the Doc's state
  Q_ASSERT( c_doc != 0 );
  QDomDocument doc( CURRENT_FILEFORMAT_TYPE_TAG );
  QDomElement root = doc.createElement( CURRENT_FILEFORMAT_TYPE_TAG );
  root.setAttribute( "version", CURRENT_FILEFORMAT_VERSION_TAG );
  root.appendChild( c_doc->domElement(doc) );
  doc.appendChild( root );

  // Attempt to save DOM to a file
  QFile file( filename );
  if ( ! file.open(QIODevice::WriteOnly) )
    return false;
  if ( ! file.write("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n")
       || ! file.write(doc.toByteArray()) ) { // write data encoded as UTF-8
    file.close();
    return false;
  }
  file.close();

  return true;
}


/*!
 * Posts VEvent(VEvent::Reset) to the views before returning.
 *
 * If the views' controllers are set to this class after this is called,
 * then \em resetViews can be set to false.
 *
 * Asserts that the \em doc() is non-null.
 */
bool DocController::initDocFromFile( const QString& filename, bool resetViews )
{
  Q_ASSERT( c_doc != 0 );

  QDomDocument doc( CURRENT_FILEFORMAT_TYPE_TAG );

  // Attempt to load DOM from a file
  QFile file( filename );
  if ( ! file.open(QIODevice::ReadOnly) )
    return false;
  if ( ! doc.setContent(&file) ) {
    file.close();
    return false;
  }
  file.close();

  // Attempt to set the Doc's state using the DOM
  // TODO: we really should do MIN and MAX version checks, at least warning the user.
  QDomElement root = doc.documentElement();
  if ( root.tagName() != CURRENT_FILEFORMAT_TYPE_TAG ) return false;
  QDomNode n = root.firstChild();
  while ( ! n.isNull() )
    {
      QDomElement e = n.toElement();   // try to convert the node to an element
      if ( ! e.isNull()                // check if it's really an element
	   && (e.tagName()             // check if the key's correct for the doc
	       == c_doc->key()) )      // - WARNING: don't use Doc::classKey()
	{
	  // INIT objects
	  c_doc->initFromDomElement( e, DomAccessibleState::InitStructure );

	  // INIT inter-object refs
	  c_doc->initFromDomElement( e, DomAccessibleState::InitStates );
      }
      n = root.nextSibling();
    }

  if ( resetViews )
    POST_VIEW_EVENT( VEvent(VEvent::Reset, this), 0 );

  return true;
}
