/*  -*- c++ -*-  (for Emacs)
 *
 *  doccontroller.h
 *  Digest
 * 
 *  Imported into Digest by Aidan Lane on Thu Jun 9 2005.
 *  Modifications Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  Nodal file:
 *
 *    projectcontroller.h
 *    Nodal
 *
 *    Imported into Nodal by Aidan Lane on Thu Feb 24 2005.
 *    Modifications Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *    Original file:
 *
 *      project.h
 *      EverGreen
 *
 *      Created by Aidan Lane on Wed Jul 14 2004.
 *      Copyright (c) 2004 Aidan Lane. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef DOCCONTROLLER_H
#define DOCCONTROLLER_H


#include <QObject>
#include "abstractcontroller.h"

#include <QCoreApplication> // TODO: move this stack to a separate class
#include <QList>
#include <QPointer>
#include <QStack> // TODO: move this stack to a separate class
#include <QString>

#include "doc.h"
#include "doccontrollerevents.h"


class DocViewBase;


/*!
 * \brief The DocController class is responsible for controlling the
 *        communication between a Doc and its views.
 *
 * Note that the doc's pointer is stored internally as a "guarded pointer".
 * Hence, if the doc is deleted, our reference to it will be automatically
 * set back to NULL.
 *
 * Warning! We always assert that the doc is non-null. Hence, you can't
 * have an instance of a DocController without a Doc. This is fine
 * for Nodal and means that we don't need to constantly check if the doc
 * is null in the release / deployment version (when assertions are stripped).
 */
class DocController : public QObject,
		      public AbstractController {

  Q_OBJECT

public:
  DocController( QObject* parent = 0 );
  DocController( AbstractModel* doc, QObject* parent = 0 );

  Doc* doc() const;

  // TODO: move this stack to a separate class
public slots:
  void undo()
  {
    if ( m_undoStack.isEmpty() ) return;

    UndoRedoItem* item = m_undoStack.pop();
    Q_ASSERT( item != 0 );

    // use sendEvent, so that other posted events don't jump in front of us
    Q_ASSERT( item->event() != 0 );
    QCoreApplication::sendEvent( this, item->event() );
    delete item->event();
    delete item;
  }
  void redo()
  {
    if ( m_redoStack.isEmpty() ) return;

    UndoRedoItem* item = m_redoStack.pop();
    Q_ASSERT( item != 0 );

    // use sendEvent, so that other posted events don't jump in front of us
    Q_ASSERT( item->event() != 0 );
    QCoreApplication::sendEvent( this, item->event() );
    delete item->event();
    delete item;
  }


protected:
  inline virtual void customEvent( QEvent* e ) {
    CEvent* ce = dynamic_cast<CEvent*>(e); // slow :-(
    if ( ce != 0 ) dispatchEvent(ce);
  }
  virtual void dispatchEvent( CEvent* );

  virtual void changeModelEvent( CChangeModelEvent* );

  virtual void renameEvent( CRenameEvent* );
  virtual void fileSaveToEvent( CFileEvent* );
  virtual void fileInitFromEvent( CFileEvent* );

  virtual bool saveDocToFile( const QString& filename ) const;
  virtual bool initDocFromFile( const QString& filename, bool resetViews = true );


  // TODO: move this stack to a separate class
  QStack<UndoRedoItem*> undoStack() const // can't return ref if thread-safe
  {
    // TODO: make me thread-safe
    return m_undoStack;
  }
  QStack<UndoRedoItem*> redoStack() const // can't return ref if thread-safe
  {
    // TODO: make me thread-safe
    return m_redoStack;
  }
  bool undoAvaliable() const
  {
    return !m_undoStack.isEmpty();
  }
  bool redoAvaliable() const
  {
    return !m_redoStack.isEmpty();
  }
  // If the event has the IsUndoEvent flag set, then the item will be added to
  // the undoStack(). If the IsRedoEvent is set, then it will be added to
  // the redoStack(). If both flags are set, then it will be added to the undoStack().
  // Note: owenership is transfered to DocController and will be deleted at its will.
  // Asserts that item->event() is non-null
  virtual void pushUndoRedoItem( UndoRedoItem* item )
  {
    // TODO: make me thread-safe
    Q_ASSERT( item->event() != 0 );
    CEvent::Flags flags = item->event()->flags();
    if ( flags & CEvent::IsUndoEvent )
      m_undoStack.push( item );
    else if ( flags & CEvent::IsRedoEvent )
      m_redoStack.push( item );
  }


private:
  // TODO: move this stack to a separate class
  QStack<UndoRedoItem*> m_undoStack;
  QStack<UndoRedoItem*> m_redoStack;

  // Temporary caches:
  QPointer<Doc> c_doc;
};


#endif // ! DOCCONTROLLER_H
