/*
 *  elementdoccontroller.cpp
 *  Digest
 * 
 *  Imported into Digest by Aidan Lane on Thu Jun 9 2005.
 *  Modifications Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  Nodal file:
 *
 *    projectcontroller.cpp
 *    Nodal
 *
 *    Imported into Nodal by Aidan Lane on Thu Feb 24 2005.
 *    Modifications Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *    Original file:
 *
 *      project.cpp
 *      EverGreen
 *
 *      Created by Aidan Lane on Wed Jul 14 2004.
 *      Copyright (c) 2004 Aidan Lane. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "elementdoccontroller.h"

#include <QCoreApplication>

#include "elementdoc.h"
#include "abstractelement.h"
#include "abstractelementattribute.h"
#include "abstractelementdocview.h"
#include "elementevents.h"


/*!
 * Constructs an element document controller with the given \em parent.
 */
ElementDocController::ElementDocController( QObject* parent )
  : AttributeDocController(parent)
{}


/*!
 * Constructs an element document controller with the given \em parent and uses
 * \em doc as its model.
 */
ElementDocController::ElementDocController( AbstractModel* doc, QObject* parent )
  : AttributeDocController(doc, parent)
{}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * ElementDoc* from AbstractModel*.
 *
 * See also AbstractController::model().
 */
ElementDoc* ElementDocController::elementDoc() const {
  return c_elementDoc;
}


/*!
 * This convenience slot method posts a CElementAttributeChangeDataEvent to this
 * controller with the given \em element, attribute (found using \em attributeType),
 * \em data, \em sender  and \em flags.
 *
 * Asserts that the \em element is non-null and that the element contains an
 * attribute that is of type \em attributeType.
 *
 * \b Note: This method is thread-safe.
 */
void ElementDocController::postElementAttributeChangeData( AbstractElement* element,
							   int attributeType,
							   const QVariant& data,
							   QObject* sender,
							   CEvent::Flags flags ) {
  Q_ASSERT( element != 0 );
  AbstractElementAttribute* a = element->attribute( attributeType );
  Q_ASSERT( a != 0 );
  QCoreApplication::postEvent( this,
			       new CElementAttributeChangeDataEvent
			       (element, a, data, sender, flags) );
}


/*!
 * The element doc controller event dispatcher.
 *
 * Asserts that the event is non-null.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 */
void ElementDocController::dispatchEvent( CEvent* event )
{
  Q_ASSERT( event != 0 );

  AttributeDocController::dispatchEvent( event );

  if ( event->moduleId() != MvcCore::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case CElementDocEvent::ElementAdd:
      elementAddEvent( static_cast<CElementAddEvent*>(event) );
      break;
    case CElementDocEvent::ElementRemove:
      elementRemoveEvent( static_cast<CElementRemoveEvent*>(event) );
      break;
    case CElementDocEvent::ElementAttributeAdd:
      elementAttributeAddEvent( static_cast<CElementAttributeAddEvent*>(event) );
      break;
    case CElementDocEvent::ElementAttributeRemove:
      elementAttributeRemoveEvent( static_cast<CElementAttributeRemoveEvent*>(event) );
      break;
    case CElementDocEvent::ElementAttributeChangeData:
      elementAttributeChangeDataEvent( static_cast<CElementAttributeChangeDataEvent*>(event) );
      break;
    case CElementDocEvent::ElementSetChangeOrder:
      elementSetChangeOrderEvent( static_cast<CElementSetChangeOrderEvent*>(event) );
      break;
    default:
      break;
    }
}



/* Post a DocViewEvent to our views.
 * Any DocView object that has the same address as the value of "exclude"
 * will be exluded from the updates.
 * Therefore, set "exclude" to 0 if you don't want to exclude anything.
 * It asserts that the view pointer is non-null.
 * BTW: I know, this is very hacky :-)
 */
#define POST_VIEW_EVENT(EVENT,EXCLUDE)	{AbstractViewSetIterator i(views()); while(i.hasNext()) { AbstractView* v=i.next(); Q_ASSERT(v!=0); Q_ASSERT(v->objectPtr()!=0); if (v->objectPtr() != (EXCLUDE)) QCoreApplication::postEvent( v->objectPtr(), new EVENT ); }}

#define POST_MODIFIED_STATE_CHANGE_EVENTS(STATE) { if ( STATE != c_elementDoc->isModified()) {QCoreApplication::postEvent( c_elementDoc, new MChangeModifiedStateEvent(STATE) ); POST_VIEW_EVENT( VModifiedStateChangedEvent(STATE, this), 0 ); } }



/*!
 * Re-implemented, as to also have the pointer returned by elementDoc() updated.
 *
 * Asserts that the \em event is non-null.
 */
void ElementDocController::changeModelEvent( CChangeModelEvent* event )
{
  Q_ASSERT( event != 0 );
  AttributeDocController::changeModelEvent( event );
  c_elementDoc
    = ( (event->model()==0)
	? 0
	: qobject_cast<ElementDoc*>(event->model()->objectPtr()) );
}



/*!
 * Asserts that the \em event and the \em event->element() are non-null.
 */
void ElementDocController::elementAddEvent( CElementAddEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->element() != 0 );

  Q_ASSERT( c_elementDoc != 0 );
  QCoreApplication::postEvent( c_elementDoc,
			       new MElementAddPtrEvent(event->element()) );

  POST_VIEW_EVENT( VElementAddedEvent( event->element(), this),
		   event->updateSender() ? 0 : event->sender() );
  POST_MODIFIED_STATE_CHANGE_EVENTS( true ); // -> Document has been modified
}


/*!
 * Asserts that both the \em event and \em event->element() are non-null.
 */
void ElementDocController::elementRemoveEvent( CElementRemoveEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->element() != 0 );
  AbstractElement* element = event->element();

  Q_ASSERT( c_elementDoc != 0 );
  QCoreApplication::postEvent( c_elementDoc,
			       new MElementRemovePtrEvent(element) );

  POST_VIEW_EVENT( VElementRemovedEvent(element, this),
		   event->updateSender() ? 0 : event->sender() );
  POST_MODIFIED_STATE_CHANGE_EVENTS( true ); // -> Document has been modified

  /*
   * Warning!!!
   * This is required, as the events posted above will refer to this element,
   * but they will only be processed when Qt gets back to its main loop...
   * If we used a normal delete, then they would refer to de/unallocated memory!
   */
  element->deleteLater(); // post a delete event
}


/*!
 * Adds the specified attribute to the given element and then informs the views.
 *
 * \b Warning: Because elements can only contain one of each type of attribute
 *             (i.e. no duplicates), any existing attribute of the same type that
 *             is to be added will be removed (and the views informed) before the
 *             new one is added.
 *
 * Asserts that the \em event, \em event->element() and \em event->attribute()
 * are all non-null.
 */
void ElementDocController::elementAttributeAddEvent( CElementAttributeAddEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->element() != 0 );
  Q_ASSERT( event->attribute() != 0 );

  AbstractElement* element = event->element();
  AbstractElementAttribute* attribute = event->attribute();

  // TODO: remove me
  Q_ASSERT( c_elementDoc );
  if ( !c_elementDoc->elements().contains(element) ) {
    qDebug("Warning: the document does not contain the element %p (yet?) "
	   "(elementAttributeAddEvent)", element);
    return;
  }

  // If the element already has an attribute of the given type, then remove that
  // first, as it can't have any duplicates.
  AbstractElementAttribute* existing = element->attribute( attribute->type() );
  if ( existing != 0 )
    {
      QCoreApplication::postEvent( element, new MEAttributeRemovePtrEvent(existing) );
      POST_VIEW_EVENT( VElementAttributeRemovedEvent(element, existing, this),
		       event->updateSender() ? 0 : event->sender() );
    }

  // Add the new attribute to the element.
  QCoreApplication::postEvent( element, new MEAttributeAddPtrEvent(attribute) );
  POST_VIEW_EVENT( VElementAttributeAddedEvent(element, attribute, this),
		   event->updateSender() ? 0 : event->sender() );

  POST_MODIFIED_STATE_CHANGE_EVENTS( true ); // as the document has been modified
}


/*!
 * Removes the specified attribute from the given element and then informs the views.
 *
 * Asserts that the \em event, \em event->element() and \em event->attribute()
 * are all non-null.
 */
void ElementDocController::elementAttributeRemoveEvent( CElementAttributeRemoveEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->element() != 0 );
  Q_ASSERT( event->attribute() != 0 );

  AbstractElementAttribute* attribute = event->attribute();

  // TODO: remove me
  Q_ASSERT( c_elementDoc );
  if ( !c_elementDoc->elements().contains(event->element()) ) {
    qDebug("Warning: the document does not contain the element %p (any longer?) "
	   "(elementAttributeRemoveEvent)", event->element());
    return;
  }

  QCoreApplication::postEvent( event->element(),
			       new MEAttributeRemovePtrEvent(attribute) );

  POST_VIEW_EVENT( VElementAttributeRemovedEvent(event->element(), attribute, this),
		   event->updateSender() ? 0 : event->sender() );

  POST_MODIFIED_STATE_CHANGE_EVENTS( true ); // as the document has been modified

  /*
   * Warning!!!
   * This is required, as the events posted above will refer to this attribute,
   * but they will only be processed when Qt gets back to its main loop...
   * If we used a normal delete, then they would refer to de/unallocated memory!
   */
  attribute->deleteLater(); // post a delete event
}


/*!
 * Updates the attribute and the views.
 *
 * Asserts that the \em event, \em event->element() and \em event->attribute()
 * are all non-null.
 */
void ElementDocController::elementAttributeChangeDataEvent( CElementAttributeChangeDataEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->element() != 0 );
  Q_ASSERT( event->attribute() != 0 );

  AbstractElement* element = event->element();
  AbstractElementAttribute* attribute = event->attribute();
  QVariant currentData = attribute->toVariant();

  // TODO: remove me
  Q_ASSERT( c_elementDoc );
  if ( !c_elementDoc->elements().contains(element) ) {
    qDebug("Warning: the document does not contain the element %p (any longer?) "
	   "(elementAttributeChangeDataEvent)", element);
    return;
  }

  if ( currentData != event->data() )
    {
      pushUndoRedoItem( new UndoRedoItem
			(new CElementAttributeChangeDataEvent
			 (element, attribute, currentData, this,
			  (event->isUndoEvent() ? CEvent::IsRedoEvent : CEvent::IsUndoEvent))) );
      attribute->setData( event->data() );
      POST_VIEW_EVENT( VElementAttributeDataChangedEvent(element, attribute, this),
		       event->updateSender() ? 0 : event->sender() );
      POST_MODIFIED_STATE_CHANGE_EVENTS( true ); // -> Doc has been modified
  }
}


/*!
 * Changes the (Z) order of the given \em event->elements() in respect to
 * elementDoc()->elements() as specified by the given \em event->action().
 *
 * Asserts that the \em event is non-null.
 */
void ElementDocController::elementSetChangeOrderEvent( CElementSetChangeOrderEvent* event )
{
  Q_ASSERT( event != 0 );
  QCoreApplication::postEvent( c_elementDoc,
			       new MElementSetChangeOrderEvent
			       (event->elements(), event->action()) );
  POST_VIEW_EVENT( VElementSetOrderChangedEvent(event->elements(), event->action(), this),
		   event->updateSender() ? 0 : event->sender() );
  POST_MODIFIED_STATE_CHANGE_EVENTS( true ); // -> Document has been modified
}
