/*
 *  abstractelement.cpp
 *  Digest
 * 
 *  Imported into Digest by Aidan Lane on Thu Jun 9 2005.
 *  Modifications Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  Nodal file:
 *
 *    pelement.cpp
 *
 *    Created by Aidan Lane on Wed Apr 6 2005.
 *    Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "abstractelement.h"

#include <QCoreApplication>
#include <QDebug>
#include <QReadLocker>

#include "elementdoc.h"
#include "abstractelementattribute.h"


#define INSTANCE_ID_TAG     "id"
#define ATTRIBUTE_LIST_TAG  "attributes"


static const QSet<int> s_nullSet;


/*!
 * Constructs a graph element with a doc controller and a type.
 *
 * \b Warning:
 * The instanceId \em must be unique across the set of elements for the entire
 * document. This is asserted if the \em doc is non-null.
 *
 * The element will become a child of the \em doc in the QObject hierarchy, given
 * that it's non-null.
 *
 * Knowning who the \em doc is means that we can query it and post to its
 * controller (using Doc::controller() ).
 *
 * However, this does \b not assert that the \em doc is non-null, as there are
 * circumstances where temporary elements need to be created that are parent-less.
 */
AbstractElement::AbstractElement( quint32 instanceId, ElementDoc* doc )
  : QObject(doc),
    m_instanceId(instanceId),
    m_doc(doc)
{
  Q_ASSERT( doc == 0                                         // null, otherwise:
	    || !m_doc->idToElement().contains(instanceId)    // ID MUST be unique
	    || m_doc->idToElement().value(instanceId)==0 );  // allow reservations
}


/*!
 * Convenience constructor. It behaves essentially like the above function.
 *
 * The instance ID is auto-generated using ElementDoc::genElementInstanceId()
 * if the \em doc is non-null.
 */
AbstractElement::AbstractElement( ElementDoc* doc )
  : QObject(doc),
    m_instanceId(0),
    m_doc(doc)
{
  if ( doc != 0 )
    m_instanceId = m_doc->genElementInstanceId();
}


/*!
 * Destroys the document element.
 */
AbstractElement::~AbstractElement()
{
  // WARNING! The attributes are QObject's that should be owned by this element.
  //          Hence, Qt will look after the memory de-allocation of them for us.
}


ElementDoc* AbstractElement::doc() const {
  return m_doc;
}


/*!
 * Convenience method.
 *
 * Simply returns the value of doc()->controller().
 */
// TODO: cache me!
AbstractController* AbstractElement::controller() const {
  Q_ASSERT( m_doc != 0 );
  return m_doc->controller();
}


/*!
 * Returns a hash of attributes, where for each entry, the key is the attribute
 * type and the value is a pointer to the corresponding attribute instance.
 *
 * Using a hash instead of a list helps us to quickly find an instance for a
 * given type of attribute, such as for editors/inspectors that are specific to
 * a given type of attribute, for sub-classes that need to update an attribute
 * of a given type, etc.
 *
 * \b Note: This \em intentionally does not return a const ref, as to ensure
 *          thread safety.
 *          However, Qt's implicit sharing should help to maintain performance.
 *
 * \b Note: The \em type is an int, as to aid extensibility.
 *
 */
// WARNING: See the doc above for why we don't return a const ref!
QHash<int, AbstractElementAttribute*> AbstractElement::attributes() const {
  QReadLocker locker( &m_attributesLock );
  return m_attributes;
}


/*!
 * Convenience method.
 *
 * Returns the pointer to the element's attribute that has the given \em type.
 *
 * Returns null if the attribute can't be found.
 *
 * \b Note: The \em type is an int, as to aid extensibility.
 */
AbstractElementAttribute* AbstractElement::attribute( int type ) const {
  QReadLocker locker( &m_attributesLock );
  return m_attributes.value( type );
}


/*!
 * Convenience method.
 *
 * Returns value of the element's attribute that has the given \em type.
 *
 * Returns an invalid variant if the attribute can't be found.
 *
 * \b Note: The \em type is an int, as to aid extensibility.
 */
QVariant AbstractElement::attributeData( int type ) const {
  QReadLocker locker( &m_attributesLock );
  AbstractElementAttribute* a = m_attributes.value( type );
  return ( a != 0
	   ? a->toVariant()
	   : QVariant() );
}


const QSet<int>& AbstractElement::requiredAttributes() const {
  return s_nullSet;
}


const QSet<int>& AbstractElement::additionalAttributes() const {
  return s_nullSet;
}


/*!
 * This will create and add an instance of each attribute type that is listed by
 * requiredAttributes() that is currently not in the attributes() collection.
 *
 * \b Warning: This can't be called by the AbstractElement's constructors,
 *             as it relies on the sub-class' v-table to have been created,
 *             as it calls requiredAttributes() and createAttribute(),
 *             which are (pure-)virtual in this class.
 *             Hence, it must be called by the sub-class' constructor().
 */
void AbstractElement::addMissingRequiredAttributes()
{
  QWriteLocker locker( &m_attributesLock );

  QSet<int> removeSet = requiredAttributes();
  removeSet -= m_attributes.keys().toSet();

  foreach ( int type, removeSet ) {
    AbstractElementAttribute* a = createAttribute( type );
    Q_ASSERT( a != 0 );
    m_attributes.insert( type, a );
  }
}


/*!
 * This convenience method will try to set the \em data for a given attribute
 * \em type. If the attribute is found, the data will be set and this will return
 * true, otheriwise this will return false.
 *
 * This will also return false if the attribute's setData() method fails.
 */
bool AbstractElement::trySetAttributeData( int type, const QVariant& data )
{
  QReadLocker locker( &m_attributesLock );
  AbstractElementAttribute* a = m_attributes.value( type );
  return ( a != 0
	   && a->setData(data) );
}


/*!
 * The elements's event dispatcher.
 *
 * Receives events and then passes them off to be processed by appropiate
 * (virtual) methods.
 *
 * \b Warning: It is assumed that the events have their correct type set
 *             (as it uses static casts).
 *
 * Asserts that the \em event is non-null.
 */
void AbstractElement::dispatchEvent( MEvent* event )
{
  Q_ASSERT( event != 0 );

  if ( event->moduleId() != MvcCore::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case MEElementEvent::AttributeAddPtr:
      attributeAddPtrEvent( static_cast<MEAttributeAddPtrEvent*>(event) ); break;
    case MEElementEvent::AttributeRemovePtr:
      attributeRemovePtrEvent( static_cast<MEAttributeRemovePtrEvent*>(event) ); break;
    default:
      return;
    }
}


/*!
 * Appends an attribute to the element's internal list.
 *
 * Warning! At this level, multiple attributes of the same type may be added -
 *          as some applications may require this.
 *
 * Asserts that the event (e) and the attribute pointer in it are both non-null.
 */
void AbstractElement::attributeAddPtrEvent( MEAttributeAddPtrEvent* e )
{
  Q_ASSERT( e != 0 );
  Q_ASSERT( e->attribute() != 0 );
  QWriteLocker locker( &m_attributesLock );
  m_attributes.insert( e->attribute()->type(), e->attribute() );
}


/*!
 * This WON'T delete the attribute (as in won't free its memory).
 * All memory (de)allocation is to be handled by the document's controller
 * (e.g. ElementDocController).
 * This is so that the controller can remove the pointer from the document
 * (by calling this method) and THEN update the views (which may still refer to
 * attribute), followed by calling deleteLater() on the attribute.
 *
 * Asserts that the event (e) and the attribute pointer in it are both non-null.
 */
void AbstractElement::attributeRemovePtrEvent( MEAttributeRemovePtrEvent* e )
{
  Q_ASSERT( e != 0 );
  Q_ASSERT( e->attribute() != 0 );

  QWriteLocker locker( &m_attributesLock );

  AbstractElementAttribute* attr = e->attribute();
  int attrType = attr->type();

  /* Remove our record of the attribute
   *
   * From the Qt 4.0.0 doc:
   * "If the hash contains multiple items with key key, this function
   * [QHash::find()] returns an iterator that points to the most recently
   * inserted value. The other values are accessible by incrementing the
   * iterator."
   */
  QHash<int, AbstractElementAttribute*>::iterator it
    = m_attributes.find(attrType); // go to first of the given type
  while ( it != m_attributes.end() && it.key() == attrType ) {
    if ( it.value() == attr ) {
      m_attributes.erase( it );
      break;
    }
    ++it;
  }
}


/*!
 * Creates an XML QDomElement that represents the AbstractElement.
 *
 * Use initFromDOMElement() to restore the AbstractElement state from the
 * resulting QDomElement.
 *
 * See also initFromDomElement().
 */
QDomElement AbstractElement::domElement( QDomDocument& doc ) const
{
  QDomElement root = DomAccessibleState::domElement( doc );

  // Set the mandatory instanceId attribute
  root.setAttribute( INSTANCE_ID_TAG, m_instanceId );

  // Append an element for each AbstractElementAttribute, if there are any
  m_attributesLock.lockForRead();
  if ( ! m_attributes.isEmpty() ) {
    QDomElement e = doc.createElement( ATTRIBUTE_LIST_TAG );
    foreach ( AbstractElementAttribute* attr, m_attributes ) {
      Q_ASSERT( attr != 0 );
      e.appendChild( attr->domElement(doc) );
    }
    root.appendChild( e );
  }
  m_attributesLock.unlock();

  return root;
}


/*!
 * Restore the AbstractElement state from a QDomElement created by domElement().
 *
 * This will remove any existing attributes first if \em modeFlags has its
 * InitStructure bit set.
 *
 * \b Note: This calls addMissingRequiredAttributes() before returning, as to
 *          ensure that the element is in a "safe state".
 *          This is very helpful for when a file is loaded that was created using
 *          an older version of the software that had a requiredAttributes() set
 *          that was different in some way to the current set (e.g. because new
 *          features where added).
 *
 * \b Important:
 * Because the exact set of classes dervied from AbstractElementAttribute that
 * can be used with the given sub-classed AbstractElement varies, the creation
 * of the (AbstractElementAttribute derived) attributes is left up to
 * AbstractElement's sub-classes.
 * This is performed via calls to createAttribute(), which must be implemented
 * in sub-classes.
 *
 * Note: Returns immediately if e.tagName() != key().
 *
 * See also domElement() and DomAccessibleState::initFromDomElement().
 */
void AbstractElement::initFromDomElement( const QDomElement& e,
					  InitModeFlags modeFlags )
{
  if ( e.tagName() != key() ) return;

  DomAccessibleState::initFromDomElement( e, modeFlags );

  if ( modeFlags & InitStructure )
    { // Remove any existing attributes
      m_attributesLock.lockForWrite(); // - lock for WRITE
      qDeleteAll( m_attributes.values() );
      m_attributes.clear();
      m_attributesLock.unlock();       // - unlock
    }

  // Get the DOM attributes
  bool toUIntOk = false;
  m_instanceId = e.attribute(INSTANCE_ID_TAG).toUInt(&toUIntOk);
  Q_ASSERT( toUIntOk );

  // Get any AbstractElementAttributes
  QDomNode n = e.firstChild();
  while ( ! n.isNull() )
    {
      QDomElement childElement = n.toElement();   // try to convert the node to an element
      if ( ! childElement.isNull()                    // check if it's really an element
	   && childElement.tagName() == ATTRIBUTE_LIST_TAG )
	{
	  // Traverse DOM node list 
	  QDomNode cc = childElement.firstChild();
	  while ( ! cc.isNull() ) {
	    QDomElement ccElement = cc.toElement();
	    if ( ! ccElement.isNull() )
	      {
		AbstractElementAttribute* attr = 0;
		if ( modeFlags & InitStructure )
		  {
		    attr = createAttribute( ccElement.tagName().toAscii() );
		    if ( attr == 0 ) {
		      qDebug() << "Unsupported attribute type:" << ccElement.tagName().toAscii(); // TODO: handle me properly!
		      cc = cc.nextSibling();
		      continue; // skip it!
		    }
		    attr->initFromDomElement( ccElement, InitStructure );
		    m_attributesLock.lockForWrite(); // - lock for WRITE
		    m_attributes.insert( attr->type(), attr );
		    m_attributesLock.unlock();       // - unlock
		  }
		// WARNING: the following should be "if", not "else if"
		if ( modeFlags & InitStates )
		  {
		    m_attributesLock.lockForRead(); // - lock for READ
		    attr = m_attributes.value( attributeKeyToType(ccElement.tagName().toAscii()) );
		    if ( attr != 0 )
		      attr->initFromDomElement( ccElement, InitStates );
		    m_attributesLock.unlock();       // - unlock
		  }
	      }
	    cc = cc.nextSibling();
	  }
	}
      n = n.nextSibling();
    }

  // Ensure that the element is in a "safe state"
  addMissingRequiredAttributes();
}
