/*  -*- c++ -*-  (for Emacs)
 *
 *  abstractelement.h
 *  Digest
 * 
 *  Imported into Digest by Aidan Lane on Thu Jun 9 2005.
 *  Modifications Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  Nodal file:
 *
 *    pelement.h
 *
 *    Created by Aidan Lane on Wed Apr 6 2005.
 *    Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

// See ElementDoc.cpp for why AbstractElement needs to be a derivative of QObject (we need deleteLater()) - among other reasons

#ifndef ABSTRACTELEMENT_H
#define ABSTRACTELEMENT_H


#include <QObject>
#include "domaccessiblestate.h"

#include <QHash>
#include <QReadLocker>
#include <QReadWriteLock>

#include "elementevents.h"

class QByteArray;

class ElementDoc;
class AbstractController;


/*!
 * This is the base class for all graph elements.
 *
 * An object cannot be instantiated from it directly, you must derive from it.
 */
class AbstractElement : public QObject, public DomAccessibleState {

  Q_OBJECT

public:
  virtual ~AbstractElement();

  // The following are pure-virtual, as to force sub-classes to implement them
  virtual int type() const = 0; // returns int, not Type, as to aid extensibility
  virtual const QByteArray& key() const = 0;
  virtual const QString& title() const = 0;

  quint32 instanceId() const { return m_instanceId; }

  ElementDoc* doc() const;
  AbstractController* controller() const;

  QHash<int, AbstractElementAttribute*> attributes() const;
  AbstractElementAttribute* attribute( int type ) const;
  template<class T> inline T* attribute() const;
  QVariant attributeData( int type ) const;

  virtual const QSet<int>& requiredAttributes() const;
  virtual const QSet<int>& additionalAttributes() const;

  virtual QDomElement domElement( QDomDocument& doc ) const;
  virtual void initFromDomElement( const QDomElement& e, InitModeFlags modeFlags );


protected:
  AbstractElement( quint32 instanceId, ElementDoc* abstractDoc );
  AbstractElement( ElementDoc* abstractDoc );

  void addMissingRequiredAttributes();
  bool trySetAttributeData( int type, const QVariant& data );

  /*! Note: The \em type is an int, as to aid extensibility. */
  virtual AbstractElementAttribute* createAttribute( int type ) = 0;
  virtual AbstractElementAttribute* createAttribute( const QByteArray& key ) = 0;
  virtual int attributeKeyToType( const QByteArray& key ) = 0;

  inline virtual void customEvent( QEvent* e ) {
    MEvent* me = dynamic_cast<MEvent*>(e); // slow :-(
    if ( me != 0 ) dispatchEvent(me);
  }
  virtual void dispatchEvent( MEvent* );

  virtual void attributeAddPtrEvent( MEAttributeAddPtrEvent* );
  virtual void attributeRemovePtrEvent( MEAttributeRemovePtrEvent* );


private:
  // You may get the following variables with domElement() and set them with
  // initFromDOMElement().
  quint32 m_instanceId;
  QHash<int, AbstractElementAttribute*> m_attributes; // type -> instance
  mutable QReadWriteLock m_attributesLock;

  // The following variables are for temporary use only.
  ElementDoc* m_doc;
};


/*!
 * Convenience method.
 *
 * Returns a pointer to the element's attribute for the given class
 * (which dictates the type).
 *
 * Returns null if the attribute can't be found.
 */
template<class T> 
T* AbstractElement::attribute() const {
  // Note: the static cast should be fairly safe, given that we specify an enum'd type
  QReadLocker locker( &m_attributesLock );
  return static_cast<T*>( m_attributes.value(T::classType()) );
}


#endif // ! ABSTRACTELEMENT_H
