/*
 *  abstractmodel.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jun 9 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "abstractmodel.h"

#include <QCoreApplication>

#include "abstractcontroller.h"
#include "controllerevents.h"


/*!
 * Constructs an abstract model.
 */
AbstractModel::AbstractModel( QObject* objectPtr )
  : MvcBase(objectPtr),
    m_controller(0)
{
}


/*!
 * Destroys the abstract model.
 *
 * If the controller() is non-null, this will post a CChangeModelEvent to it with
 * null (0) as the parameter.
 */
AbstractModel::~AbstractModel()
{
  if ( !m_controllerObject.isNull() )
    QCoreApplication::postEvent( m_controllerObject, new CChangeModelEvent(0) );
}


/*!
 * Returns the model's controller.
 */
AbstractController* AbstractModel::controller() const
{ // note: m_controllerObject is a guarded pointer, m_controller isn't
  return ( m_controllerObject.isNull()
	   ? 0
	   : m_controller );
}


/*!
 * Convenience method.
 *
 * Equivalent to:
 * \code
 * QCoreApplication::postEvent( controller()->objectPtr(), event );
 * \endcode
 */
void AbstractModel::postControllerEvent( CEvent* event ) {
  Q_ASSERT( ! m_controllerObject.isNull() );
  QCoreApplication::postEvent( m_controllerObject, event );
}


/*!
 * The model's event dispatcher.
 *
 * Asserts that the event is non-null.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 */
void AbstractModel::dispatchEvent( MEvent* event )
{	
  Q_ASSERT( event != 0 );

  if ( event->moduleId() != MvcCore::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case MEvent::ChangeController:
      changeControllerEvent( static_cast<MChangeControllerEvent*>(event) );
      break;

    default:
      break;
    }
}


/*!
 * Changes the controller that the model should post messages to.
 *
 * Typically, only a controller should post a MChangeControllerEvent to a model,
 * as a result of a CChangeModelEvent being posted to it.
 *
 * Asserts that the \em event is non-null.
 */
void AbstractModel::changeControllerEvent( MChangeControllerEvent* event )
{
  Q_ASSERT( event != 0 );
  m_controller = event->controller();
  m_controllerObject = ( (m_controller==0)
			 ? 0
			 : m_controller->objectPtr() );
}
