/*  -*- c++ -*-  (for Emacs)
 *
 *  attributedoc.cpp
 * 
 *  Created by Aidan Lane on Mon Dec 19 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "attributedoc.h"

#include <QReadLocker>
#include <QWriteLocker>

#include "attributedoccontroller.h"


/*!
 * Default constructor.
 */
AttributeDoc::AttributeDoc( QObject* parent )
  : Doc(parent)
{}


/*!
 * Convenience constructor.
 */
AttributeDoc::AttributeDoc( const QString& name, QObject* parent )
  : Doc(name, parent)
{}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * AttributeDocController* from AbstractController*.
 *
 * See also AbstractModel::controller().
 */
AttributeDocController* AttributeDoc::attributeDocController() const {
  return c_attributeDocController;
}


/*!
 * \b Note: This \em intentionally does not return a const ref, as to ensure
 *          thread safety.
 *          However, Qt's implicit sharing should help to maintain performance.
 *
 * \b Note: This method is thread-safe.
 */
QHash<int, QVariant> AttributeDoc::attributes() const {
  QReadLocker locker( &m_attributesLock );
  return m_attributes;
}


/*!
 * \b Note: This \em intentionally does not return a const ref, as to ensure
 *          thread safety.
 *          However, Qt's implicit sharing should help to maintain performance.
 *
 * \b Note: This method is thread-safe.
 */
QVariant AttributeDoc::attributeValue( int type ) const {
  QReadLocker locker( &m_attributesLock );
  return m_attributes.value( type );
}


/*!
 * \b Note: This method is thread-safe.
 */
bool AttributeDoc::containsAttribute( int type ) const {
  QReadLocker locker( &m_attributesLock );
  return m_attributes.contains( type );
}


/*!
 * The model's event dispatcher.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 *
 * Asserts that the \em event is non-null.
 */
void AttributeDoc::dispatchEvent( MEvent* event )
{	
  Q_ASSERT( event != 0 );

  Doc::dispatchEvent( event );

  if ( event->moduleId() != MvcCore::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case MAttributeDocEvent::AttributesSetValues:
      attributesSetValuesEvent( static_cast<MAttributesSetValuesEvent*>(event) ); break;
    case MAttributeDocEvent::AttributesRemove:
      attributesRemoveEvent( static_cast<MAttributesRemoveEvent*>(event) ); break;
    default:
      break;
    }
}


/*!
 * Re-implemented, as to also have the pointer returned by
 * attributeDocController() updated.
 *
 * Asserts that the \em event is non-null.
 */
void AttributeDoc::changeControllerEvent( MChangeControllerEvent* event )
{
  Q_ASSERT( event != 0 );
  Doc::changeControllerEvent( event );
  c_attributeDocController
    = ( (event->controller()==0)
	? 0
	: qobject_cast<AttributeDocController*>(event->controller()->objectPtr()) );
}


/*!
 * Asserts that the \em event is non-null.
 *
 * \b Note: This method is thread-safe.
 */
void AttributeDoc::attributesSetValuesEvent( MAttributesSetValuesEvent* event ) {
  Q_ASSERT( event != 0 );
  QWriteLocker locker( &m_attributesLock );
  m_attributes.unite( event->attributes() );
}


/*!
 * Asserts that the \em event is non-null.
 *
 * \b Note: This method is thread-safe.
 */
void AttributeDoc::attributesRemoveEvent( MAttributesRemoveEvent* event ) {
  Q_ASSERT( event != 0 );
  QWriteLocker locker( &m_attributesLock );
  foreach ( int type, event->attributeTypes() )
    m_attributes.remove( type );
}
