/*  -*- c++ -*-  (for Emacs)
 *
 *  abstractelementattribute.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Tue Sep 6 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "abstractelementattribute.h"

#include <QReadLocker>
#include <QWriteLocker>


AbstractElementAttribute::AbstractElementAttribute( AbstractElement* parent )
  : QObject(parent)
{}


/*!
 * Returns the attribute's state as a QString.
 *
 * This method is required by ElementDoc::domElement(), which itself is
 * used for save & load. Hence, the QString returned should be compact while
 * also easy enough for setData() to parse. Speed is not a huge concern in the
 * implementation of this method.
 *
 * Note: This method is required even alongside toVariant(). This is because
 *       although toVariant().toString() will work for some (primitive) data
 *       types, it will not work for others, such as QRect, QList<QVariant>
 *       and user defined data types.
 *
 * \b Note: This function is \em thread-safe.
 */
QString AbstractElementAttribute::toString() const {
  m_lock.lockForRead();
  QString ret = worker_toString();
  m_lock.unlock();
  return ret;
}


/*!
 * Sets the attribute's state using the data contained in the given QString.
 *
 * This method is required by ElementDoc::initFromDomElement(), which
 * itself is used for save & load. Speed is not a huge concern in the
 * implementation of this method. Instead it must simply parse data that has
 * been generated using toString().
 *
 * Returns false if an error occurred; else true.
 *
 * \b Note: This function is \em thread-safe.
 */
bool AbstractElementAttribute::setData( const QString& data ) {
  m_lock.lockForWrite();
  bool ret = worker_setData( data );
  m_lock.unlock();
  return ret;
}


/*!
 * Returns the attribute's state as a QVariant.
 *
 * This is used for getting the attribute's state and will most likey be called
 * often. Hence, the implementation of this method must be optimised for
 * speed.
 *
 * \b Note: This function is \em thread-safe.
 */
QVariant AbstractElementAttribute::toVariant() const {
  m_lock.lockForRead();
  QVariant ret = worker_toVariant();
  m_lock.unlock();
  return ret;
}


/*!
 * Sets the attribute's state using the data contained in the given QVariant.
 *
 * This is used for setting the attribute's state and will most likey be called
 * often. Hence, the implementation of this method must be optimised for
 * speed.
 *
 * Returns false if an error occurred; else true.
 *
 * \b Note: This function is \em thread-safe.
 */
bool AbstractElementAttribute::setData( const QVariant& data ) {
  m_lock.lockForWrite();
  bool ret = worker_setData( data );
  m_lock.unlock();
  return ret;
}


/*!
 * Creates an XML QDomElement that represents the AbstractElementAttribute.
 *
 * Use initFromDOMElement() to restore the AbstractElementAttribute state
 * from the resulting QDomElement.
 *
 * Uses toString() to get the state.
 *
 * \b Note: This function is \em thread-safe
 *    (as it only calls toString()).
 *
 * See also initFromDomElement().
 */
QDomElement AbstractElementAttribute::domElement( QDomDocument& doc ) const
{
  QDomElement root = DomAccessibleState::domElement( doc );
  root.appendChild( doc.createTextNode(toString()) );
  return root;
}


/*!
 * Restore the AbstractElementAttribute state from a QDomElement created by
 * domElement().
 *
 * Note: Returns immediately if e.tagName() != key().
 *
 * Uses setData(const QString&) to set the state.
 *
 * \b Note: This function is \em thread-safe
 *          (as it only calls setData(const QString&)).
 *
 * See also domElement() and DomAccessibleState::initFromDomElement().
 */
void AbstractElementAttribute::initFromDomElement( const QDomElement& e,
						      InitModeFlags modeFlags )
{
  if ( e.tagName() != key() ) return;

  DomAccessibleState::initFromDomElement( e, modeFlags );

  if ( modeFlags & InitStates  )
    {
      QDomNode n = e.firstChild();
      while ( ! n.isNull() ) {
	if ( n.isText() ) {
	  setData( n.toText().data() );
	  break; // there should only be one occurrence anyway
	}
	n = n.nextSibling();
      }
    }
}
