/*
 *  doc.cpp
 *  Digest
 * 
 *  Imported into Digest by Aidan Lane on Thu Jun 9 2005.
 *  Modifications Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  Nodal file:
 *
 *    project.cpp
 *    Nodal
 *
 *    Imported into Nodal by Aidan Lane on Thu Feb 24 2005.
 *    Modifications Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *    Original file:
 *
 *      project.cpp
 *      EverGreen
 *
 *      Created by Aidan Lane on Wed Jul 14 2004.
 *      Copyright (c) 2004 Aidan Lane. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "doc.h"

#include <QByteArray>

#include "doccontroller.h"


#define NAME_TAG  "name"


/*!
 * Default constructor.
 */
Doc::Doc( QObject* parent )
  : QObject(parent),
    AbstractModel(this),
    m_modified(false)
{}


/*!
 * Convenience constructor.
 */
Doc::Doc( const QString& name, QObject* parent )
  : QObject(parent),
    AbstractModel(this),
    m_name(name),
    m_modified(false)
{}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * DocController* from AbstractController*.
 *
 * See also AbstractModel::controller().
 */
DocController* Doc::docController() const {
  return c_docController;
}


/*!
 * The model's event dispatcher.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 *
 * Asserts that the \em event is non-null.
 */
void Doc::dispatchEvent( MEvent* event )
{	
  Q_ASSERT( event != 0 );

  AbstractModel::dispatchEvent( event );

  if ( event->moduleId() != MvcCore::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case MDocEvent::Rename:
      renameEvent( static_cast<MRenameEvent*>(event) ); break;
    case MDocEvent::ChangeFilename:
      changeFilenameEvent( static_cast<MChangeFilenameEvent*>(event) ); break;
    case MDocEvent::ChangeModifiedState:
      changeModifiedStateEvent( static_cast<MChangeModifiedStateEvent*>(event) ); break;
    default:
      break;
    }
}


/*!
 * Re-implemented, as to also have the pointer returned by docController()
 * updated.
 *
 * Asserts that the \em event is non-null.
 */
void Doc::changeControllerEvent( MChangeControllerEvent* event )
{
  Q_ASSERT( event != 0 );
  AbstractModel::changeControllerEvent( event );
  c_docController
    = ( (event->controller()==0)
	? 0
	: qobject_cast<DocController*>(event->controller()->objectPtr()) );
}


/*!
 * Changes the doc's name. 
 *
 * Asserts that the \em event is non-null.
 */
void Doc::renameEvent( MRenameEvent* event )
{
  Q_ASSERT( event != 0 );
  m_name = event->name();
}


/*!
 * Changes the doc's filename.
 *
 * Asserts that the \em event is non-null.
 */
void Doc::changeFilenameEvent( MChangeFilenameEvent* event )
{
  Q_ASSERT( event != 0 );
  m_filename = event->filename();
}


/*!
 * Changes the doc's modification state.
 * If "true", then it will indicate that the doc has been modified since it
 * was lasted saved.
 *
 * Asserts that the \em event is non-null.
 */
void Doc::changeModifiedStateEvent( MChangeModifiedStateEvent* event )
{
  Q_ASSERT( event != 0 );
  m_modified = event->documentModified();
}


const QByteArray& Doc::classKey() {
  static const QByteArray key( "doc" );
  return key;
}


/*!
 * Creates an XML QDomElement that represents the Doc.
 *
 * Use initFromDOMElement() to restore the Doc state from the resulting
 * QDomElement.
 */
QDomElement Doc::domElement( QDomDocument& doc ) const
{
  QDomElement root = DomAccessibleState::domElement( doc );

  // Set "Name" attribute IFF it's not empty
  if ( ! m_name.isEmpty() )
    root.setAttribute( NAME_TAG, m_name );

  return root;
}


/*!
 * Restore the Doc state from a QDomElement created by domElement().
 *
 * CURRENT ASSUMPTION: If (mode & BuildObjectStrcuture)==1 then Doc
 *                     is assumed and asserted to be currently empty!
 *
 * Please see DomAccessibleState::initFromDomElement() for more information.
 *
 * Note: Returns immediately if e.tagName() != typeName().
 */
void Doc::initFromDomElement( const QDomElement& e, InitModeFlags modeFlags )
{
  DomAccessibleState::initFromDomElement( e, modeFlags );

  if ( e.tagName() != key() ) return;

  if ( modeFlags & InitStructure )
    {
      // Get the attributes
      m_name = e.attribute( NAME_TAG );
    }
}
