/*
 *  elementdoc.cpp
 *  Digest
 * 
 *  Imported into Digest by Aidan Lane on Thu Jun 9 2005.
 *  Modifications Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  Nodal file:
 *
 *    project.cpp
 *    Nodal
 *
 *    Imported into Nodal by Aidan Lane on Thu Feb 24 2005.
 *    Modifications Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *    Original file:
 *
 *      project.cpp
 *      EverGreen
 *
 *      Created by Aidan Lane on Wed Jul 14 2004.
 *      Copyright (c) 2004 Aidan Lane. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "elementdoc.h"

#include <QDebug> // TODO: remove me!

#include <cstdlib> // for srandomdev() and random()

#include "abstractelement.h"
#include "elementdoccontroller.h"


#define ELEMENT_LIST_TAG  "elements"
#define ELEMENT_ID_TAG    "id"


/*!
 * Default constructor.
 */
ElementDoc::ElementDoc( QObject* parent )
  : AttributeDoc(parent)
{
  srandomdev(); // init random() number generator
}


ElementDoc::ElementDoc( const QString& name, QObject* parent )
  : AttributeDoc(name, parent)
{
  srandomdev(); // init random() number generator
}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * ElementDocController* from AbstractController*.
 *
 * See also AbstractModel::controller().
 */
ElementDocController* ElementDoc::elementDocController() const {
  return c_elementDocController;
}


/*!
 * Returns the \em ordered list of elements.
 *
 * \b Note: We use a list as the primary means of managing the elements, as to
 *          allow for ordering of the elements. This is usually required in
 *          diagrams, where the order in which elements are drawn is important.
 *
 * \b Note: This \em intentionally does not return a const ref, as to ensure
 *          thread safety.
 *          However, Qt's implicit sharing should help to maintain performance.
 */
QList<AbstractElement*> ElementDoc::elements() const {
  QReadLocker locker( &m_elementsLock );
  return m_elements;
}


/*!
 * Returns an \em un-ordered list of elements of a given type.
 *
 * This method is very useful for when a set of operations needs to be performed
 * on one exact type of element
 *
 * \b Warning: As stated above, unlike elements(), this returns an un-ordered list.
 *
 * \b Note: This uses pre-cached data, hence, this method is very cheap.
 *
 * \b Note: This \em intentionally does not return a const ref, as to ensure
 *          thread safety.
 *          However, Qt's implicit sharing should help to maintain performance.
 */
QList<AbstractElement*> ElementDoc::elements( int type ) const {
  QReadLocker locker( &m_elementsLock );
  return c_typeToElements.values( type );
}


/*!
 * Returns a hash of elements, where for each entry, the key is the element
 * instance id and the value is a pointer to the corresponding element object.
 *
 * Using a hash instead of a list helps us to quickly find an instance of a
 * given element, such as by the XML DOM methods for save/load and copy/paste,
 * etc.
 *
 * \b Note: This \em intentionally does not return a const ref, as to ensure
 *          thread safety.
 *          However, Qt's implicit sharing should help to maintain performance.
 */
QHash<quint32, AbstractElement*> ElementDoc::idToElement() const {
  QReadLocker locker( &m_elementsLock );
  return c_idToElement;
}


/*!
 * The model's event dispatcher.
 *
 * Asserts that the event is non-null.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 */
void ElementDoc::dispatchEvent( MEvent* event )
{	
  Q_ASSERT( event != 0 );

  // Let our superclass handle the event first
  AttributeDoc::dispatchEvent( event );

  if ( event->moduleId() != MvcCore::id() ) return; // prevent event type conflicts

  // Continue to handle the event unconditionally, as we may need to perform
  // extra operations when we receive an event that is also handled by Doc.
  switch ( event->type() )
    {
    case MElementEvent::ElementAddPtr:
      elementAddPtrEvent( static_cast<MElementEvent*>(event) ); break;
    case MElementEvent::ElementRemovePtr:
      elementRemovePtrEvent( static_cast<MElementEvent*>(event) ); break;
    case MElementEvent::ElementSetChangeOrder:
      elementSetChangeOrderEvent( static_cast<MElementSetChangeOrderEvent*>(event) ); break;
    default:
      break;
    }
}


/*!
 * Re-implemented, as to also have the pointer returned by elementDocController()
 * updated.
 *
 * Asserts that the \em event is non-null.
 */
void ElementDoc::changeControllerEvent( MChangeControllerEvent* event )
{
  Q_ASSERT( event != 0 );
  AttributeDoc::changeControllerEvent( event );
  c_elementDocController
    = ( (event->controller()==0)
	? 0
	: qobject_cast<ElementDocController*>(event->controller()->objectPtr()) );
}


/*!
 * Appends the given \em event->element() pointer to the \em front of the
 * document's list of elements().
 *
 * Asserts that both \em event and \em event->element() are non-null.
 */
void ElementDoc::elementAddPtrEvent( MElementEvent* event ) {
  Q_ASSERT( event != 0 );
  elementAddPtr( event->element() );
}

void ElementDoc::elementAddPtr( AbstractElement* element )
{
  Q_ASSERT( element != 0 );
  m_elementsLock.lockForWrite();
  m_elements.prepend( element ); // add to the FRONT
  c_typeToElements.insert( element->type(), element );
  c_idToElement.insert( element->instanceId(), element ); // will overwrite reserved
  m_elementsLock.unlock();
}


/*!
 * Removes the given \em event->element() pointer from the document's list of
 * elements().
 *
 * This will \em not delete the element (as in won't free its memory).
 * All memory de-allocation is to be handled by the document's controller
 * (e.g. ElementDocController).
 * This is so that the controller can remove the pointer from the document
 * (by calling this method) and THEN update the views (which may still refer to
 * element), followed by calling deleteLater() on the element.
 *
 * Asserts that both \em event and \em event->element() are non-null.
 */
void ElementDoc::elementRemovePtrEvent( MElementEvent* event ) {
  Q_ASSERT( event != 0 );
  elementRemovePtr( event->element() );
}


void ElementDoc::elementRemovePtr( AbstractElement* element )
{
  Q_ASSERT( element != 0 );

  m_elementsLock.lockForWrite();

  // Warning: We can't just go and do "c_typeToElements.remove( element->type() )",
  //          as that would remove *ALL* of the element entries for that type.
  QMultiHash<int, AbstractElement*>::iterator i
    = c_typeToElements.find( element->type() );
  while ( i != c_typeToElements.end() ) {
    if ( i.value() == element ) {
      i = c_typeToElements.erase( i );
      break;
    }
    ++i;
  }

  c_idToElement.remove( element->instanceId() );
  m_elements.removeAll( element );

  m_elementsLock.unlock();
}


/*!
 * Changes the (Z) order of the given \em event->elements() in respect to
 * elements() as specified by the given \em event->action().
 *
 * Supported actions are:
 * BringToFront, SendToBack, BringForward, and SendBackward.
 *
 * Assumes that elements() contains no duplicates.
 * Any duplicates will be removed and only the first occurrence will ever be used.
 *
 * Asserts that the \em event is non-null.
 */
void ElementDoc::elementSetChangeOrderEvent( MElementSetChangeOrderEvent* event )
{	
  Q_ASSERT( event != 0 );

  MElementSetChangeOrderEvent::Action action = event->action();

  if ( event->elements().isEmpty() ) return; // as the actions rely on it being not empty

  m_elementsLock.lockForWrite();

  /* We must retain the order of the elements, hence we first build up an ordered
   * list elements from the given set.
   */
  QList<AbstractElement*> orderedElements;
  {
    // Warning: The indexes are practically invalid as soon as something is
    //          inserted or removed, but we only care about the ORDER anyway.
    QMap<int, AbstractElement*> indexToElement; // use map as it's ordered
    foreach ( AbstractElement* e, event->elements() )
      indexToElement.insert( m_elements.indexOf(e), e );
    orderedElements = indexToElement.values();
  }

  if ( action == MElementSetChangeOrderEvent::BringToFront
       || action == MElementSetChangeOrderEvent::BringForward )
    { // Front: Work in REVERSE order to maintain relative order
      // Warning: This assumes that orderedElements is not empty.
      int frontIndex = ( action == MElementSetChangeOrderEvent::BringToFront
			 ? 0
			 : m_elements.indexOf(orderedElements.first()) - 1 );
      if ( frontIndex < 0 ) frontIndex = 0;
      QListIterator<AbstractElement*> it( orderedElements );
      it.toBack();
      while ( it.hasPrevious() )
	m_elements.move( m_elements.indexOf(it.previous()), frontIndex );
    }
  else if ( action == MElementSetChangeOrderEvent::SendToBack
	    || action == MElementSetChangeOrderEvent::SendBackward )
    { // Back: Work in ORIGINAL order to maintain relative order
      // Warning: This assumes that orderedElements is not empty.
      int backIndex = ( action == MElementSetChangeOrderEvent::SendToBack
			? m_elements.size() - 1
			: m_elements.indexOf(orderedElements.last()) + 1 );
      if ( backIndex >= m_elements.size() ) backIndex = m_elements.size() - 1;
      QListIterator<AbstractElement*> it( orderedElements );
      while ( it.hasNext() )
	m_elements.move( m_elements.indexOf(it.next()), backIndex );
    }

  m_elementsLock.unlock();
}


/*!
 * Creates an XML QDomElement that represents the ElementDoc.
 *
 * Use initFromDOMElement() to restore the ElementDoc state from the resulting
 * QDomElement.
 */
QDomElement ElementDoc::domElement( QDomDocument& doc ) const
{
  QDomElement root = AttributeDoc::domElement( doc );

  // Append an element for each AbstractElement, if there are any
  m_elementsLock.lockForRead();
  if ( ! m_elements.isEmpty() ) {
    QDomElement e = doc.createElement( ELEMENT_LIST_TAG );
    foreach ( AbstractElement* n, m_elements ) {
      Q_ASSERT( n != 0 );
      e.appendChild( n->domElement(doc) );
    }
    root.appendChild( e );
  }
  m_elementsLock.unlock();

  return root;
}


/*!
 * Restore the ElementDoc state from a QDomElement created by domElement().
 *
 * This will remove any existing elements first if \em modeFlags has its
 * InitStructure bit set.
 *
 * Please see DomAccessibleState::initFromDomElement() for more information.
 *
 * Note: Returns immediately if \em e.tagName() != key().
 */
void ElementDoc::initFromDomElement( const QDomElement& e, InitModeFlags modeFlags )
{
  bool toUIntOk = false;

  if ( e.tagName() != key() ) return;

  AttributeDoc::initFromDomElement( e, modeFlags );

  if ( modeFlags & InitStructure )
    { // Remove any existing elements
      m_elementsLock.lockForWrite(); // - lock for WRITE
      qDeleteAll( m_elements );
      m_elements.clear();
      c_typeToElements.clear();
      c_idToElement.clear();
      m_elementsLock.unlock();       // - unlock
    }

  // Get any elements
  QDomNode child = e.firstChild();
  while ( ! child.isNull() )
    {
      QDomElement childElement = child.toElement();   // try to convert the element to an element
      if ( ! childElement.isNull()                    // check if it's really an element
	   && childElement.tagName() == ELEMENT_LIST_TAG )
	{
	  // Traverse DOM node list
	  QDomNode cc = childElement.firstChild();
	  while ( ! cc.isNull() ) {
	    QDomElement ccElement = cc.toElement();
	    if ( ! ccElement.isNull() ) {

	      if ( modeFlags & InitStructure ) {//InitStructure
		// Create and init new AbstractElement
		AbstractElement* element
		  = createElement( ccElement.tagName().toAscii(), // we save into Ascii format
				   ccElement.attribute(ELEMENT_ID_TAG).toUInt(&toUIntOk) ); // this's a bit hacky!
		Q_ASSERT( toUIntOk != 0 );
		if ( element == 0 ) {
		  qDebug() << "Unsupported element type:" << ccElement.tagName(); // TODO: handle me properly!
		  cc = cc.nextSibling();
		  continue; // skip it!
		}
		element->initFromDomElement( ccElement, InitStructure );
		m_elementsLock.lockForWrite(); // - lock for WRITE
		m_elements.append( element );
		c_typeToElements.insert( element->type(), element );
		c_idToElement.insert( element->instanceId(), element ); 
		m_elementsLock.unlock();       // - unlock
	      }
	      
	      if ( modeFlags & InitStates ) { // InitStates - yes, this is meant to be "if"
		m_elementsLock.lockForRead(); // - lock for READ
		AbstractElement* element
		  = c_idToElement.value( ccElement.attribute(ELEMENT_ID_TAG).toUInt(&toUIntOk) ); // this's a bit hacky!
		Q_ASSERT( toUIntOk != 0 );
		if ( element != 0 )
		  element->initFromDomElement( ccElement, InitStates );
		m_elementsLock.unlock();       // - unlock
	      }
	    }
	    cc = cc.nextSibling();
	  }
	}
      child = child.nextSibling();
    }
}


/*!
 * Generates a unique element instance ID and reserves a place for it.
 *
 * The IDs will be practically random, which helps generation speed, as it
 * reduces clustering.
 *
 * \b Note:
 * Reservation is \em very important. For example, if there was no reservation
 * and this method were to be called twice in a row without actually inserting
 * a new element immediately after the first call, then this method could return
 * the same ID both times.
 * However, reservation is not persistent, which will prevent code that does not
 * follow through on a reservation from eventually rendering the document
 * incapable of holding new elements.
 *
 * \b Note:
 * Even though it is slower, gap-filling is used. This is because elements may be
 * randomly added and deleted, while the IDs remain persistent across save & load
 * (which allows elements to refer to one another). This allows the document to
 * (theoretically) contain up to 4,294,967,296 elements at any given time.
 */
quint32 ElementDoc::genElementInstanceId()
{
  m_elementsLock.lockForRead();
  quint32 initial = (quint32)random() % ((quint32)-1); // mod is ok with random()
  quint32 id = initial;                  // initially random
  while ( c_idToElement.contains(id) ) { // linear search -> if gap, we'll find it
    id = (id+3) % ((quint32)-1);
    Q_ASSERT( id != initial ); // TODO: don't just give up?
  }
  c_idToElement.insert( id, 0 );
  m_elementsLock.unlock();
  return id;
}
