/*  -*- c++ -*-  (for Emacs)
 *
 *  elementdoc.h
 *  Digest
 * 
 *  Imported into Digest by Aidan Lane on Thu Jun 9 2005.
 *  Modifications Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  Nodal file:

 *    project.h
 *    Nodal
 *
 *    Imported into Nodal by Aidan Lane on Thu Feb 24 2005.
 *    Modifications Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *    Original file:
 *
 *      project.h
 *      EverGreen
 *
 *      Created by Aidan Lane on Wed Jul 14 2004.
 *      Copyright (c) 2004 Aidan Lane. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef ELEMENTDOC_H
#define ELEMENTDOC_H


#include "attributedoc.h"

#include <QHash>
#include <QList>
#include <QMultiHash>
#include <QReadLocker>
#include <QReadWriteLock>
#include <QSet>

#include "elementdocevents.h"

class QByteArray;

class ElementDocController;


/*!
 * \brief The ElementDoc class extends Doc to provide functionality to
 *        allow abstract "elements" to be used in a document.
 *
 * \em Warning: The controller must be set in order for instances of this class
 * or any of its model class (such as AbstractElement) to send messages back
 * to it.
 */
class ElementDoc : public AttributeDoc {

  Q_OBJECT

public:
  ElementDoc( QObject* parent );
  ElementDoc( const QString& name, QObject* parent );

  ElementDocController* elementDocController() const;

  QList<AbstractElement*> elements() const;
  QList<AbstractElement*> elements( int type ) const;
  template<class T> inline QList<T*> elements() const;
  QHash<quint32, AbstractElement*> idToElement() const;

  virtual QDomElement domElement( QDomDocument& doc ) const;
  virtual void initFromDomElement( const QDomElement& e, InitModeFlags modeFlags );

  virtual quint32 genElementInstanceId();


protected:
  virtual void dispatchEvent( MEvent* );

  virtual void changeControllerEvent( MChangeControllerEvent* event );

  virtual void elementAddPtrEvent( MElementEvent* );
  virtual void elementRemovePtrEvent( MElementEvent* );
  virtual void elementSetChangeOrderEvent( MElementSetChangeOrderEvent* );

  void elementAddPtr( AbstractElement* element );
  void elementRemovePtr( AbstractElement* element );

  virtual AbstractElement* createElement( const QByteArray& key,
					  quint32 instanceId ) = 0;


private:
  /* Persistent (use domElement() and initFromDOMElement()):
   *
   * Note: We use a list as the primary means of managing the elements, as to
   *       allow for ordering of the elements. This is usually required in
   *       diagrams, where the order in which elements are drawn is important.
   */
  QList<AbstractElement*> m_elements;

  // Temporary variables:
  mutable QReadWriteLock  m_elementsLock;

  // Temporary caches:
  QPointer<ElementDocController>    c_elementDocController;
  QMultiHash<int, AbstractElement*> c_typeToElements; // type        -> instance
  QHash<quint32, AbstractElement*>  c_idToElement;    // instance id -> instance
};



/*!
 * Convenience method.
 *
 * Returns an \em unordered list of pointers to elements of a given type,
 * as dictated by the given class.
 */
template<class T> 
QList<T*> ElementDoc::elements() const
{
  // Note: the static cast should be fairly safe, given that we specify an enum'd type
  QReadLocker locker( &m_elementsLock );
  QList<T*> set;
  foreach ( AbstractElement* element, c_typeToElements.values(T::classType()) )
    set += static_cast<T*>( element );
  return set;
}


#endif  // ! ELEMENTDOC_H
