/*  -*- c++ -*-  (for Emacs)
 *
 *  mvccontainers.h
 *  Digest
 * 
 *  Created by Aidan Lane on Sun Aug 28 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef MVCCONTAINERS_H
#define MVCCONTAINERS_H


#include <QHash>
#include <QSet>
#include <QVariant>

class AbstractModel;
class AbstractView;
class AbstractController;

class AbstractElement;


typedef QSetIterator<AbstractElement*> AbstractElementSetIterator;

class AbstractElementSet : public QSet<AbstractElement*> {
public:
  AbstractElementSet() {}
  AbstractElementSet( const AbstractElementSet& other )
    : QSet<AbstractElement*>(other) {}
  AbstractElementSet( const QSet<AbstractElement*>& other )
    : QSet<AbstractElement*>(other) {}
  AbstractElementSet( AbstractElement* element ) {
    insert( element );
  }
};


class AbstractModelContainer {
public:
  AbstractModelContainer( AbstractModel* model )
    : m_model(model) {}
  AbstractModel* model() const { return m_model; }
private:
  AbstractModel* m_model;
};


class AbstractViewContainer {
public:
  AbstractViewContainer( AbstractView* view )
    : m_view(view) {}
  AbstractView* view() const { return m_view; }
private:
  AbstractView* m_view;
};


class AbstractControllerContainer {
public:
  AbstractControllerContainer( AbstractController* controller )
    : m_controller(controller) {}
  AbstractController* controller() const { return m_controller; }
private:
  AbstractController* m_controller;
};


/*!
 * \b Note:
 * QSet has been used instead of QList, as the element order does not matter,
 * it removes duplicates automatically and (obviously) provides set operations
 * that are very useful when adding and removing items from selections.
 * Also, it's lookup speed is also very good.
 */
class AbstractElementSetContainer {
public:
  AbstractElementSetContainer( const AbstractElementSet& elements )
    : m_elements(elements) {}
  const AbstractElementSet& elements() const { return m_elements; }
private:
  const AbstractElementSet m_elements;
};


class ElementOrderActionContainer {
public:
  enum Action {
    BringToFront, SendToBack, BringForward, SendBackward
  };
  ElementOrderActionContainer( Action action )
    : m_action(action) {}
  Action action() const { return m_action; }
private:
  Action m_action;
};


/*!
 * \b Note: The name "attributeTypes" has been used instead of "types", as this
 *          container is used with sub-classes of QEvent - which has a memeber
 *          called "type", thus we need to avoid confusion.
 */
class AttributeTypesContainer {
public:
  AttributeTypesContainer( const QSet<int>& attributeTypes )
    : m_attributeTypes(attributeTypes) {}
  AttributeTypesContainer( int type ) {
    m_attributeTypes += type;
  }
  const QSet<int>& attributeTypes() const { return m_attributeTypes; }
private:
  QSet<int> m_attributeTypes;
};


/*!
 * Contains a hash of attribute types (integers), which point to attribute
 * values (QVariants).
 */
class AttributesContainer {
public:
  AttributesContainer( const QHash<int, QVariant>& attributes )
    : m_attributes(attributes) {}
  AttributesContainer( int attributeType,
		       const QVariant& attributeValue ) {
    m_attributes.insert( attributeType, attributeValue );
  }
  const QHash<int, QVariant>& attributes() const {
    return m_attributes;
  }
private:
  QHash<int, QVariant> m_attributes;
};



#endif //  ! MVCCONTAINERS_H
