/*
 *  abstractview.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jun 9 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "abstractview.h"

#include <QCoreApplication>

#include "abstractmodel.h"
#include "abstractcontroller.h"


/*!
 * Constructs an abstract view.
 */
AbstractView::AbstractView( QObject* objectPtr )
  : MvcBase(objectPtr),
    m_controller(0),
    m_model(0)
{
}


/*!
 * Destroys the abstract view.
 *
 * If the controller() is non-null, this will post a CViewAttachEvent to it with
 * itself as the parameter.
 */
AbstractView::~AbstractView()
{
  if ( !m_controllerObject.isNull() )
    QCoreApplication::postEvent( m_controllerObject, new CViewDetachEvent(this) );
}


AbstractController* AbstractView::controller() const
{ // note: m_controllerObject is a guarded pointer, m_controller isn't
  return ( m_controllerObject.isNull()
	   ? 0
	   : m_controller );
}


AbstractModel* AbstractView::model() const
{ // note: m_modelObject is a guarded pointer, m_model isn't
  return ( m_modelObject.isNull()
	   ? 0
	   : m_model );
}


/*!
 * Convenience method.
 *
 * Equivalent to:
 * \code
 * QCoreApplication::postEvent( controller()->objectPtr(), event );
 * \endcode
 */
void AbstractView::postControllerEvent( CEvent* event ) {
  Q_ASSERT( ! m_controllerObject.isNull() );
  QCoreApplication::postEvent( m_controllerObject, event );
}


/*!
 * The view's event dispatcher.
 *
 * Asserts that the event is non-null.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 */
void AbstractView::dispatchEvent( VEvent* event )
{	
  Q_ASSERT( event != 0 );

  if ( event->moduleId() != MvcCore::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case VEvent::ChangeController:
      changeControllerEvent( static_cast<VChangeControllerEvent*>(event) ); break;
    case VEvent::ModelChanged:
      modelChangedEvent( static_cast<VModelChangedEvent*>(event) ); break;
    case VEvent::Reset:
      resetEvent( event ); break;
    default:
      break;
    }
}


/*!
 * Changes the controller that the view should post messages to.
 *
 * This will post a VModelChangedEvent to itself (the view), so that the model()
 * pointer is in sync with the controller's. If the given controller is null,
 * then a null model pointer will be posted. The VModelChangedEvent is posted,
 * rather than being sent, as to allow sub-classes to update themselves before
 * its modelChangedEvent() is called.
 * The controller is expected to post a VModelChangedEvent to every view
 * whenever its \em model changes.
 *
 * \b Note: 
 *
 * Typically, only a controller should post a VChangeControllerEvent to a view,
 * as a result of a CChangeViewEvent being posted to it.
 *
 * Asserts that the \em event is non-null.
 */
void AbstractView::changeControllerEvent( VChangeControllerEvent* event )
{
  Q_ASSERT( event != 0 );

  m_controller = event->controller();
  m_controllerObject = ( (m_controller==0)
			 ? 0
			 : m_controller->objectPtr() );

  /* Warning: We can't simply post a VChangeControllerEvent to the child view,
   *          as they need to be detached from their current controller
   *          (if there is one) and attached to the new controler
   *          (again, if there is one).
   */
  foreach ( AbstractView* child, m_childViews )
    {
      Q_ASSERT( child != 0 );
      AbstractController* currentController = child->controller();
      if ( m_controller != currentController ) { // as the following is VERY costly:
	if ( currentController != 0 ) currentController->detachView( child );
	if ( m_controller != 0 ) m_controller->attachView( child );
      }
    }

  QCoreApplication::postEvent( objectPtr(), new VModelChangedEvent
			       ( (m_controller==0)
				 ? 0
				 : m_controller->model(),
				 this ) );
}


/*!
 * Updates model() to reflect the one given, which should be the same as
 * controller::model().
 *
 * Derivative classes may re-implement this (but still \em must call their
 * super-class's version, which may be this, AbstractView) if they want to
 * cast the model's type to something more convenient.
 *
 * A VEvent(VEvent::Reset) will be posted back to the view, regardless of whether
 * or not the model is null (allowing the view to "clear" itself).
 * Hence, model synchronisation code should be located in the resetEvent() handler.
 * The reset event is posted, rather than being sent, as to allow sub-classes to
 * update themselves before its resetEvent() is called.
 *
 * Asserts that the \em event is non-null.
 */
void AbstractView::modelChangedEvent( VModelChangedEvent* event )
{
  Q_ASSERT( event != 0 );
  m_model = event->model();
  m_modelObject = ( (m_model==0)
		    ? 0
		    : m_model->objectPtr() );

  QCoreApplication::postEvent( objectPtr(),
			       new VEvent(VEvent::Reset, this) );
}


/*!
 * A view receives this event when the controller() needs to reset / synchronise
 * the view's knowledge (and possibly display) of the model().
 *
 * When the view receives a VModelChangedEvent, a reset event will automatically
 * be posted to the view by the modelChangedEvent() handler if the model is
 * non-null.
 *
 * This implementation does nothing.
 *
 * It should be re-implemented it to suit the specific view sub-classs.
 */
void AbstractView::resetEvent( VEvent* ) {}


/*!
 * Returns the set of child views that this parent view is managing.
 */
const QSet<AbstractView*> AbstractView::childViews() {
  return m_childViews;
}


/*!
 * Once added, whenever the controller of this parent view is changed, the child
 * view will be detached from its old controller (if it has one) and attached
 * to the new controller (again, if there is one).
 *
 * The view's controller is updated immediately.
 *
 * \b Note: Ownership is not reassigned.
 *
 * Asserts that the \em view is non-null.
 */
void AbstractView::addChildView( AbstractView* view )
{
  Q_ASSERT( view != 0 );
  m_childViews += view;

  AbstractController* currentController = view->controller();
  if ( m_controller != currentController ) { // as the following is VERY costly:
    if ( currentController != 0 )
      currentController->detachView( view );
    if ( m_controller != 0 )
      m_controller->attachView( view );
  }
}


/*!
 * \b Note: The child \em view is not deleted, it just won't be managed by this
 *          parent view anymore.
 */
void AbstractView::removeChildView( AbstractView* view ) {
  m_childViews -= view;
}


/*!
 * \b Note: The child views won't be deleted, they just won't be managed by this
 *          parent view anymore.
 */
void AbstractView::removeAllChildViews() {
  m_childViews.clear();
}
