/*  -*- c++ -*-  (for Emacs)
 *
 *  viewevents.h
 *  Digest
 * 
 *  Created by Aidan Lane on Mon Aug 29 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef VIEWEVENTS_H
#define VIEWEVENTS_H


#include "mvcevent.h"
#include "mvccontainers.h"


class QObject;

class AbstractController;
class AbstractModel;


/*!
 * Base class for View events in the Model-View-Controller event
 * system.
 *
 * Using and extending VEvent offers thread-safe event queuing (using postEvent)
 * and minial coupling.
 *
 * The sender() facility becomes useful when a view has multiple controllers
 * speaking to it. For example, an AbstractDiagramView is combined with an
 * AbstractSettingsView. In this case, the events would need to be distributed
 * to the correct dispatchEvent() method, based on the sender.
 * The sender is of type MvcBase*, as not only do controllers need to post events
 * to views, but views needs to be able to post events back to themselves.
 */
class VEvent : public MvcEvent {
public:
  enum Type {
    ChangeController  = MvcEvent::User,
    ModelChanged,
    Reset,
    User              = MvcEvent::User + 1000
  };

  VEvent( Type type, MvcBase* sender )
    : MvcEvent((MvcEvent::Type)type),
      m_sender(sender) {}

  /*! Returns the desination type for this event, which is MvcEvent::View. */
  inline static DestType classDestType() { return View; }
  /*! Returns classDestType(). */
  virtual DestType destType() const { return classDestType(); }

  MvcBase* sender() const { return m_sender; }

private:
  // Note: The sender can't be const, as the view may need to send message back.
  MvcBase* m_sender;
};


class VChangeControllerEvent : public VEvent, public AbstractControllerContainer {
public:
  VChangeControllerEvent( AbstractController* controller, MvcBase* sender )
    : VEvent(ChangeController, sender),
      AbstractControllerContainer(controller) {}
};


class VModelChangedEvent : public VEvent, public AbstractModelContainer {
public:
  VModelChangedEvent( AbstractModel* model, MvcBase* sender )
    : VEvent(ModelChanged, sender),
      AbstractModelContainer(model) {}
};


#endif // ! VIEWEVENTS_H
