/*
 *  diagramcontroller.cpp
 *  Digest
 * 
 *  Imported into Digest by Aidan Lane on Thu Jun 9 2005.
 *  Modifications Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  Nodal file:
 *
 *    projectcontroller.cpp
 *    Nodal
 *
 *    Imported into Nodal by Aidan Lane on Thu Feb 24 2005.
 *    Modifications Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *    Original file:
 *
 *      project.cpp
 *      EverGreen
 *
 *      Created by Aidan Lane on Wed Jul 14 2004.
 *      Copyright (c) 2004 Aidan Lane. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "diagramcontroller.h"

#include <QCoreApplication>
#include <QReadLocker>
#include <QWriteLocker>

// TODO: remove unused headers!
#include "diagram.h"
#include "abstractdiagramelement.h"
#include "abstractelementattribute.h"
#include "abstractdiagramview.h"
#include "diagramelementevents.h"
#include "diagramviewevents.h"


/*!
 * Constructs a digaram controller with the given \em parent.
 */
DiagramController::DiagramController( QObject* parent )
  : ElementDocController(parent)
{}


/*!
 * Constructs a digaram controller with the given \em parent and uses \em diagram
 * as its model.
 */
DiagramController::DiagramController( AbstractModel* diagram, QObject* parent )
  : ElementDocController(diagram, parent)
{}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * Diagram* from AbstractModel*.
 *
 * See also AbstractController::model().
 */
Diagram* DiagramController::diagram() const {
  return c_diagram;
}


/*!
 * This convenience slot (method) posts a CElementChangeOrderEvent for each in
 * elementSelection() using the BringToFront action.
 *
 * \b Note: This method is thread-safe.
 */
void DiagramController::elementSelectionBringToFront() {
  QReadLocker locker( &m_elementSelectionLock );
  QCoreApplication::postEvent( this, new CElementSetChangeOrderEvent
			       (m_elementSelection,
				CElementSetChangeOrderEvent::BringToFront, this) );
}

/*!
 * This convenience slot (method) posts a CElementChangeOrderEvent for each in
 * elementSelection() using the SendToBack action.
 *
 * \b Note: This method is thread-safe.
 */
void DiagramController::elementSelectionSendToBack() {
  QReadLocker locker( &m_elementSelectionLock );
  QCoreApplication::postEvent( this, new CElementSetChangeOrderEvent
			       (m_elementSelection,
				CElementSetChangeOrderEvent::SendToBack, this) );
}

/*!
 * This convenience slot (method) posts a CElementChangeOrderEvent for each in
 * elementSelection() using the BringForward action.
 *
 * \b Note: This method is thread-safe.
 */
void DiagramController::elementSelectionBringForward() {
  QReadLocker locker( &m_elementSelectionLock );
  QCoreApplication::postEvent( this, new CElementSetChangeOrderEvent
			       (m_elementSelection,
				CElementSetChangeOrderEvent::BringForward, this) );
}

/*!
 * This convenience slot (method) posts a CElementChangeOrderEvent for each in
 * elementSelection() using the SendBackward action.
 *
 * \b Note: This method is thread-safe.
 */
void DiagramController::elementSelectionSendBackward() {
  QReadLocker locker( &m_elementSelectionLock );
  QCoreApplication::postEvent( this, new CElementSetChangeOrderEvent
			       (m_elementSelection,
				CElementSetChangeOrderEvent::SendBackward, this) );
}


/*!
 * The element doc controller event dispatcher.
 *
 * Asserts that the event is non-null.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 */
void DiagramController::dispatchEvent( CEvent* event )
{
  Q_ASSERT( event != 0 );

  ElementDocController::dispatchEvent( event );

  if ( event->moduleId() != MvcDiagram::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case CDiagramEvent::ElementControlPointInsert:
      controlPointInsertEvent( static_cast<CControlPointInsertEvent*>(event) );
      break;
    case CDiagramEvent::ElementControlPointRemove:
      controlPointRemoveEvent( static_cast<CControlPointRemoveEvent*>(event) );
      break;
    case CDiagramEvent::ElementControlPointMove:
      controlPointMoveEvent( static_cast<CControlPointMoveEvent*>(event) );
      break;
    case CDiagramEvent::ChangeElementSelection:
      changeElementSelectionEvent( static_cast<CChangeElementSelectionEvent*>(event) );
      break;
    case CDiagramEvent::RemoveAllSelectedElements:
      removeAllSelectedElementsEvent( static_cast<CDiagramEvent*>(event) );
      break;
    default:
      break;
    }
}



/* Post a DocViewEvent to our views.
 * Any DocView object that has the same address as the value of "exclude"
 * will be exluded from the updates.
 * Therefore, set "exclude" to 0 if you don't want to exclude anything.
 * It asserts that the view pointer is non-null.
 * BTW: I know, this is very hacky :-)
 */
#define POST_VIEW_EVENT(EVENT,EXCLUDE)	{AbstractViewSetIterator i(views()); while(i.hasNext()) { AbstractView* v=i.next(); Q_ASSERT(v!=0); Q_ASSERT(v->objectPtr()!=0); if (v->objectPtr() != (EXCLUDE)) QCoreApplication::postEvent( v->objectPtr(), new EVENT ); }}

#define POST_MODIFIED_STATE_CHANGE_EVENTS(STATE) { if ( STATE != c_diagram->isModified()) {QCoreApplication::postEvent( c_diagram, new MChangeModifiedStateEvent(STATE) ); POST_VIEW_EVENT( VModifiedStateChangedEvent(STATE, this), 0 ); } }


/*!
 * Re-implemented, as to also have the pointer returned by diagram() updated.
 *
 * Asserts that the \em event is non-null.
 */
void DiagramController::changeModelEvent( CChangeModelEvent* event )
{
  Q_ASSERT( event != 0 );
  ElementDocController::changeModelEvent( event );
  c_diagram = ( (event->model()==0)
		? 0
		: qobject_cast<Diagram*>(event->model()->objectPtr()) );
}


/*!
 * If the given element is currently selected, this will update the
 * elementSelection() and post a CChangeElementSelectionEvent \em before the
 * element is removed, allowing views to remove any references to it.
 *
 * Asserts that the event and the element is refers to are both non-null.
 */
void DiagramController::elementRemoveEvent( CElementRemoveEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->element() != 0 );

  if ( m_elementSelection.contains(event->element()) )
    {
      // Post the change event to OURSELF, leaving the updates to the selection
      // and views to changeElementSelectionEvent() (maximising code reusage :-)
      AbstractElementSet removeSet;
      removeSet += event->element();
      QCoreApplication
	::postEvent( this,
		     new CChangeElementSelectionEvent(AbstractElementSet(),
						      removeSet) );
    }

  ElementDocController::elementRemoveEvent( event );
}


/*!
 * Asserts that the event and event->element() are both non-null.
 */
void DiagramController::controlPointInsertEvent( CControlPointInsertEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->element() != 0 );

  // TODO: assert that insertion is allowed for the given element type and add it to the documentation!

  Q_ASSERT( c_diagram != 0 );
  QCoreApplication::postEvent( event->element(),
			       new MEControlPointInsertEvent( event->index(),
							      event->pos()) );

  POST_VIEW_EVENT( VControlPointInsertedEvent(event->element(), event->index(),
					      event->pos(), this),
		   event->updateSender() ? 0 : event->sender() );
  POST_MODIFIED_STATE_CHANGE_EVENTS( true ); // -> Diagram has been modified
}


/*!
 * Asserts that the event and event->element() are both non-null.
 */
void DiagramController::controlPointRemoveEvent( CControlPointRemoveEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->element() != 0 );

  // TODO: assert that insertion is allowed for the given element type and add it to the documentation!

  Q_ASSERT( c_diagram != 0 );
  QCoreApplication::postEvent( event->element(),
			       new MEControlPointRemoveEvent(event->index()) );

  POST_VIEW_EVENT( VControlPointRemovedEvent(event->element(),
					     event->index(), this),
		   event->updateSender() ? 0 : event->sender() );
  POST_MODIFIED_STATE_CHANGE_EVENTS( true ); // -> Diagram has been modified
}


/*!
 * Asserts that the event and event->element() are both non-null.
 */
void DiagramController::controlPointMoveEvent( CControlPointMoveEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->element() != 0 );

  // TODO: assert that insertion is allowed for the given element type and add it to the documentation!

  Q_ASSERT( c_diagram != 0 );
  QCoreApplication::postEvent( event->element(),
			       new MEControlPointMoveEvent(event->index(),
							   event->pos()) );

  POST_VIEW_EVENT( VControlPointMovedEvent(event->element(), event->index(),
					   event->pos(), this),
		   event->updateSender() ? 0 : event->sender() );
  POST_MODIFIED_STATE_CHANGE_EVENTS( true ); // -> Diagram has been modified
}


/*!
 * Note: If an element appears in both CChangeElementSelectionEvent::addToSelection()
 *       and CChangeElementSelectionEvent::removeFromSelection(), then the element
 *       won't end up being in the elementSelection(), as it will first be added
 *       and then immediately removed. However, the two sets will be sent verbatim
 *       to the views.
 *
 * This will post a VElementSelectionChangedEvent after making the change to the
 * elementSelection().
 *
 * Asserts that the event is non-null.
 */
void DiagramController::changeElementSelectionEvent( CChangeElementSelectionEvent* event )
{
  Q_ASSERT( event != 0 );

  QWriteLocker locker( &m_elementSelectionLock );

  m_elementSelection += event->addToSelection();
  m_elementSelection -= event->removeFromSelection();

  POST_VIEW_EVENT( VElementSelectionChangedEvent(event->addToSelection(),
						 event->removeFromSelection(),
						 this),
		   event->updateSender() ? 0 : event->sender() );

  // Note: There is NO need to post MODIFIED events to the views,
  //       as the diagram model has NOT changed.
}


/*!
 * Convenience event for mass removal/deletion of elements that are selected.
 */
void DiagramController::removeAllSelectedElementsEvent( CDiagramEvent* )
{
  QWriteLocker locker( &m_elementSelectionLock );

  const AbstractElementSet removeSet = m_elementSelection; // make a copy before we destroy it

  QCoreApplication
    ::postEvent( this,
		 new CChangeElementSelectionEvent(AbstractElementSet(), // add nothing
						  removeSet) );   // remove current selection

  foreach ( AbstractElement* element, removeSet ) {
    Q_ASSERT( element != 0 );
    QCoreApplication::postEvent( this, new CElementRemoveEvent(element) );
  }
}
