/*  -*- c++ -*-  (for Emacs)
 *
 *  diagramcontrollerevents.h
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jun 14 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef DIAGRAMCONTROLLEREVENTS_H
#define DIAGRAMCONTROLLEREVENTS_H


#include "MvcCore/elementdoccontrollerevents.h"

#include "mvcdiagram.h"
#include "diagramcontainers.h"

class AbstractElement;


class CDiagramEvent : public CElementDocEvent {
public:
  enum Type {
    ElementControlPointInsert  = CElementDocEvent::User,
    ElementControlPointRemove,
    ElementControlPointMove,
    ChangeElementSelection,
    RemoveAllSelectedElements,
    User                       = CElementDocEvent::User + 1000
  };

  CDiagramEvent( Type type,
		 QObject* sender = 0, Flags flags = NoFlags )
    : CElementDocEvent((CElementDocEvent::Type)type, sender, flags) {}

  /*! Returns MvcDiagram::id(). */
  inline static MvcModuleId_t classModuleId() { return MvcDiagram::id(); }
  /*! Returns classModuleId(). */
  virtual MvcModuleId_t moduleId() const { return classModuleId(); }
};


class CDiagramElementEvent : public CDiagramEvent,
			     public AbstractElementContainer {
public:
  CDiagramElementEvent( Type type, AbstractElement* element,
			QObject* sender = 0,
			Flags flags = NoFlags )
    : CDiagramEvent(type, sender, flags),
      AbstractElementContainer(element) {}
};


/*!
 * Note: \em int has been used, not \em quint32, as it's what Qt has been
 *       standardised on for its indices.
 */
class CControlPointEvent : public CDiagramElementEvent, public IndexContainer {
public:
  CControlPointEvent( Type type, AbstractElement* element, int index,
		      QObject* sender = 0, Flags flags = NoFlags )
    : CDiagramElementEvent(type, element, sender, flags),
      IndexContainer(index) {}
};



class CControlPointInsertEvent : public CControlPointEvent, public ControlPointContainer {
public:
  CControlPointInsertEvent( AbstractElement* element,
			    int index, const QPointF& pos,
			    QObject* sender = 0,
			    Flags flags = NoFlags )
    : CControlPointEvent(ElementControlPointInsert, element, index,
			 sender, flags),
      ControlPointContainer(pos) {}
};


/*!
 * Convenience and consistency class.
 *
 * This is merely CControlPointEvent with the type set to
 * VControlPointEvent::Remove.
 * However, since there needs to be a class for Insert and Move,
 * having this class keeps the interface consistent.
 */
class CControlPointRemoveEvent : public CControlPointEvent {
public:
  CControlPointRemoveEvent( AbstractElement* element, int index,
			    QObject* sender = 0,
			    Flags flags = NoFlags )
    : CControlPointEvent(ElementControlPointRemove, element, index,
			 sender, flags) {}
};


class CControlPointMoveEvent : public CControlPointEvent, public ControlPointContainer {
public:
  CControlPointMoveEvent( AbstractElement* element,
			  int index, const QPointF& pos,
			  QObject* sender = 0,
			  Flags flags = NoFlags )
    : CControlPointEvent(ElementControlPointMove, element, index,
			 sender, flags),
      ControlPointContainer(pos) {}
};


/*!
 * Users often want to add or remove one item to or from a selection at a time,
 * so this structure is very efficient.
 *
 * To select all elements:
 * \code
 * CElementSelectionChangedEvent( myDiagram->elements(), AbstractElementSet() )
 * \endcode
 *
 * To de-select all selected elements:
 * \code
 * CElementSelectionChangedEvent( AbstractElementSet(), myDiagramController->elementSelection() )
 * \endcode
 *
 * Note: QSet has been used instead of QList, as the element order does not
 *       matter, it removes duplicates automatically and (obviously) provides
 *       set operations that are very useful when adding and removing items from
 *       selections (it's lookup speed is also very good).
 */
class CChangeElementSelectionEvent : public CDiagramEvent {

public:
  CChangeElementSelectionEvent( const AbstractElementSet& addToSelection,
				const AbstractElementSet& removeFromSelection,
				QObject* sender = 0, Flags flags = NoFlags )
    : CDiagramEvent(ChangeElementSelection, sender, flags),
      m_addToSelection(addToSelection),
      m_removeFromSelection(removeFromSelection) {}

  const AbstractElementSet& addToSelection() {
    return m_addToSelection;
  }
  const AbstractElementSet& removeFromSelection() {
    return m_removeFromSelection;
  }

private:
  const AbstractElementSet m_addToSelection;
  const AbstractElementSet m_removeFromSelection;
};


#endif //  ! DIAGRAMCONTROLLEREVENTS_H
