/*
 *  diagramelementpolygon.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jun 13 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "diagramelementpolygon.h"

#include <QByteArray>
#include <QMutex>
#include <QStringList>


// Separators
#define VALUE_SEP   ","
#define VEC_SEP     ";"


//TODO: remove the need for this hack - it should be in Qt
static bool registeredPolygonF = false;


DiagramElementPolygon::DiagramElementPolygon( AbstractElement* parent,
					      const QPolygonF& polygon )
  : AbstractDiagramElementAttribute(parent),
    m_polygon(polygon)
{
  if ( ! registeredPolygonF ) {
    registeredPolygonF = true;
    qRegisterMetaType<QPolygonF>("QPolygonF");
  }
}


/*!
 * Returns the polygon as a QString.
 *
 * The format of the string allows for extensibility, as each field is explicitly
 * stated as a "tag" (like XML element attributes).
 * However, as compactness is important for storage, the fields values are not
 * grouped by QPoint. Instead they are stored as a vector for each field type for
 * each point. This format makes it easy to fetch data from specific fields
 * without having to parse all of the data (e.g. parse only "x" and "y", not "z").
 *
 * Also of note: field values and field vectors are separated using single
 * characters, not brackets (they being ",", ";", respectively).
 * The decision to not use bracketing wasn't driven by compactness (although
 * that is a nice side-effect), but instead by the ease of parsing, given the
 * availability of a "split-on" function.
 *
 * Lastly, a textual string has been used instead of binary data, as this code
 * is designed to be cross platform, where the database could reside on a machine
 * of any architecture, communicating with a client of also any architecture.
 * It also makes the data (mildly) human-readable.
 *
 * Example:
 * \code
 *   "x=0,1,2,1,2;y=3,1,2,4,5;"
 * \endcode
 */
QString DiagramElementPolygon::worker_toString() const
{
  // Ensure that there is 1+ point -> required for the unconditional removal
  // of the last separator.
  if ( m_polygon.isEmpty() ) return QString();

  /* Note: We add add separators unconditionally in the loops, so that we don't
   *       have to continually evaluate a condition that will return the same
   *       result until the very last element. Thus, following the loops we
   *       perform an unconditional removal of the last separator, as to keep
   *       the final string as concise as possible.
   */
  QString xStr = "x=";
  QString yStr = "y=";
  foreach ( const QPointF& pt, m_polygon ) {
    xStr += QString::number(pt.x()) + VALUE_SEP;
    yStr += QString::number(pt.y()) + VALUE_SEP;
  }
  xStr.chop( 1 ); // remove the last separator
  yStr.chop( 1 ); // ditto. 
  return xStr + VEC_SEP + yStr;
}


/*!
 * Sets the polygon using the co-ordinates in the given \em data string.
 *
 * This method parses data that has been generated using toString().
 */
bool DiagramElementPolygon::worker_setData( const QString& data )
{
  m_polygon.clear();

  if ( data.isEmpty() ) return true; // not really a problem -> ret true

  foreach ( const QString& vecStr, data.split(VEC_SEP, QString::SkipEmptyParts) )
    {
      if ( vecStr.isEmpty() ) continue;

      QStringList fieldAndValues = vecStr.split( "=" );
      //TODO: as this is imput data, alert user if fieldAndValues.size != 2
      Q_ASSERT( fieldAndValues.size() == 2 ); // TODO: REMOVE DANGEROUS ASSERT
      const QString& field = fieldAndValues.at(0);
      QStringList values = fieldAndValues.at(1).split( VALUE_SEP,
						       QString::SkipEmptyParts );
      bool convOk = false;

      // TODO: optimise and clean this up
      // TODO: factorize code!
      // TODO DOC: a bit of checking, but don't need there's no "double-handling" of data
      if ( field == "x" )
	for ( int i=0; i < values.size(); ++i ) {
	  if ( i >= m_polygon.size() ) m_polygon += QPointF();
	  m_polygon[i].rx() = values.at(i).toFloat(&convOk);
	  //TODO: as this is input data, alert user if toFloat not OK - ask if -> 0 ok
	  Q_ASSERT( convOk ); // TODO: REMOVE DANGEROUS ASSERT
	}
      else if ( field == "y" )
	for ( int i=0; i < values.size(); ++i ) {
	  if ( i >= m_polygon.size() ) m_polygon += QPointF();
	  m_polygon[i].ry() = values.at(i).toFloat(&convOk);
	  //TODO: as this is input data, alert user if toFloat not OK - ask if -> 0 ok
	  Q_ASSERT( convOk ); // TODO: REMOVE DANGEROUS ASSERT
	} 
    }

  return true;
}


QVariant DiagramElementPolygon::worker_toVariant() const {
  return QVariant::fromValue( m_polygon );
}


bool DiagramElementPolygon::worker_setData( const QVariant& data ) {
  m_polygon = data.value<QPolygonF>();
  return true;
}
