/*
 *  abstractdiagramelement.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jun 13 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "abstractdiagramelement.h"

#include <QCoreApplication>
#include <QDebug> // TODO: remove me!

#include "abstractelementattribute.h"
#include "diagramelementattributefactory.h"


/*!
 * Constructs an element for Diagram \em doc with a given \em instanceId.
 */
AbstractDiagramElement::AbstractDiagramElement( qint32 instanceId, ElementDoc* doc,
						const QList<QPointF>& controlPoints )
  : AbstractElement(instanceId, doc),
    m_controlPoints(controlPoints)
{
}


/*!
 * Convenience constructor. It behaves essentially like the above function.
 *
 * The element's \em instanceId is automatically generated using
 * ElementDoc::genElementInstanceId.
 */
AbstractDiagramElement::AbstractDiagramElement( ElementDoc* doc,
						const QList<QPointF>& controlPoints )
  : AbstractElement(doc),
    m_controlPoints(controlPoints)
{
}


/*!
 * This will create and add an instance of each attribute type that is listed by
 * requiredAttributes().
 *
 * \b Warning: This can't be called by the AbstractDiagramElement's constructors,
 * as it relies on the sub-class' v-table to have been created, as it calls
 * requiredAttributes(), which is (pure-)virtual.
 * Hence, it must be called by the sub-class' constructor().
 */
void AbstractDiagramElement::addRequiredAttributes()
{
  foreach ( int type, requiredAttributes() ) {
    MEAttributeAddPtrEvent event( createAttribute(type) );
    dispatchEvent( &event ); // DON'T go through  the controller - have them ready NOW!
  }
}


void AbstractDiagramElement::dispatchEvent( MEvent* event )
{
  Q_ASSERT( event != 0 );

  AbstractElement::dispatchEvent( event );

  if ( event->moduleId() != MvcDiagram::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case MEDiagramElementEvent::ControlPointInsert:
      controlPointInsertEvent( static_cast<MEControlPointInsertEvent*>(event) );
      break;
    case MEDiagramElementEvent::ControlPointRemove:
      controlPointRemoveEvent( static_cast<MEControlPointRemoveEvent*>(event) );
      break;
    case MEDiagramElementEvent::ControlPointMove:
      controlPointMoveEvent( static_cast<MEControlPointMoveEvent*>(event) );
      break;
    default:
      break;
    }
}


/*!
 * This implementation simply inserts the control point into the controlPoints()
 * list.
 */
void AbstractDiagramElement::controlPointInsertEvent( MEControlPointInsertEvent* event )
{
  Q_ASSERT( event != 0 );
  m_controlPoints.insert( event->index(), event->pos() );
}


/*!
 * This implementation simply remove the control point from the controlPoints()
 * list.
 */
void AbstractDiagramElement::controlPointRemoveEvent( MEControlPointRemoveEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->index() >= 0 );
  Q_ASSERT( event->index() < m_controlPoints.size() );
  m_controlPoints.removeAt( event->index() );
}


/*!
 * This implementation simply updates the controlPoints() item at event->index()
 * to reflect event->pos().
 *
 * Note: event->index() must be a valid index position in the conrtrol point list
 *       (i.e., 0 <= event->index() < controlPoints().size()).
 */
void AbstractDiagramElement::controlPointMoveEvent( MEControlPointMoveEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->index() >= 0 );
  Q_ASSERT( event->index() < m_controlPoints.size() );
  m_controlPoints[event->index()] = event->pos();
}


/*!
 * Note: The \em type is an int, as to aid extensibility.
 */
AbstractElementAttribute*
AbstractDiagramElement::createAttribute( int type ) {
  return DiagramElementAttributeFactory
    ::create( (MvcDiagram::ElementAttributeType)type, this );
}


AbstractElementAttribute*
AbstractDiagramElement::createAttribute( const QByteArray& key ) {
  return DiagramElementAttributeFactory::create( key, this );
}

int AbstractDiagramElement::attributeKeyToType( const QByteArray& key ) {
  return DiagramElementAttributeFactory::type( key );
}
