/*
 *  abstractdiagramview.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jun 15 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "abstractdiagramview.h"

#include <QCoreApplication>

#include "diagram.h"
#include "diagramcontroller.h"


AbstractDiagramView::AbstractDiagramView( QObject* objectPtr )
  : AbstractElementDocView(objectPtr)
{
}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * DiagramController* from AbstractController*.
 *
 * See also AbstractView::controller().
 */
DiagramController* AbstractDiagramView::diagramController() const {
  return c_diagramController;
}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * Diagram* from AbstractModel*.
 *
 * See also AbstractView::model().
 */
Diagram* AbstractDiagramView::diagram() const {
  return c_diagram;
}


/*!
 * Convenience method.
 *
 * Equivalent to:
 * \code
 * #include <QCoreApplication>
 * #include "MvcDiagram/diagramcontroller.h"
 * QCoreApplication::postEvent( diagramController(), event );
 * \endcode
 */
void AbstractDiagramView::postDiagramEvent( CEvent* event ) {
  Q_ASSERT( ! c_diagramController.isNull() );
  QCoreApplication::postEvent( c_diagramController, event );
}


void AbstractDiagramView::dispatchEvent( VEvent* event )
{
  Q_ASSERT( event != 0 );

  AbstractElementDocView::dispatchEvent( event );

  if ( event->moduleId() != MvcDiagram::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case VDiagramEvent::ElementControlPointInserted:
      controlPointInsertedEvent( static_cast<VControlPointInsertedEvent*>(event) );
      break;
    case VDiagramEvent::ElementControlPointRemoved:
      controlPointRemovedEvent( static_cast<VControlPointRemovedEvent*>(event) );
      break;
    case VDiagramEvent::ElementControlPointMoved:
      controlPointMovedEvent( static_cast<VControlPointMovedEvent*>(event) );
      break;
    case VDiagramEvent::ElementSelectionChanged:
      elementSelectionChangedEvent( static_cast<VElementSelectionChangedEvent*>(event) );
      break;
    default:
      break;
    }
}


/*!
 * Re-implemented, as to also have the pointer returned by diagramController()
 * updated.
 *
 * Asserts that the \em event is non-null.
 */
void AbstractDiagramView::changeControllerEvent( VChangeControllerEvent* event )
{
  Q_ASSERT( event != 0 );
  AbstractElementDocView::changeControllerEvent( event );
  c_diagramController
    = ( (event->controller()==0)
	? 0
	: qobject_cast<DiagramController*>(event->controller()->objectPtr()) );

  changeDiagramControllerEvent( event );
}


/*!
 * Re-implemented, as to also have the pointer returned by diagram() updated.
 *
 * Asserts that the \em event is non-null.
 */
void AbstractDiagramView::modelChangedEvent( VModelChangedEvent* event )
{
  Q_ASSERT( event != 0 );
  AbstractElementDocView::modelChangedEvent( event );
  c_diagram = ( (event->model()==0)
	    ? 0
	    : qobject_cast<Diagram*>(event->model()->objectPtr()) );

  diagramModelChangedEvent( event );
}


void AbstractDiagramView::resetEvent( VEvent* event )
{
  diagramResetEvent( event );
}
