/*  -*- c++ -*-  (for Emacs)
 *
 *  diagramviewevents.h
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jun 15 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef DIAGRAMVIEWEVENTS_H
#define DIAGRAMVIEWEVENTS_H


#include "MvcCore/elementdocviewevents.h"

#include "mvcdiagram.h"
#include "diagramcontainers.h"


class VDiagramEvent : public VElementDocEvent {
public:
  enum Type {
    ElementControlPointInserted  = VElementDocEvent::User,
    ElementControlPointRemoved,
    ElementControlPointMoved,
    ElementSelectionChanged,
    User                         = VElementDocEvent::User + 1000
  };

  VDiagramEvent( Type type, MvcBase* sender )
    : VElementDocEvent((VElementDocEvent::Type)type, sender) {}

  /*! Returns MvcDiagram::id(). */
  inline static MvcModuleId_t classModuleId() { return MvcDiagram::id(); }
  /*! Returns classModuleId(). */
  virtual MvcModuleId_t moduleId() const { return classModuleId(); }
};


class VDiagramElementEvent : public VDiagramEvent,
			     public AbstractElementContainer {
public:
  VDiagramElementEvent( Type type, AbstractElement* element,
			MvcBase* sender )
    : VDiagramEvent(type, sender),
      AbstractElementContainer(element) {}
};


/*!
 * Note: \em int has been used, not \em quint32, as it's what Qt has been
 *       standardised on for its indices.
 */
class VControlPointEvent : public VDiagramElementEvent, public IndexContainer {
public:
  VControlPointEvent( Type type, AbstractElement* element,
		      int index, MvcBase* sender )
    : VDiagramElementEvent(type, element, sender),
      IndexContainer(index) {}
};



class VControlPointInsertedEvent : public VControlPointEvent,
				   public ControlPointContainer {
public:
  VControlPointInsertedEvent( AbstractElement* element,
			      int index, const QPointF& pos, MvcBase* sender )
    : VControlPointEvent(ElementControlPointInserted, element, index, sender),
      ControlPointContainer(pos) {}
};


/*!
 * Convenience and consistency class.
 *
 * This is merely VControlPointEvent with the type set to
 * VControlPointEvent::Removed.
 * However, since there needs to be a class for Inserted and Moved,
 * having this class keeps the interface consistent.
 */
class VControlPointRemovedEvent : public VControlPointEvent {
public:
  VControlPointRemovedEvent( AbstractElement* element,
			     int index, MvcBase* sender )
    : VControlPointEvent(ElementControlPointRemoved, element, index, sender) {}
};


class VControlPointMovedEvent : public VControlPointEvent,
				public ControlPointContainer {
public:
  VControlPointMovedEvent( AbstractElement* element,
			   int index, const QPointF& pos, MvcBase* sender )
    : VControlPointEvent(ElementControlPointMoved, element, index, sender),
      ControlPointContainer(pos) {}
};


/*!
 * The data that event provides dramatically simplifies the selection handler
 * code for views (vs. just telling it the new selection) and isn't much of a
 * burden on the DiagramController either.
 *
 * Users often want to add or remove one item to or from a selection at a time,
 * so this structure is very efficient.
 *
 * The entire selection can still retrieved from
 * DiagramController::elementSelection().
 *
 * Note: QSet has been used instead of QList, as the element order does not
 *       matter, it removes duplicates automatically and (obviously) provides
 *       set operations that are very useful when adding and removing items from
 *       selections (it's lookup speed is also very good).
 */
class VElementSelectionChangedEvent : public VDiagramEvent {
public:
  VElementSelectionChangedEvent( const AbstractElementSet& addedToSelection,
				 const AbstractElementSet& removedFromSelection,
				 MvcBase* sender )
    : VDiagramEvent(ElementSelectionChanged, sender),
      m_addedToSelection(addedToSelection),
      m_removedFromSelection(removedFromSelection) {}

  const AbstractElementSet& addedToSelection() {
    return m_addedToSelection;
  }
  const AbstractElementSet& removedFromSelection() {
    return m_removedFromSelection;
  }

private:
  const AbstractElementSet m_addedToSelection;
  const AbstractElementSet m_removedFromSelection;
};


#endif //  ! DIAGRAMVIEWEVENTS_H
