/*  -*- c++ -*-  (for Emacs)
 *
 *  digestdbcontroller.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Fri Aug 12 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "digestdbcontroller.h"

#include <QCoreApplication>

#include "digestdbmodel.h"
#include "abstractdigestdbview.h"


/*!
 * Constructs a Digest database controller with the given \em parent.
 */
DigestDbController::DigestDbController( QObject* parent )
  : QObject(parent),
    AbstractController(this)
{
}


/*!
 * Constructs a Digest database controller with the given \em parent and
 * \em model.
 */
DigestDbController::DigestDbController( DigestDbModel* model, QObject* parent )
  : QObject(parent),
    AbstractController(this)
{
  setModel( model );
}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * DigestDbModel* from AbstractModel*.
 *
 * See also AbstractController::model().
 */
DigestDbModel* DigestDbController::digestDbModel() const {
  return c_digestDbModel;
}


/*!
 * The controller's event dispatcher.
 *
 * Asserts that the event is non-null.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 */
void DigestDbController::dispatchEvent( CEvent* event )
{
  Q_ASSERT( event );

  AbstractController::dispatchEvent( event );

  if ( event->moduleId() != MvcDigestDb::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case CDigestDbEvent::ClassAdd:
      classAddEvent( static_cast<CClassAddEvent*>(event) ); break;
    case CDigestDbEvent::ClassUpdate:
      classUpdateEvent( static_cast<CClassUpdateEvent*>(event) ); break;
    case CDigestDbEvent::ClassesRemove:
      classesRemoveEvent( static_cast<CClassesRemoveEvent*>(event) ); break;

    case CDigestDbEvent::CollectionAdd:
      collectionAddEvent( static_cast<CCollectionAddEvent*>(event) ); break;
    case CDigestDbEvent::CollectionUpdate:
      collectionUpdateEvent( static_cast<CCollectionUpdateEvent*>(event) ); break;
    case CDigestDbEvent::CollectionsRemove:
      collectionsRemoveEvent( static_cast<CCollectionsRemoveEvent*>(event) ); break;

    case CDigestDbEvent::ExperimentAdd:
      experimentAddEvent( static_cast<CExperimentAddEvent*>(event) ); break;
    case CDigestDbEvent::ExperimentUpdate:
      experimentUpdateEvent( static_cast<CExperimentUpdateEvent*>(event) ); break;
    case CDigestDbEvent::ExperimentsRemove:
      experimentsRemoveEvent( static_cast<CExperimentsRemoveEvent*>(event) ); break;

    case CDigestDbEvent::GestureAdd:
      gestureAddEvent( static_cast<CGestureAddEvent*>(event) ); break;
    case CDigestDbEvent::GestureUpdate:
      gestureUpdateEvent( static_cast<CGestureUpdateEvent*>(event) ); break;
    case CDigestDbEvent::GesturesChangeClasses:
      gesturesChangeClassesEvent( static_cast<CGesturesChangeClassesEvent*>(event) ); break;
    case CDigestDbEvent::GesturesChangeCollections:
      gesturesChangeCollectionsEvent( static_cast<CGesturesChangeCollectionsEvent*>(event) ); break;
    case CDigestDbEvent::GesturesRemove:
      gesturesRemoveEvent( static_cast<CGesturesRemoveEvent*>(event) ); break;

    case CDigestDbEvent::TrainedRecogAdd:
      trainedRecogAddEvent( static_cast<CTrainedRecogAddEvent*>(event) ); break;
    case CDigestDbEvent::TrainedRecogUpdate:
      trainedRecogUpdateEvent( static_cast<CTrainedRecogUpdateEvent*>(event) ); break;
    case CDigestDbEvent::TrainedRecogsRemove:
      trainedRecogsRemoveEvent( static_cast<CTrainedRecogsRemoveEvent*>(event) ); break;

    default:
      break;
    }
}



/* Post an event to our views.
 * Any view object that has the same address as the value of "exclude"
 * will be exluded from the updates.
 * Therefore, set "exclude" to 0 if you don't want to exclude anything.
 * It asserts that the view pointer is non-null.
 * BTW: I know, this is very hacky :-)
 */
#define POST_VIEW_EVENT(EVENT,EXCLUDE)	{AbstractViewSetIterator i(views()); while(i.hasNext()) { AbstractView* v=i.next(); Q_ASSERT(v!=0); Q_ASSERT(v->objectPtr()!=0); if (v->objectPtr() != (EXCLUDE)) QCoreApplication::postEvent( v->objectPtr(), new EVENT ); }}



/*!
 * Re-implemented, as to also have the pointer returned by digestDbModel()
 * updated.
 *
 * Asserts that the \em event is non-null.
 */
void DigestDbController::changeModelEvent( CChangeModelEvent* event )
{
  Q_ASSERT( event );
  AbstractController::changeModelEvent( event );
  c_digestDbModel
    = ( (event->model()==0)
	? 0
	: qobject_cast<DigestDbModel*>(event->model()->objectPtr()) );
}


/*! Asserts that the event is non-null. */
void DigestDbController::classAddEvent( CClassAddEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MClassAddEvent(event->record()) );
  // Note: We're posting the wrong IDs, but AbstractDigestDbView will fix it
  IdSet idSet;
  idSet += event->record().id;
  POST_VIEW_EVENT( VClassesEvent(VDigestDbEvent::ClassesAdded,
				 idSet, this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}

/*! Asserts that the event is non-null. */
void DigestDbController::classUpdateEvent( CClassUpdateEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MClassUpdateEvent(event->record()) );
  IdSet idSet;
  idSet += event->record().id;
  POST_VIEW_EVENT( VClassesEvent(VDigestDbEvent::ClassesUpdated,
				 idSet, this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}

/*! Asserts that the event is non-null. */
void DigestDbController::classesRemoveEvent( CClassesRemoveEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  POST_VIEW_EVENT( VClassesEvent(VDigestDbEvent::ClassesAboutToBeRemoved,
				 event->idSet(), this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MClassesRemoveEvent(event->idSet()) );
  POST_VIEW_EVENT( VClassesEvent(VDigestDbEvent::ClassesRemoved,
				 event->idSet(), this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}



/*! Asserts that the event is non-null. */
void DigestDbController::collectionAddEvent( CCollectionAddEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MCollectionAddEvent(event->record()) );
  // Note: We're posting the wrong ID, but AbstractDigestDbView will fix it
  IdSet idSet;
  idSet += event->record().id;
  POST_VIEW_EVENT( VCollectionsEvent(VDigestDbEvent::CollectionsAdded,
				    idSet, this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}

/*! Asserts that the event is non-null. */
void DigestDbController::collectionUpdateEvent( CCollectionUpdateEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MCollectionUpdateEvent(event->record()) );
  IdSet idSet;
  idSet += event->record().id;
  POST_VIEW_EVENT( VCollectionsEvent(VDigestDbEvent::CollectionsUpdated,
				     idSet, this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}

/*! Asserts that the event is non-null. */
void DigestDbController::collectionsRemoveEvent( CCollectionsRemoveEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  POST_VIEW_EVENT( VCollectionsEvent(VDigestDbEvent::CollectionsAboutToBeRemoved,
				     event->idSet(), this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MCollectionsRemoveEvent(event->idSet()) );
  POST_VIEW_EVENT( VCollectionsEvent(VDigestDbEvent::CollectionsRemoved,
				     event->idSet(), this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}



/*! Asserts that the event is non-null. */
void DigestDbController::experimentAddEvent( CExperimentAddEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MExperimentAddEvent(event->record()) );
  // Note: We're posting the wrong ID, but AbstractDigestDbView will fix it
  IdSet idSet;
  idSet += event->record().id;
  POST_VIEW_EVENT( VExperimentsEvent(VDigestDbEvent::ExperimentsAdded,
				     idSet, this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}

/*! Asserts that the event is non-null. */
void DigestDbController::experimentUpdateEvent( CExperimentUpdateEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MExperimentUpdateEvent(event->record()) );
  IdSet idSet;
  idSet += event->record().id;
  POST_VIEW_EVENT( VExperimentsEvent(VDigestDbEvent::ExperimentsUpdated,
				     idSet, this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}

/*! Asserts that the event is non-null. */
void DigestDbController::experimentsRemoveEvent( CExperimentsRemoveEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  POST_VIEW_EVENT( VExperimentsEvent(VDigestDbEvent::ExperimentsAboutToBeRemoved,
				     event->idSet(), this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MExperimentsRemoveEvent(event->idSet()) );
  POST_VIEW_EVENT( VExperimentsEvent(VDigestDbEvent::ExperimentsRemoved,
				     event->idSet(), this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}


/*! Asserts that the event is non-null. */
void DigestDbController::gestureAddEvent( CGestureAddEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MGestureAddEvent(event->record()) );
  // Note: We're posting the wrong ID, but AbstractDigestDbView will fix it
  IdSet idSet;
  idSet += event->record().id;
  POST_VIEW_EVENT( VGesturesEvent(VDigestDbEvent::GesturesAdded,
				  idSet, this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}

/*! Asserts that the event is non-null. */
void DigestDbController::gestureUpdateEvent( CGestureUpdateEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MGestureUpdateEvent(event->record()) );
  IdSet idSet;
  idSet += event->record().id;
  POST_VIEW_EVENT( VGesturesEvent(VDigestDbEvent::GesturesUpdated,
				  idSet, this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}

/*!
 * \b Note: If an ID appears in both CGesturesChangeClassesEvent::addSet() and
 *          CGesturesChangeClassesEvent::removeSet(), then the the ID will be in
 *          the final classes set, as it will first be removed and then
 *          immediately added.
 *          However, the two sets will be sent verbatim to the views.
 *
 * Asserts that the event is non-null.
 */
void DigestDbController::gesturesChangeClassesEvent( CGesturesChangeClassesEvent* ev )
{
  Q_ASSERT( ev );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MGesturesChangeClassesEvent
			       (ev->idSet(), ev->addSet(), ev->removeSet()) );
  POST_VIEW_EVENT( VGesturesChangedClassesEvent
		   (ev->idSet(), ev->addSet(), ev->removeSet(),
		    this, ev->sender()),
		   ev->updateSender() ? 0 : ev->sender() );
}

/*!
 * \b Note: If an ID appears in both CGesturesChangeCollectionsEvent::addSet()
 *          and CGesturesChangeCollectionsEvent::removeSet(), then the the ID
 *          will be in the final collections set, as it will first be removed and
 *          then immediately added.
 *          However, the two sets will be sent verbatim to the views.
 *
 * Asserts that the event is non-null.
 */
void DigestDbController::gesturesChangeCollectionsEvent( CGesturesChangeCollectionsEvent* ev )
{
  Q_ASSERT( ev );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MGesturesChangeCollectionsEvent
			       (ev->idSet(), ev->addSet(), ev->removeSet()) );
  POST_VIEW_EVENT( VGesturesChangedCollectionsEvent
		   (ev->idSet(), ev->addSet(), ev->removeSet(),
		    this, ev->sender()),
		   ev->updateSender() ? 0 : ev->sender() );
}


/*! Asserts that the event is non-null. */
void DigestDbController::gesturesRemoveEvent( CGesturesRemoveEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  POST_VIEW_EVENT( VGesturesEvent(VDigestDbEvent::GesturesAboutToBeRemoved,
				  event->idSet(), this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MGesturesRemoveEvent(event->idSet()) );
  POST_VIEW_EVENT( VGesturesEvent(VDigestDbEvent::GesturesRemoved,
				  event->idSet(), this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}


/*! Asserts that the event is non-null. */
void DigestDbController::trainedRecogAddEvent( CTrainedRecogAddEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MTrainedRecogAddEvent(event->record()) );
  // Note: We're posting the wrong ID, but AbstractDigestDbView will fix it
  IdSet idSet;
  idSet += event->record().id;
  POST_VIEW_EVENT( VTrainedRecogsEvent(VDigestDbEvent::TrainedRecogsAdded,
				       idSet, this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}

/*! Asserts that the event is non-null. */
void DigestDbController::trainedRecogUpdateEvent( CTrainedRecogUpdateEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MTrainedRecogUpdateEvent(event->record()) );
  IdSet idSet;
  idSet += event->record().id;
  POST_VIEW_EVENT( VTrainedRecogsEvent(VDigestDbEvent::TrainedRecogsUpdated,
				       idSet, this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}

/*! Asserts that the event is non-null. */
void DigestDbController::trainedRecogsRemoveEvent( CTrainedRecogsRemoveEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( c_digestDbModel );
  POST_VIEW_EVENT( VTrainedRecogsEvent(VDigestDbEvent::TrainedRecogsAboutToBeRemoved,
				       event->idSet(), this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
  QCoreApplication::postEvent( c_digestDbModel,
			       new MTrainedRecogsRemoveEvent(event->idSet()) );
  POST_VIEW_EVENT( VTrainedRecogsEvent(VDigestDbEvent::TrainedRecogsRemoved,
				       event->idSet(), this, event->sender()),
		   event->updateSender() ? 0 : event->sender() );
}
