/*  -*- c++ -*-  (for Emacs)
 *
 *  digestdbmodel.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Fri Aug 12 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "digestdbmodel.h"

#include <QByteArray>
#include <QDebug> // TODO: remove me!
#include <QSqlError>
#include <QStringList>
#include <QVariant>

#ifdef DIGEST_DEBUG
#include <QSqlDriver> // used by a few assertions
#endif // DIGEST_DEBUG

#include "digestdbcontroller.h"


// Note: We're using an easy to search format (e.g. 20050801) (not ISO though)
#define DATE_STORE_FORMAT_STR  "yyyyMMdd"

// Warning: We obviously don't want to translate these.
static const QString s_classTable             = "Class";
static const QString s_classMapTable          = "ClassMap";
static const QString s_collectionTable        = "Collection";
static const QString s_collectionMapTable     = "CollectionMap";
static const QString s_gestureTable           = "Gesture";
static const QString s_experimentTable        = "Experiment";
static const QString s_experimentResultTable  = "ExperimentResult";
static const QString s_trainedRecogTable      = "TrainedRecogniser";


DigestDbModel::DigestDbModel( const QSqlDatabase& database, QObject* parent )
  : QObject(parent),
    AbstractModel(this),
    m_db(database)
{
  prepareQueries( m_db );
}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * DigestDbController* from AbstractController*.
 *
 * See also AbstractModel::controller().
 */
DigestDbController* DigestDbModel::digestDbController() const {
  return c_digestDbController;
}


void DigestDbModel::prepareQueries( const QSqlDatabase& db )
{
  // Make all of the queries to use to use the THIS database
  m_classFetchAllQuery = m_classFetchLabelQuery
    = m_classMapInsertQuery = m_classMapDeleteQuery
    = m_collectFetchAllQuery = m_collectFetchLabelQuery
    = m_collectMapInsertQuery = m_collectMapDeleteQuery
    = m_experimentInsertQuery = m_experimentUpdateQuery = m_experimentDeleteQuery
    = m_experimentFetchAllBasicQuery = m_experimentFetchLabelQuery = m_experimentFetchQuery
    = m_gestInsertQuery =  m_gestUpdateQuery = m_gestDeleteQuery = m_gestFetchQuery
    = m_recogInsertQuery = m_recogUpdateQuery = m_recogDeleteQuery
    = m_recogFetchAllBasicQuery = m_recogFetchLabelQuery = m_recogFetchQuery
    = QSqlQuery( db );

  // NOTE: Positional (rather than named) binding has been used,
  //       as that's what SQLite uses natively (so I believe :-).

  Q_ASSERT( m_gestInsertQuery.driver() );
  Q_ASSERT( m_gestInsertQuery.driver()->hasFeature(QSqlDriver::PreparedQueries) );
  Q_ASSERT( m_gestInsertQuery.driver()->hasFeature(QSqlDriver::PositionalPlaceholders) );


  /*
   * Class Table
   */

  m_classFetchAllQuery.prepare( "SELECT id,label FROM Class" ); // safer not to use "*"
  m_classFetchLabelQuery.prepare( "SELECT label FROM Class WHERE id=?" );


  /*
   * ClassMap Table
   */

  m_classMapInsertQuery.prepare( "INSERT INTO ClassMap "
				 "VALUES (?, ?)" );

  m_classMapDeleteQuery.prepare( "DELETE FROM ClassMap "
				 "WHERE gestureId=?" );


  /*
   * Collection Table
   */

  m_collectFetchAllQuery.prepare( "SELECT id,label FROM Collection" ); // safer not to use "*"
  m_collectFetchLabelQuery.prepare( "SELECT label FROM Collection WHERE id=?" );


  /*
   * CollectionMap Table
   */

  m_collectMapInsertQuery.prepare( "INSERT INTO CollectionMap "
				   "VALUES (?, ?)" );

  m_collectMapDeleteQuery.prepare( "DELETE FROM CollectionMap "
				   "WHERE gestureId=?" );


  /*
   * Experiment Table
   */

  // Note: The ID is absent from m_experInsertQuery, as it's auto-generated.
  m_experimentInsertQuery.prepare( "INSERT INTO Experiment "
				   "(label, external, date, notes, "
				   "trainedRecogniserId, "
				   "cachedCorrectText, cachedIncorrectText) "
				   "VALUES (?, ?, ?, ?, ?, ?, ?)" );

  m_experimentUpdateQuery.prepare( "UPDATE Experiment "
				   "SET label=?, external=?, date=?, "
				   "notes=?, trainedRecogniserId=?, "
				   "cachedCorrectText=?, cachedIncorrectText=? "
				   "WHERE id=?" );

  m_experimentDeleteQuery.prepare( "DELETE FROM Experiment "
				   "WHERE id=?" );

  m_experimentFetchAllBasicQuery.prepare( "SELECT id,label FROM Experiment" );

  m_experimentFetchLabelQuery.prepare( "SELECT label FROM Experiment WHERE id=?" );

  m_experimentFetchQuery.prepare( "SELECT label, external, date, notes, "
				  "trainedRecogniserId "
				  "FROM Experiment "
				  "WHERE id=?" );


  /*
   * Gesture Table
   */

  // Note: The ID is absent from m_gestInsertQuery, as it's auto-generated.
  m_gestInsertQuery.prepare( "INSERT INTO Gesture "
			     "(label, date, notes, strokeCount, strokeData) "
			     "VALUES (?, ?, ?, ?, ?)" );

  m_gestUpdateQuery.prepare( "UPDATE Gesture "
			     "SET label=?, date=?, notes=?, "
			     "strokeCount=?, strokeData=? "
			     "WHERE id=?" );

  m_gestDeleteQuery.prepare( "DELETE FROM Gesture "
			     "WHERE id=?" );
			     

  // Note: No need to fetch strokeCount, as we can get it from strokeData.
  //       - we only record it to make queries easier.
  m_gestFetchQuery.prepare( "SELECT label, date, notes, strokeData "
			    "FROM Gesture "
			    "WHERE id=?" );


  /*
   * Trained Recogniser Table
   */

  // Note: The ID is absent from m_recogInsertQuery, as it's auto-generated.
  m_recogInsertQuery.prepare( "INSERT INTO TrainedRecogniser "
			      "(label, recogniserKey, external, date, notes, "
			      "ready, modelFile, orderedFeatures, trainingSet) "
			      "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)" );

  m_recogUpdateQuery.prepare( "UPDATE TrainedRecogniser "
			      "SET label=?, recogniserKey=?, external=?, date=?, "
			      "notes=?, ready=?, modelFile=?, "
			      "orderedFeatures=?, trainingSet=? "
			      "WHERE id=?" );

  m_recogDeleteQuery.prepare( "DELETE FROM TrainedRecogniser "
			      "WHERE id=?" );

  m_recogFetchAllBasicQuery.prepare( "SELECT id,label FROM TrainedRecogniser" );

  m_recogFetchLabelQuery.prepare( "SELECT label FROM TrainedRecogniser WHERE id=?" );

  m_recogFetchQuery.prepare( "SELECT label, recogniserKey, external, date, notes, "
			     "ready, modelFile, orderedFeatures, trainingSet "
			     "FROM TrainedRecogniser "
			     "WHERE id=?" );
}



/*!
 * The model's event dispatcher.
 *
 * Asserts that the event is non-null.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 */
void DigestDbModel::dispatchEvent( MEvent* event )
{
  Q_ASSERT( event );

  AbstractModel::dispatchEvent( event );

  if ( event->moduleId() != MvcDigestDb::id() ) return; // prevent event type conflicts

  /* From the SQLite Wiki:
   * http://www.sqlite.org/cvstrac/wiki?p=PerformanceConsiderations
   * Accessed on Feb 15, 2006 @ 10:40am EST
   *
   * Transactions and performance
   *
   * When doing lots of updates/inserts on a table it is a good idea to contain
   * them within a transaction,
   *
   *   begin;
   *   insert into table values (..);
   *   insert into table values (..);
   *   insert into table values (..);
   *   ....
   *   commit;
   *
   * This will make SQLite write all the data to the disk in one go, vastly
   * increasing performance.
   *
   * However, if you are writing to a temporary table, transactions have less
   * effect because disk writes are not flushed to the table after each write.
   *
   * I did a timed test with inserting 1000 records into a table in various ways
   * to compare performance:
   *  - main database table without transaction - 5 seconds
   *  - main database table with transaction - 0.1 seconds
   *  - temporary table without transaction - 2 seconds
   *  - temporary table with transaction - 0.1 seconds
   *
   * So, performance is still vastly quicker with a transaction when writing to a
   * temporary, but transactions have the drawback of locking the entire database
   * file for the duration of the transaction, even though only a temporary file
   * is being written to, so, in multithreaded applications, it may be worth
   * putting up with the lower performance to avoid this database locking behaviour.
   *
   * --Paul Smith
   *
   *
   * Personal experience while developing GestureLab reconfirmed the results above.
   *
   * Adding ~250 gestures to an collection on a PowerBook G4, with a 4200RPM HDD
   * and using SQLite 3.1.3, the results were:
   *   - main database table without transaction - ~60 seconds
   *   - main database table with transaction - << 1 second
   *
   * -- Aidan Lane
   */

  m_db.transaction(); // BEGIN transaction

  switch ( event->type() )
    {
    case MDigestDbEvent::ClassAdd:
      classAddEvent( static_cast<MClassAddEvent*>(event) ); break;
    case MDigestDbEvent::ClassUpdate:
      classUpdateEvent( static_cast<MClassUpdateEvent*>(event) ); break;
    case MDigestDbEvent::ClassesRemove:
      classesRemoveEvent( static_cast<MClassesRemoveEvent*>(event) ); break;

    case MDigestDbEvent::CollectionAdd:
      collectionAddEvent( static_cast<MCollectionAddEvent*>(event) ); break;
    case MDigestDbEvent::CollectionUpdate:
      collectionUpdateEvent( static_cast<MCollectionUpdateEvent*>(event) ); break;
    case MDigestDbEvent::CollectionsRemove:
      collectionsRemoveEvent( static_cast<MCollectionsRemoveEvent*>(event) ); break;

    case MDigestDbEvent::ExperimentAdd:
      experimentAddEvent( static_cast<MExperimentAddEvent*>(event) ); break;
    case MDigestDbEvent::ExperimentUpdate:
      experimentUpdateEvent( static_cast<MExperimentUpdateEvent*>(event) ); break;
    case MDigestDbEvent::ExperimentsRemove:
      experimentsRemoveEvent( static_cast<MExperimentsRemoveEvent*>(event) ); break;

    case MDigestDbEvent::GestureAdd:
      gestureAddEvent( static_cast<MGestureAddEvent*>(event) ); break;
    case MDigestDbEvent::GestureUpdate:
      gestureUpdateEvent( static_cast<MGestureUpdateEvent*>(event) ); break;
    case MDigestDbEvent::GesturesChangeClasses:
      gesturesChangeClassesEvent( static_cast<MGesturesChangeClassesEvent*>(event) ); break;
    case MDigestDbEvent::GesturesChangeCollections:
      gesturesChangeCollectionsEvent( static_cast<MGesturesChangeCollectionsEvent*>(event) ); break;
    case MDigestDbEvent::GesturesRemove:
      gesturesRemoveEvent( static_cast<MGesturesRemoveEvent*>(event) ); break;

    case MDigestDbEvent::TrainedRecogAdd:
      trainedRecogAddEvent( static_cast<MTrainedRecogAddEvent*>(event) ); break;
    case MDigestDbEvent::TrainedRecogUpdate:
      trainedRecogUpdateEvent( static_cast<MTrainedRecogUpdateEvent*>(event) ); break;
    case MDigestDbEvent::TrainedRecogsRemove:
      trainedRecogsRemoveEvent( static_cast<MTrainedRecogsRemoveEvent*>(event) ); break;

    default:
      break;
    }

  m_db.commit(); // COMMIT transaction
}


/*!
 * Re-implemented, as to also have the pointer returned by digestDbController()
 * updated.
 *
 * Asserts that the \em event is non-null.
 */
void DigestDbModel::changeControllerEvent( MChangeControllerEvent* event )
{
  Q_ASSERT( event );
  AbstractModel::changeControllerEvent( event );
  c_digestDbController
    = ( (event->controller()==0)
	? 0
	: qobject_cast<DigestDbController*>(event->controller()->objectPtr()) );
}


void DigestDbModel::classAddEvent( MClassAddEvent* event )
{
  // Remember: The ID field is auto-generated.
  Q_ASSERT( event );
  QSqlQuery q( m_db );
  m_lastInsertedClassIdSet.clear();
  q.prepare( "INSERT INTO Class (label) VALUES (?)" );
  q.addBindValue( event->record().label ); // looks after all quoting issues
  if ( q.exec()
       && q.isActive() )
    {
      bool intOk = false;
      Q_ASSERT( q.driver() );
      Q_ASSERT( q.driver()->hasFeature(QSqlDriver::LastInsertId) );
      int id = q.lastInsertId().toInt( &intOk );
      Q_ASSERT( intOk );
      if ( id < 0 ) {
	handleDbError();
	return;
      }
      m_lastInsertedClassIdSet += id;
    }
}

void DigestDbModel::classUpdateEvent( MClassUpdateEvent* event )
{
  Q_ASSERT( event );
  const DClassRecord& record = event->record();

  // TODO: pre-prepare the query
  QSqlQuery q( m_db );
  q.prepare( "UPDATE Class SET label=? WHERE id=?" );
  q.addBindValue( record.label ); // looks after all quoting issues
  q.addBindValue( record.id );    // YES, this is last - part of the WHERE

  if ( ! q.exec()
       || ! q.isActive() ) {
    handleDbError();
    return;
  }
}

void DigestDbModel::classesRemoveEvent( MClassesRemoveEvent* event )
{
  Q_ASSERT( event );
  if ( removeFromTable(s_classTable, "id", event->idSet()) )
    removeFromTable( s_classMapTable, "classId", event->idSet() );
}


void DigestDbModel::collectionAddEvent( MCollectionAddEvent* event )
{
  // Remember: The ID field is auto-generated.
  Q_ASSERT( event );
  QSqlQuery q( m_db );
  m_lastInsertedCollectionIdSet.clear();
  q.prepare( "INSERT INTO Collection (label) VALUES (?)" );
  q.addBindValue( event->record().label ); // looks after all quoting issues
  if ( q.exec()
       && q.isActive() )
    {
      bool intOk = false;
      Q_ASSERT( q.driver() );
      Q_ASSERT( q.driver()->hasFeature(QSqlDriver::LastInsertId) );
      int id = q.lastInsertId().toInt( &intOk );
      Q_ASSERT( intOk );
      if ( id < 0 ) {
	handleDbError();
	return;
      }
      m_lastInsertedCollectionIdSet += id;
    }
}

void DigestDbModel::collectionUpdateEvent( MCollectionUpdateEvent* event )
{
  Q_ASSERT( event );
  const DCollectionRecord& record = event->record();

  // TODO: pre-prepare the query
  QSqlQuery query;
  query.prepare( "UPDATE Collection SET label=? WHERE id=?" );
  query.addBindValue( record.label ); // looks after all quoting issues
  query.addBindValue( record.id );    // YES, this is last - part of the WHERE

  if ( ! query.exec()
       || ! query.isActive() ) {
    handleDbError();
    return;
  }
}

void DigestDbModel::collectionsRemoveEvent( MCollectionsRemoveEvent* event )
{
  Q_ASSERT( event );
  if ( removeFromTable(s_collectionTable, "id", event->idSet()) )
    removeFromTable( s_collectionMapTable, "collectionId", event->idSet() );
}



void DigestDbModel::experimentAddEvent( MExperimentAddEvent* event )
{
  // Remember: The ID field is auto-generated.
  Q_ASSERT( event );
  const DExperimentRecord& record = event->record();
  m_lastInsertedExperimentIdSet.clear();
  m_experimentInsertQuery.addBindValue( record.label );
  m_experimentInsertQuery.addBindValue( record.external );
  m_experimentInsertQuery.addBindValue( record.date.toString(DATE_STORE_FORMAT_STR) );
  m_experimentInsertQuery.addBindValue( record.notes );
  m_experimentInsertQuery.addBindValue( record.trainedRecogniserId );
  m_experimentInsertQuery.addBindValue( record.cachedCorrectText );
  m_experimentInsertQuery.addBindValue( record.cachedIncorrectText );
  if ( m_experimentInsertQuery.exec()
       && m_experimentInsertQuery.isActive() )
    {
      bool intOk = false;
      Q_ASSERT( m_experimentInsertQuery.driver() );
      Q_ASSERT( m_experimentInsertQuery.driver()->hasFeature(QSqlDriver::LastInsertId) );
      int id = m_experimentInsertQuery.lastInsertId().toInt( &intOk );
      Q_ASSERT( intOk );
      if ( id < 0 ) {
	handleDbError();
	return;
      }
      m_lastInsertedExperimentIdSet += id;
    }
}

void DigestDbModel::experimentUpdateEvent( MExperimentUpdateEvent* event )
{
  Q_ASSERT( event );
  const DExperimentRecord& record = event->record();

  m_experimentUpdateQuery.addBindValue( record.label );
  m_experimentUpdateQuery.addBindValue( record.external );
  m_experimentUpdateQuery.addBindValue( record.date.toString(DATE_STORE_FORMAT_STR) );
  m_experimentUpdateQuery.addBindValue( record.notes );
  m_experimentUpdateQuery.addBindValue( record.trainedRecogniserId );
  m_experimentUpdateQuery.addBindValue( record.cachedCorrectText );
  m_experimentUpdateQuery.addBindValue( record.cachedIncorrectText );
  m_experimentUpdateQuery.addBindValue( record.id ); // YES, this is last - part of WHERE

  if ( ! m_experimentUpdateQuery.exec()
       || ! m_experimentUpdateQuery.isActive() ) { // isActive -> success
    handleDbError();
    return;
  }
}

void DigestDbModel::experimentsRemoveEvent( MExperimentsRemoveEvent* event )
{
  Q_ASSERT( event );
  if ( removeFromTable(s_experimentTable, "id", event->idSet()) )
    removeFromTable( s_experimentResultTable, "experimentId", event->idSet() );
}



/*!
 * Inserts the gesture into the SQL database.
 *
 * The ID in the given DGestureRecord is ignored, as it's generated by the DB backend.
 *
 * Asserts that the event is non-null.
 */
void DigestDbModel::gestureAddEvent( MGestureAddEvent* event )
{
  Q_ASSERT( event );
  const DGestureRecord& g = event->record();
  bool intOk = false;

  // Insert the gesture the Gesture table
  // Remember: The ID field is auto-generated.
  m_gestInsertQuery.addBindValue( g.label );
  m_gestInsertQuery.addBindValue( g.date.toString(DATE_STORE_FORMAT_STR) );
  m_gestInsertQuery.addBindValue( g.notes );
  m_gestInsertQuery.addBindValue( g.strokes.size() );
  m_gestInsertQuery.addBindValue( g.strokes.toString() );

  if ( ! m_gestInsertQuery.exec()
       || ! m_gestInsertQuery.isActive() ) { // isActive -> success
    handleDbError();
    return;
  }

  // Make note of the identity of the new gesture
  Q_ASSERT( m_gestInsertQuery.driver() );
  Q_ASSERT( m_gestInsertQuery.driver()->hasFeature(QSqlDriver::LastInsertId) );
  m_lastInsertedGestureIdSet.clear();
  m_lastInsertedGestureIdSet += m_gestInsertQuery.lastInsertId().toInt(&intOk);
  Q_ASSERT( intOk );
  if ( *m_lastInsertedGestureIdSet.begin() < 0 )  {
    handleDbError();
    return;
  }

  // Insert the mappings into the ClassMap and CollectionMap tables
  gesturesChangeCategories( s_classMapTable, "classId",
			    m_lastInsertedGestureIdSet, g.classes, IdSet() );
  gesturesChangeCategories( s_collectionMapTable, "collectionId",
			    m_lastInsertedGestureIdSet, g.collections, IdSet() );
}

void DigestDbModel::gestureUpdateEvent( MGestureUpdateEvent* event )
{
  Q_ASSERT( event );
  const DGestureRecord& g = event->record();

  m_gestUpdateQuery.addBindValue( g.label );
  m_gestUpdateQuery.addBindValue( g.date.toString(DATE_STORE_FORMAT_STR) );
  m_gestUpdateQuery.addBindValue( g.notes );
  m_gestUpdateQuery.addBindValue( g.strokes.size() );
  m_gestUpdateQuery.addBindValue( g.strokes.toString() );
  m_gestUpdateQuery.addBindValue( g.id ); // YES, this is last - part of the WHERE

  if ( ! m_gestUpdateQuery.exec()
       || ! m_gestUpdateQuery.isActive() ) { // isActive -> success
    handleDbError();
    return;
  }

  /* Update the class and collection mappings.
   * NOTE: The algorithm used here ISN'T efficient, given that a given mapping
   *       may be deleted and then inserted back in again, but after initially
   *       implementing a more sophisticated alg, I believe that this shorter,
   *       cleaner approach is better.
   */
  IdSet idSet;
  idSet += g.id;

  m_classMapDeleteQuery.addBindValue( g.id );
  if ( ! m_classMapDeleteQuery.exec() 
       || ! m_classMapDeleteQuery.isActive() ) {
    handleDbError();
    return;
  }
  gesturesChangeCategories( s_classMapTable, "classId",
			    idSet, g.classes, IdSet() );

  m_collectMapDeleteQuery.addBindValue( g.id );
  if ( ! m_collectMapDeleteQuery.exec() 
       || ! m_collectMapDeleteQuery.isActive() ) {
    handleDbError();
    return;
  }
  gesturesChangeCategories( s_collectionMapTable, "collectionId",
			    idSet, g.collections, IdSet() );
}

/*!
 * \b Note: If an ID appears in both CGesturesChangeClassesEvent::addSet() and
 *          CGesturesChangeClassesEvent::removeSet(), then the the ID will be in
 *          the final classes set, as it will first be removed and then
 *          immediately added back.
 *
 * Asserts that the event is non-null.
 */
void DigestDbModel::gesturesChangeClassesEvent( MGesturesChangeClassesEvent* ev )
{
  Q_ASSERT( ev );
  gesturesChangeCategories( s_classMapTable, "classId",
			    ev->idSet(), ev->addSet(), ev->removeSet() );
}

/*!
 * \b Note: If an ID appears in both CGesturesChangeCollectionsEvent::addSet()
 *          and CGesturesChangeCollectionsEvent::removeSet(), then the the ID
 *          will be in the final collections set, as it will first be removed and
 *          then immediately added back.
 *
 * Asserts that the event is non-null.
 */
void DigestDbModel::gesturesChangeCollectionsEvent( MGesturesChangeCollectionsEvent* ev )
{
  Q_ASSERT( ev );
  gesturesChangeCategories( s_collectionMapTable, "collectionId",
			    ev->idSet(), ev->addSet(), ev->removeSet() );
}

void DigestDbModel::gesturesRemoveEvent( MGesturesRemoveEvent* event )
{
  Q_ASSERT( event );
  if ( removeFromTable(s_classMapTable, "gestureId", event->idSet()) )
    if ( removeFromTable(s_collectionMapTable, "gestureId", event->idSet()) )
      removeFromTable( s_gestureTable, "id", event->idSet() );
}



void DigestDbModel::trainedRecogAddEvent( MTrainedRecogAddEvent* event )
{
  Q_ASSERT( event );
  const DTrainedRecogRecord& record = event->record();

  // TODO: optimise me: write directly to a QByteArray a we build it
  QStringList orderedFeaturesStrList;
  foreach ( const QByteArray& key, record.orderedFeatures )
    orderedFeaturesStrList += key;

  // TODO: optimise me:
  QStringList trainingSetStrList;
  foreach ( const int id, record.trainingSet )
    trainingSetStrList += QString::number(id);

  m_recogInsertQuery.addBindValue( record.label );
  m_recogInsertQuery.addBindValue( record.recogniserKey );
  m_recogInsertQuery.addBindValue( record.external );
  m_recogInsertQuery.addBindValue( record.date.toString(DATE_STORE_FORMAT_STR) );
  m_recogInsertQuery.addBindValue( record.notes );
  m_recogInsertQuery.addBindValue( record.ready );
  m_recogInsertQuery.addBindValue( record.modelFile );
  m_recogInsertQuery.addBindValue( orderedFeaturesStrList.join(",").toAscii() );
  m_recogInsertQuery.addBindValue( trainingSetStrList.join(",").toAscii() );

  if ( ! m_recogInsertQuery.exec()
       || ! m_recogInsertQuery.isActive() ) { // isActive -> success
    handleDbError();
    return;
  }

  // Make note of the identity of the new trained recogniser, m_lastInsertedTrainedRecogId
  bool intOk = false;
  m_lastInsertedTrainedRecogIdSet.clear(); // just in case anything fails
  Q_ASSERT( m_recogInsertQuery.driver() );
  Q_ASSERT( m_recogInsertQuery.driver()->hasFeature(QSqlDriver::LastInsertId) );
  m_lastInsertedTrainedRecogIdSet += m_recogInsertQuery.lastInsertId().toInt(&intOk);
  Q_ASSERT( intOk );
  if ( *m_lastInsertedTrainedRecogIdSet.begin() < 0 ) {
    handleDbError();
    return;
  }
}

void DigestDbModel::trainedRecogUpdateEvent( MTrainedRecogUpdateEvent* event )
{
  Q_ASSERT( event );
  const DTrainedRecogRecord& record = event->record();

  // TODO: optimise me: write directly to a QByteArray a we build it
  QStringList orderedFeaturesStrList;
  foreach ( const QByteArray& key, record.orderedFeatures )
    orderedFeaturesStrList += key;

  // TODO: optimise me:
  QStringList trainingSetStrList;
  foreach ( const int id, record.trainingSet )
    trainingSetStrList += QString::number(id);

  m_recogUpdateQuery.addBindValue( record.label );
  m_recogUpdateQuery.addBindValue( record.recogniserKey );
  m_recogUpdateQuery.addBindValue( record.external );
  m_recogUpdateQuery.addBindValue( record.date.toString(DATE_STORE_FORMAT_STR) );
  m_recogUpdateQuery.addBindValue( record.notes );
  m_recogUpdateQuery.addBindValue( record.ready );
  m_recogUpdateQuery.addBindValue( record.modelFile );
  m_recogUpdateQuery.addBindValue( orderedFeaturesStrList.join(",").toAscii() );
  m_recogUpdateQuery.addBindValue( trainingSetStrList.join(",").toAscii() );
  m_recogUpdateQuery.addBindValue( record.id ); // YES, this is last - part of WHERE

  if ( ! m_recogUpdateQuery.exec()
       || ! m_recogUpdateQuery.isActive() ) { // isActive -> success
    handleDbError();
    return;
  }
}

void DigestDbModel::trainedRecogsRemoveEvent( MTrainedRecogsRemoveEvent* event )
{
  Q_ASSERT( event );
  removeFromTable( s_trainedRecogTable, "id", event->idSet() );
}


QHash<int, QString> DigestDbModel::fetchClasses() const
{
  QHash<int, QString> classes;
  bool intOk = false;

  m_classFetchAllQuery.exec();
  while ( m_classFetchAllQuery.next() ) {
    classes.insert( m_classFetchAllQuery.value(0).toInt(&intOk),
		    m_classFetchAllQuery.value(1).toString() );
    Q_ASSERT( intOk );
  }

  return classes;
}


// TODO: cache the labels in a hash!
QString DigestDbModel::fetchClassLabel( int id ) const
{
  // TODO: re-enable the use of the prepared query: m_classFetchLabelQuery
  //       - stopped using it because after using it insert and remove would fail to work.
#if 0
  m_classFetchLabelQuery.addBindValue( QString::number(id) );
  m_classFetchLabelQuery.exec();
  if ( m_classFetchLabelQuery.next() ) // only use the FIRST occurrence
    return m_classFetchLabelQuery.value(0).toString();
#else
  QSqlQuery query( "SELECT label FROM Class WHERE id=" + QString::number(id) );
  if ( query.next() ) // only use the FIRST occurrence
    return query.value(0).toString();
#endif

  return QString();
}


bool DigestDbModel::classLabelExists( const QString& label ) const {
  QSqlQuery q( "SELECT id FROM Class WHERE label=?", m_db );
  q.addBindValue( label ); //  looks after all quoting issues
  return q.next(); // next == true -> label exists
}


QHash<int, QString> DigestDbModel::fetchCollections() const
{
  QHash<int, QString> collections;
  bool intOk = false;

  m_collectFetchAllQuery.exec();
  while ( m_collectFetchAllQuery.next() ) {
    collections.insert( m_collectFetchAllQuery.value(0).toInt(&intOk),
			m_collectFetchAllQuery.value(1).toString() );
    Q_ASSERT( intOk );
  }

  return collections;
}


// TODO: cache the labels in a hash!
QString DigestDbModel::fetchCollectionLabel( int id ) const
{
  // TODO: re-enable the use of the prepared query: m_collectFetchLabelQuery
  //       - stopped using it because after using it insert and remove would fail to work.
#if 0
  m_collectFetchLabelQuery.addBindValue( QString::number(id) );
  m_collectFetchLabelQuery.exec();
  if ( m_collectFetchLabelQuery.next() ) // only use the FIRST occurrence
    return m_collectFetchLabelQuery.value(0).toString();
#else
  QSqlQuery query( "SELECT label FROM Collection WHERE id=" + QString::number(id), m_db );
  if ( query.next() ) // only use the FIRST occurrence
    return query.value(0).toString();
#endif

  return QString();
}


bool DigestDbModel::collectionLabelExists( const QString& label ) const {
  QSqlQuery q( "SELECT id FROM Collection WHERE label=?", m_db );
  q.addBindValue( label ); //  looks after all quoting issues
  return q.next(); // next == true -> label exists
}


QHash<int, QString> DigestDbModel::fetchExperimentsBasic() const
{
  QHash<int, QString> experiments;
  bool intOk = false;

  m_experimentFetchAllBasicQuery.exec();
  while ( m_experimentFetchAllBasicQuery.next() ) {
    experiments.insert( m_experimentFetchAllBasicQuery.value(0).toInt(&intOk),
			m_experimentFetchAllBasicQuery.value(1).toString() );
    Q_ASSERT( intOk );
  }

  return experiments;
}


// TODO: cache the labels in a hash!
QString DigestDbModel::fetchExperimentLabel( int id ) const
{
  // TODO: re-enable the use of the prepared query: m_experimentFetchLabelQuery
  //       - stopped using it because after using it insert and remove would fail to work.
#if 0
  m_experimentFetchLabelQuery.addBindValue( QString::number(id) );
  m_experimentFetchLabelQuery.exec();
  if ( m_experimentFetchLabelQuery.next() ) // only use the FIRST occurrence
    return m_experimentFetchLabelQuery.value(0).toString();
#else
  QSqlQuery query( "SELECT label FROM Experiment WHERE id=" + QString::number(id), m_db );
  if ( query.next() ) // only use the FIRST occurrence
    return query.value(0).toString();
#endif

  return QString();
}


DExperimentRecord DigestDbModel::fetchExperiment( int id, bool* ok ) const
{
  DExperimentRecord rec;
  bool resultOk = false;
  bool intOk = false;

  // TODO: re-enable the use of the prepared query: m_experimentFetchQuery
  //       - stopped using it because after using it insert and remove would fail to work.
  QSqlQuery query( "SELECT label, external, date, notes, trainedRecogniserId, "
		   "cachedCorrectText, cachedIncorrectText "
		   "FROM Experiment "
		   "WHERE id=" + QString::number(id), m_db );
  #if 0
  m_experimentFetchQuery.addBindValue( id );
  #endif

  if (
#if 0
m_experimentFetchQuery.exec()
&&
#endif
       query.isActive()
       && query.next() ) // only use the FIRST occurrence
    {
      rec.id = id;
      rec.label = query.value(0).toString();
      rec.external = query.value(1).toBool();
      rec.date = QDate::fromString( query.value(2).toString(), DATE_STORE_FORMAT_STR );
      rec.notes = query.value(3).toString();
      rec.trainedRecogniserId = query.value(4).toInt(&intOk);
      rec.cachedCorrectText = query.value(5).toString();
      rec.cachedIncorrectText = query.value(6).toString();
      Q_ASSERT( intOk );

      resultOk = true; // all seems well...
    }

  if ( ok ) *ok = resultOk;
  return rec;
}


/*!
 * Returns the number of gestures in the database.
 *
 * Returns -1 if there was an error.
 */
int DigestDbModel::gestureCount() const
{
  int c = -1;
  bool intOk = false;
  QSqlQuery q( "SELECT COUNT(id) FROM Gesture", m_db );
  if ( q.next() ) {
    c = q.value(0).toInt( &intOk );
    Q_ASSERT( intOk );
  }
  return c;
}


/*!
 * Fetches the first gesture from the "Gesture" table that has the given ID.
 *
 * Only returning the first match sould not be a problem, given that the ID
 * should be unique anyway.
 *
 * If a gesture with the given ID could not be found or if there was some other
 * error, then "ok" will be set to false, otherwise it will be set to true.
 * If a null pointer is passed for "ok", then it will not attempt to set it.
 */
// TODO: INTRODUCE CACHING AND GET ALL DIGEST COMPONENTS TO USE THIS!!!
DGestureRecord DigestDbModel::fetchGesture( int id, bool* ok ) const
{
  // TODO: see if the query can be cached?!?!?
  // Note: We don't select "id" (we already know it) nor "strokeCount" (in strokes)
  // Note: Use of quotes around the fields, e.g. "'id'=" does not work with Qt+SQLite
  QString idStr = QString::number( id );
  QSqlQuery gq( "SELECT label,date,notes,strokeData FROM Gesture WHERE id=" + idStr );
  DGestureRecord g;
  bool resultOk = false;
  bool intOk = false;

  if ( gq.next() ) // only use the FIRST occurrence - should only be one anyway
    {
      // Note: The indices used below reflect the SELECT statement
      g.id = id;
      g.label = gq.value(0).toString();
      g.date = QDate::fromString( gq.value(1).toString(), DATE_STORE_FORMAT_STR );
      g.notes = gq.value(2).toString();
      g.strokes = StrokeList::fromString( gq.value(3).toString() );

      // get classes it belongs to
      // TODO: use a prepared query
      QSqlQuery classq( "SELECT classId "
			"FROM ClassMap "
			"WHERE gestureId=" + idStr );
      Q_ASSERT( g.classes.isEmpty() ); // assumed true by the following:
      while ( classq.next() ) {	
	g.classes += classq.value(0).toInt(&intOk);
	Q_ASSERT( intOk );
      }

      // get collections it belongs to
      // TODO: use a prepared query
      QSqlQuery collectionq( "SELECT collectionId "
			     "FROM CollectionMap "
			     "WHERE gestureId=" + idStr );
      Q_ASSERT( g.collections.isEmpty() ); // assumed true by the following:
      while ( collectionq.next() ) {	
	g.collections += collectionq.value(0).toInt(&intOk);
	Q_ASSERT( intOk );
      }

      resultOk = true; // all seems well...
    }

  if ( ok ) *ok = resultOk;
  return g;
}


QHash<int, QString> DigestDbModel::fetchTrainedRecogsBasic() const
{
  QHash<int, QString> recogs;
  bool intOk = false;

  m_recogFetchAllBasicQuery.exec();
  while ( m_recogFetchAllBasicQuery.next() ) {
    recogs.insert( m_recogFetchAllBasicQuery.value(0).toInt(&intOk),
		   m_recogFetchAllBasicQuery.value(1).toString() );
    Q_ASSERT( intOk );
  }

  return recogs;
}


// TODO: cache the labels in a hash!
QString DigestDbModel::fetchTrainedRecogLabel( int id ) const
{
  // TODO: re-enable the use of the prepared query: m_recogFetchLabelQuery
  //       - stopped using it because after using it insert and remove would fail to work.
#if 0
  m_recogFetchLabelQuery.addBindValue( QString::number(id) );
  m_recogFetchLabelQuery.exec();
  if ( m_recogFetchLabelQuery.next() ) // only use the FIRST occurrence
    return m_recogFetchLabelQuery.value(0).toString();
#else
  QSqlQuery query( "SELECT label FROM TrainedRecog WHERE id=" + QString::number(id) );
  if ( query.next() ) // only use the FIRST occurrence
    return query.value(0).toString();
#endif

  return QString();
}


DTrainedRecogRecord DigestDbModel::fetchTrainedRecog( int id, bool* ok ) const
{
  DTrainedRecogRecord rec;
  bool resultOk = false;
  bool intOk = false;

  // TODO: re-enable the use of the prepared query: m_recogFetchQuery
  //       - stopped using it because after using it insert and remove would fail to work.
  QSqlQuery query( "SELECT label, recogniserKey, external, date, notes, "
		   "ready, modelFile, orderedFeatures, trainingSet "
		   "FROM TrainedRecogniser "
		   "WHERE id=" + QString::number(id) );
  #if 0
  m_recogFetchQuery.addBindValue( id );
  #endif

  if (
#if 0
m_recogFetchQuery.exec()
&&
#endif
       query.isActive()
       && query.next() ) // only use the FIRST occurrence
    {
      rec.id = id;
      rec.label = query.value(0).toString();
      rec.recogniserKey = query.value(1).toByteArray();
      rec.external = query.value(2).toBool();
      rec.date = QDate::fromString( query.value(3).toString(), DATE_STORE_FORMAT_STR );
      rec.notes = query.value(4).toString();
      rec.ready = query.value(5).toBool();
      rec.modelFile = query.value(6).toString();

      QStringList orderedFeaturesStrList
	= query.value(7).toString().split(",", QString::SkipEmptyParts);
      foreach ( const QString& str, orderedFeaturesStrList )
	rec.orderedFeatures += str.toAscii();

      QStringList trainingSetStrList
	= query.value(8).toString().split(",", QString::SkipEmptyParts);
      foreach ( const QString& str, trainingSetStrList ) {
	rec.trainingSet += str.toInt(&intOk);
	Q_ASSERT( intOk );
      }

      resultOk = true; // all seems well...
    }

  if ( ok ) *ok = resultOk;
  return rec;
}


/*!
 * Unlike idSetToString(), this method fetches the actual labels of the
 * classes and also sorts them.
 *
 * This method was designed to be used for presentation purposes.
 */
QString DigestDbModel::classesToString( const QSet<int>& classes,
					const QString& joinSep ) const
{
  QStringList strs;
  QSetIterator<int> it( classes );
  while ( it.hasNext() )
    strs += fetchClassLabel(it.next()); // CLASSES SPECIFIC
  strs.sort(); // sort the class names
  return strs.join( joinSep );
}


/*!
 * Unlike idSetToString(), this method fetches the actual labels of the
 * collections and also sorts them.
 *
 * This method was designed to be used for presentation purposes.
 */
QString DigestDbModel::collectionsToString( const QSet<int>& collections,
					    const QString& joinSep ) const
{
  QStringList strs;
  QSetIterator<int> it( collections );
  while ( it.hasNext() )
    strs += fetchCollectionLabel(it.next()); // COLLECTIONS SPECIFIC
  strs.sort(); // sort the collection names
  return strs.join( joinSep );
}


/*!
 * \b Note: If an ID appears in both addSet and removeSet, then the the ID will
 *          will be in the final category set, as it will first be removed and
 *          then immediately added back.
 */
bool DigestDbModel::gesturesChangeCategories( const QString& table,
					      const QString& categoryColumn,
					      const IdSet& idSet,
					      const IdSet& addSet,
					      const IdSet& removeSet )
{
  QSqlQuery q( m_db );

  // Remove the given gesture idSet from the specified class removeSet
  // Warning: Don't use a prepared query because the result of idSetToString()
  //          will be quoted - breaking the query.
  if ( !q.exec( QString("DELETE FROM %1 WHERE gestureId IN (%2) AND %3 IN (%4)")
		.arg(table)
		.arg(idSetToString(idSet))
		.arg(categoryColumn)
		.arg(idSetToString(removeSet)) )
       || !q.isActive() ) { // isActive -> success
    handleDbError();
    return false;
  }

  // Add the given gesture idSet to the specified class addSet
  q.prepare( QString("INSERT INTO %1 VALUES (?, ?)").arg(table) );
  foreach ( int gestureId, idSet ) {
    foreach ( int categoryId, addSet ) {
      q.addBindValue( gestureId );
      q.addBindValue( categoryId );
      if ( !q.exec() || !q.isActive() ) { // isActive -> success
	handleDbError();
	return false;
      }
    }
  }

  return true;
}


bool DigestDbModel::removeFromTable( const QString& table,
				     const QString& idColumn,
				     const IdSet& idSet )
{
  // Warning: Don't use a prepared query because the result of idSetToString()
  //          will be quoted - breaking the query.
  QSqlQuery q( m_db );
  q.exec( QString("DELETE FROM %1 WHERE %2 IN (%3)")
	  .arg(table)
	  .arg(idColumn)
	  .arg(idSetToString(idSet)) );
  if ( !q.exec() || !q.isActive() ) { // isActive -> success
    handleDbError();
    return false;
  }
  return true;
}


/*!
 * Unlike classesToString() and collectionsToString(), this method does \b not
 * fetch the labels of the IDs, it simply takes a set like [4,5,7,8] and returns
 * a string like "4,5,7,8" (with the commas).
 *
 * This method was designed to be used when building SQL queries.
 */
QString DigestDbModel::idSetToString( const IdSet& idSet )
{
  QString s;
  foreach ( int id, idSet )
    s += QString::number(id) + ",";
  if ( !s.isEmpty() ) s.chop(1); // chop the last "," off
  return s;
}


// TODO: impl me, replace me with exception handling???
void DigestDbModel::handleDbError()
{
  qDebug("SQL ERROR: handleDbError - rolling back transaction...");
  m_db.rollback(); // ROLLBACK transaction
  qDebug("done.");
}


bool DigestDbModel::createTables( const QSqlDatabase& db )
{
  QSqlQuery query( db );

  /*
   * WARNING: SQLite does not like:
   *          "INT", it MUST be "INTEGER" for auto-incrementing.
   *          "AUTO_INCREMENT", MUST also be "AUTOINCREMENT".
   *
   * Note: SQLite uses variable-length records, hence, no explicit length is
   *       required. Therefore, we don't use VARCHAR, but TEXT instead.
   */

  // TODO: introduce error checking! (but NOT assertions)

  // TODO: change datatypes to those that will be compatible
  // with both SQLite AND other SQL databases!!! (e.g. VARCHAR vs. TEXT)

  query.exec( "CREATE TABLE Gesture("
	      "'id' INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
	      "'label' TEXT,"
	      "'date' TEXT," // TODO: revise datatype
	      "'notes' TEXT,"
	      "'strokeCount' INTEGER,"
	      "'strokeData' BLOB)" );

  query.exec( "CREATE TABLE Class("
	      "'id' INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
	      "'label' TEXT)" );

  query.exec( "CREATE TABLE ClassMap("
	      "'gestureId' INTEGER,"
	      "'classId' INTEGER)" );

  query.exec( "CREATE TABLE Collection("
	      "'id' INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
	      "'label' TEXT)" );

  query.exec( "CREATE TABLE CollectionMap("
	      "'gestureId' INTEGER,"
	      "'collectionId' INTEGER)" );

  /* TODO: DOC: rewrite this to match current system
   * Notes:
   *
   * 1. The class ID of the gesture that is tested is also recorded, even though
   *    it's redundant. We do so because the gesture could be removed at any time
   *    - but we wan't the results to still be of use.
   *
   * 2. The results could have been kept more compact if each of the class->prob
   *    entries were linked together in one field, BUT the format used aids
   *    querying of the SQL database.
   *
   * 3. There may be multiple rows with the same experimentId, testGestureId and
   *    testClassId. This will occur if for example the recogniser in experiment
   *    E returns with a prob. of 0.9 that gesture G of class C belongs to class
   *    X and 0.1 that be belongs to class Y.
   *
   * 4. As noted for the experiment table: Even though the Experiment Assistant
   *    may be used to perform experiments with multiple trained recognisers at
   *    once, each recogniser tested will have it's own experiment ID and thus
   *    also a row of the Experiment table assigned to it (for that experiment
   *    run).
   */
  query.exec( "CREATE TABLE ExperimentResult("
	      "'experimentId' INTEGER,"
	      "'testGestureId' INTEGER,"
	      "'testClassId' INTEGER,"
	      "'highestResultClassId' INTEGER,"
	      "'highestResultClassProb' REAL,"
	      "'otherResults' BLOB)" );

  /* TODO: DOC: rewrite this to match current system
   * Notes:
   *
   * 1. Even though the Experiment Assistant may be used to perform experiments
   *    with multiple trained recognisers at once, each recogniser tested will
   *    have it's own experiment ID and thus also a row of the Experiment table
   *    assigned to it (for that experiment run).
   *
   * 2. The database can be used to store the results of external experiments,
   *    thus the reason for having the "external" boolean field.
   *
   * 3. The actual results of each experiment are stored in the ExperimentResult
   *    table.
   */
  query.exec( "CREATE TABLE Experiment("
	      "'id' INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
	      "'label' TEXT,"
	      "'external' INTEGER,"
	      "'date' TEXT,"
	      "'notes' TEXT,"
	      "'trainedRecogniserId' INTEGER,"
	      "'cachedCorrectText' TEXT,"
	      "'cachedIncorrectText' TEXT)" );

  query.exec( "CREATE TABLE TrainedRecogniser("
	      "'id' INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
	      "'label' TEXT,"
	      "'recogniserKey' TEXT,"
	      "'external' INTEGER,"
	      "'date' TEXT,"
	      "'notes' TEXT,"
	      "'ready' INTEGER," // VERY important, prevents a race condition
	      "'modelFile' TEXT,"
	      "'orderedFeatures' BLOB,"
	      "'trainingSet' BLOB)" );

  return true;
}
