/*  -*- c++ -*-  (for Emacs)
 *
 *  digestdbmodel.h
 *  Digest
 * 
 *  Created by Aidan Lane on Fri Aug 12 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef DIGESTDBMODEL_H
#define DIGESTDBMODEL_H


#include "MvcCore/abstractmodel.h"

#include <QCache>
#include <QList>
#include <QPointer>
#include <QSqlDatabase>
#include <QSqlQuery>

#include "digestdbmodelevents.h"
#include "sharedmacros.h"


class DigestDbController;


// TODO: re-write the following for DigestDbModel!!
/*!
 * \brief REWRITE: The DbAccessor class provides convenient methods for accessing a
 *        QSqlDatabase, along with providing caching to speed-up reads.
 *
 * The methods of this class have been separated from the QSqlDatabase class,
 * as to provide the performance benifits (e.g. preparing QSqlQuery objects with
 * value binding), while at the same time allowing for thread-safe access
 * to the database (but not to a single DbAccessor instance).
 *
 * Hence, each instance of DbAccessor has its own prepared QSqlQuery objects,
 * allowing the different instances to be accessed simultaneously by different
 * threads (in comparison to having the QSqlQuery objects static within
 * QSqlDatabase) in a safe manner.
 *
 * As for caching of gestures (to improve read performance), it should actually
 * be better to have different caches for different callers. Different callers
 * will most likely need to access different gestures and do so in a different
 * way (e.g. read one every so often or read a large list of them, one after the
 * other). For example, the cache used by the gesture browser should clearly not
 * be interfered with by an implementation of an AbstractRecogniserTrainer.
 *
 * Finally, other than in the constructor, it's not worth having a "setter" for
 * the specific QSqlDatabase to use, as \b all of the members variables would
 * need to be re-initialised (QSqlQuery objects and caches). Hence, if you need
 * to change the database, simply re-create the DbAccessor object.
 * 
 */
class DigestDbModel : public QObject, public AbstractModel {

  Q_OBJECT

public:
  DigestDbModel( const QSqlDatabase& database, QObject* parent = 0 );

  DigestDbController* digestDbController() const;

  DECLARE_CLASS_KEY( "digestdbmodel" );

  /*! Returns MvcDigestDb::id(). */
  inline static MvcModuleId_t classModuleId() { return MvcDigestDb::id(); }
  /*! Returns classModuleId(). */
  virtual MvcModuleId_t moduleId() const { return classModuleId(); }

  // TODO: add warning about modifying DB withing sending notification!
  //       -> please read only!!!!
  const QSqlDatabase& database() const { return m_db; }

  // TODO: add setDatabase() ??? - would need a full update of all views!

  static bool createTables( const QSqlDatabase& db );


  QHash<int, QString> fetchClasses() const;
  QString fetchClassLabel( int id ) const;
  bool classLabelExists( const QString& label ) const;
  const IdSet& lastInsertedClassIdSet() const {
    return m_lastInsertedClassIdSet;
  }

  QHash<int, QString> fetchCollections() const;
  QString fetchCollectionLabel( int id ) const;
  bool collectionLabelExists( const QString& label ) const;
  const IdSet& lastInsertedCollectionIdSet() const {
    return m_lastInsertedCollectionIdSet;
  }

  QHash<int, QString> fetchExperimentsBasic() const;
  QString fetchExperimentLabel( int id ) const;
  DExperimentRecord fetchExperiment( int id, bool* ok = 0 ) const;
  const IdSet& lastInsertedExperimentIdSet() const {
    return m_lastInsertedExperimentIdSet;
  }

  int gestureCount() const;
  DGestureRecord fetchGesture( int id, bool* ok = 0 ) const;
  const IdSet& lastInsertedGestureIdSet() const {
    return m_lastInsertedGestureIdSet;
  }

  QHash<int, QString> fetchTrainedRecogsBasic() const;
  QString fetchTrainedRecogLabel( int id ) const;
  DTrainedRecogRecord fetchTrainedRecog( int id, bool* ok = 0 ) const;
  const IdSet& lastInsertedTrainedRecogIdSet() const {
    return m_lastInsertedTrainedRecogIdSet;
  }

  // Note: The default separator is ", " (i.e. with a space after the comma),
  //       as this is designed for display / printing.
  QString classesToString( const QSet<int>& classes,
			   const QString& joinSep = QString(", ") ) const;
  QString collectionsToString( const QSet<int>& collections,
			       const QString& joinSep = QString(", ") ) const;

  static QString idSetToString( const IdSet& idSet );


protected:
  virtual void prepareQueries( const QSqlDatabase& db );

  inline virtual void customEvent( QEvent* e ) {
    MEvent* me = dynamic_cast<MEvent*>(e);
    if ( me != 0 ) dispatchEvent(me);
  }
  virtual void dispatchEvent( MEvent* );

  virtual void changeControllerEvent( MChangeControllerEvent* event );

  virtual void classAddEvent( MClassAddEvent* );
  virtual void classUpdateEvent( MClassUpdateEvent* );
  virtual void classesRemoveEvent( MClassesRemoveEvent* );

  virtual void collectionAddEvent( MCollectionAddEvent* );
  virtual void collectionUpdateEvent( MCollectionUpdateEvent* );
  virtual void collectionsRemoveEvent( MCollectionsRemoveEvent* );

  virtual void experimentAddEvent( MExperimentAddEvent* );
  virtual void experimentUpdateEvent( MExperimentUpdateEvent* );
  virtual void experimentsRemoveEvent( MExperimentsRemoveEvent* );

  virtual void gestureAddEvent( MGestureAddEvent* );
  virtual void gestureUpdateEvent( MGestureUpdateEvent* );
  virtual void gesturesChangeClassesEvent( MGesturesChangeClassesEvent* );
  virtual void gesturesChangeCollectionsEvent( MGesturesChangeCollectionsEvent* );
  virtual void gesturesRemoveEvent( MGesturesRemoveEvent* );

  virtual void trainedRecogAddEvent( MTrainedRecogAddEvent* );
  virtual void trainedRecogUpdateEvent( MTrainedRecogUpdateEvent* );
  virtual void trainedRecogsRemoveEvent( MTrainedRecogsRemoveEvent* );

  virtual bool gesturesChangeCategories( const QString& table,
					 const QString& categoryColumn,
					 const IdSet& idSet,
					 const IdSet& addSet,
					 const IdSet& removeSet );
  virtual bool removeFromTable( const QString& table,
				const QString& idColumn, const IdSet& idSet );

  void handleDbError();


private:
  // Temporary state:
  mutable QSqlDatabase m_db;

  mutable QSqlQuery  m_classFetchAllQuery;
  mutable QSqlQuery  m_classFetchLabelQuery;
  IdSet              m_lastInsertedClassIdSet;

  QSqlQuery m_classMapInsertQuery;
  QSqlQuery m_classMapDeleteQuery;

  mutable QSqlQuery m_collectFetchAllQuery;
  mutable QSqlQuery m_collectFetchLabelQuery;
  IdSet   m_lastInsertedCollectionIdSet;

  QSqlQuery m_collectMapInsertQuery;
  QSqlQuery m_collectMapDeleteQuery;

  QSqlQuery m_experimentInsertQuery;
  QSqlQuery m_experimentUpdateQuery;
  QSqlQuery m_experimentDeleteQuery;
  mutable QSqlQuery m_experimentFetchAllBasicQuery;
  mutable QSqlQuery m_experimentFetchLabelQuery;
  mutable QSqlQuery m_experimentFetchQuery;
  IdSet m_lastInsertedExperimentIdSet;

  QSqlQuery m_gestInsertQuery;
  QSqlQuery m_gestUpdateQuery;
  QSqlQuery m_gestDeleteQuery;
  mutable QSqlQuery m_gestFetchQuery;
  IdSet m_lastInsertedGestureIdSet;

  QSqlQuery m_recogInsertQuery;
  QSqlQuery m_recogUpdateQuery;
  QSqlQuery m_recogDeleteQuery;
  mutable QSqlQuery m_recogFetchAllBasicQuery;
  mutable QSqlQuery m_recogFetchLabelQuery;
  mutable QSqlQuery m_recogFetchQuery;
  IdSet m_lastInsertedTrainedRecogIdSet;

  // Temporary caches:
  QPointer<DigestDbController> c_digestDbController;
};


#endif  // ! DIGESTDBMODEL_H
