/*
 *  abstractdigestdbview.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Fri Aug 12 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "abstractdigestdbview.h"

#include <QCoreApplication>

#include "digestdbcontroller.h"
#include "digestdbmodel.h"


/*!
 * \class AbstractDigestDbView
 *
 * \brief The AbstractDigestDbView class is an abstract base class for Digest
 *        Database (DigestDb) views.
 */


AbstractDigestDbView::AbstractDigestDbView( QObject* objectPtr )
  : AbstractView(objectPtr)
{
}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * DigestDbController* from AbstractController*.
 *
 * See also AbstractView::controller().
 */
DigestDbController* AbstractDigestDbView::digestDbController() const {
  return c_digestDbController;
}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * DigestDbModel* from AbstractModel*.
 *
 * See also AbstractView::model().
 */
DigestDbModel* AbstractDigestDbView::digestDbModel() const {
  return c_digestDbModel;
}


/*!
 * Convenience method.
 *
 * Equivalent to:
 * \code
 * digestDbModel()->database()
 * \endcode
 */
// TODO: cache it?
const QSqlDatabase& AbstractDigestDbView::database() const {
  Q_ASSERT( c_digestDbModel ); // TODO: replace this with an IF ?
  return c_digestDbModel->database();
}


/*!
 * Convenience method.
 *
 * Equivalent to:
 * \code
 * #include <QCoreApplication>
 * #include "MvcDigestDb/digestDbcontroller.h"
 * QCoreApplication::postEvent( digestDbController(), event );
 * \endcode
 */
void AbstractDigestDbView::postDigestDbEvent( CEvent* event ) {
  Q_ASSERT( ! c_digestDbController.isNull() );
  QCoreApplication::postEvent( c_digestDbController, event );
}


void AbstractDigestDbView::dispatchEvent( VEvent* event )
{
  Q_ASSERT( event );

  AbstractView::dispatchEvent( event );

  // NOTE: The *Added cases are a bit hacky, but they do the job and the
  //       architecture didn't require any changes :-) ...

  if ( event->moduleId() != MvcDigestDb::id() ) return; // prevent event type conflicts
  
  Q_ASSERT( c_digestDbModel ); // do this AFTER we check the event type

  QObject* originalSender = static_cast<VDigestDbEvent*>(event)->originalSender();

  switch ( event->type() )
    {
      // CLASS
    case VDigestDbEvent::ClassesAdded:
      {
	VClassesEvent repairedEvent( VDigestDbEvent::ClassesAdded,
				     c_digestDbModel->lastInsertedClassIdSet(),
				     event->sender(), originalSender );
	classesEvent( &repairedEvent );
      }
      break;
    case VDigestDbEvent::ClassesAboutToBeRemoved:
    case VDigestDbEvent::ClassesRemoved:
    case VDigestDbEvent::ClassesUpdated:
      classesEvent( static_cast<VClassesEvent*>(event) );
      break;

      // COLLECTION
    case VDigestDbEvent::CollectionsAdded:
      {
	VCollectionsEvent repairedEvent( VDigestDbEvent::CollectionsAdded,
					 c_digestDbModel
					 ->lastInsertedCollectionIdSet(),
					 event->sender(), originalSender );
	collectionsEvent( &repairedEvent );
      }
      break;
    case VDigestDbEvent::CollectionsAboutToBeRemoved:
    case VDigestDbEvent::CollectionsRemoved:
    case VDigestDbEvent::CollectionsUpdated:
      collectionsEvent( static_cast<VCollectionsEvent*>(event) );
      break;

      // EXPERIMENT
    case VDigestDbEvent::ExperimentsAdded:
      {
	VExperimentsEvent repairedEvent( VDigestDbEvent::ExperimentsAdded,
					 c_digestDbModel
					 ->lastInsertedExperimentIdSet(),
					 event->sender(), originalSender );
	experimentsEvent( &repairedEvent );
      }
      break;
    case VDigestDbEvent::ExperimentsAboutToBeRemoved:
    case VDigestDbEvent::ExperimentsRemoved:
    case VDigestDbEvent::ExperimentsUpdated:
      experimentsEvent( static_cast<VExperimentsEvent*>(event) );
      break;

      // GESTURE
    case VDigestDbEvent::GesturesAdded:
      {
	VGesturesEvent repairedEvent( VDigestDbEvent::GesturesAdded,
				      c_digestDbModel
				      ->lastInsertedGestureIdSet(),
				      event->sender(), originalSender );
	gesturesEvent( &repairedEvent );
      }
      break;
    case VDigestDbEvent::GesturesAboutToBeRemoved:
    case VDigestDbEvent::GesturesChangedClasses:
    case VDigestDbEvent::GesturesChangedCollections:
    case VDigestDbEvent::GesturesRemoved:
    case VDigestDbEvent::GesturesUpdated:
      gesturesEvent( static_cast<VGesturesEvent*>(event) );
      break;

      // TRAINED RECOGNISER
    case VDigestDbEvent::TrainedRecogsAdded:
      {
	VTrainedRecogsEvent repairedEvent( VDigestDbEvent::TrainedRecogsAdded,
					   c_digestDbModel
					   ->lastInsertedTrainedRecogIdSet(),
					   event->sender(), originalSender );
	trainedRecogsEvent( &repairedEvent );
      }
      break;
    case VDigestDbEvent::TrainedRecogsAboutToBeRemoved:
    case VDigestDbEvent::TrainedRecogsRemoved:
    case VDigestDbEvent::TrainedRecogsUpdated:
      trainedRecogsEvent( static_cast<VTrainedRecogsEvent*>(event) );
      break;

    default:
      break;
    }
}


/*!
 * Re-implemented, as to also have the pointer returned by digestDbController()
 * updated.
 *
 * Asserts that the \em event is non-null.
 */
void AbstractDigestDbView::changeControllerEvent( VChangeControllerEvent* event )
{
  Q_ASSERT( event );
  AbstractView::changeControllerEvent( event );
  c_digestDbController
    = ( (event->controller()==0)
	? 0
	: qobject_cast<DigestDbController*>(event->controller()->objectPtr()) );

  changeDigestDbControllerEvent( event );
}


/*!
 * Re-implemented, as to also have the pointer returned by digestDbModel()
 * updated.
 *
 * Asserts that the \em event is non-null.
 */
void AbstractDigestDbView::modelChangedEvent( VModelChangedEvent* event )
{
  Q_ASSERT( event );
  AbstractView::modelChangedEvent( event );
  c_digestDbModel
    = ( (event->model()==0)
	? 0
	: qobject_cast<DigestDbModel*>(event->model()->objectPtr()) );

  digestDbModelChangedEvent( event );
}


void AbstractDigestDbView::resetEvent( VEvent* event )
{
  digestDbResetEvent( event );
}


/*!
 * Conveniance event handler.
 *
 * \em Warning: It will be called in \em addition to changeControllerEvent().
 * It's useful when mixing multiple views together.
 *
 * This implementation does nothing.
 */
void AbstractDigestDbView::changeDigestDbControllerEvent( VChangeControllerEvent* ) {}


/*!
 * Conveniance event handler.
 *
 * \em Warning: It will be called in \em addition to modelChangedEvent().
 * It's useful when mixing multiple views together.
 *
 * This implementation does nothing.
 */
void AbstractDigestDbView::digestDbModelChangedEvent( VModelChangedEvent* ) {}
 


/*!
 * Conveniance event handler.
 *
 * \em Warning: It will be called in \em addition to resetEvent().
 * It's useful when mixing multiple views together.
 *
 * This implementation does nothing.
 */
void AbstractDigestDbView::digestDbResetEvent( VEvent* ) {}
