/*  -*- c++ -*-  (for Emacs)
 *
 *  digestdbviewevents.h
 *  Digest
 * 
 *  Created by Aidan Lane on Fri Aug 12 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef DIGESTDBVIEWEVENTS_H
#define DIGESTDBVIEWEVENTS_H


#include "MvcCore/viewevents.h"
#include "MvcCore/genericcontainers.h"

#include "mvcdigestdb.h"
#include "digestdbcontainers.h"


/*!
 * \brief The VDigestDbEvent class is the base class of all Digest Database
 *        (DigestDb) view events.
 *
 * The \em originalSender can be a very useful property.
 * It was added to this system because when a new record that uses an
 * auto-generated ID is inserted into the database, the caller may need to
 * perform something in response to that that is dependent on that ID
 * (known only once inserted) but only when it itself initiates the original
 * insertion.
 *
 * For example, the GestureBrowser allows the user to add a new "Class" when
 * editing a gesture. In this case, it's intuitive to have the new class added
 * to the gesture's list of classes. However, GestureBrowser can't blindly listen
 * out for VDigestDbEvent::GesturesAdded events and perform this addition, as a
 * new class may be added via something else, which should \b not be added to the
 * gesture.
 * Hence, the GestureBrowser should only update the gesture that's currenly being
 * edited when the originalSender() is itself.
 *
 * Remember that a burst of view events may be delivered at any time and in any
 * order from one or more senders.
 */
class VDigestDbEvent : public VEvent {
public:
  enum Type {
    ClassesAdded                  = VEvent::User,
    ClassesAboutToBeRemoved,    /*! In case view needs more info to de-ref it. */
    ClassesRemoved,             /*! Use this to trigger SQL query re-select.   */
    ClassesUpdated,
    CollectionsAdded,
    CollectionsAboutToBeRemoved,
    CollectionsRemoved,
    CollectionsUpdated,
    ExperimentsAdded,
    ExperimentsAboutToBeRemoved,
    ExperimentsRemoved,
    ExperimentsUpdated,
    GesturesAdded,
    GesturesAboutToBeRemoved,
    GesturesChangedClasses,
    GesturesChangedCollections,
    GesturesRemoved,
    GesturesUpdated,
    TrainedRecogsAdded,
    TrainedRecogsAboutToBeRemoved,
    TrainedRecogsRemoved,
    TrainedRecogsUpdated,
    User                        = VEvent::User + 1000
  };

  VDigestDbEvent( Type type, MvcBase* sender, QObject* originalSender )
    : VEvent((VEvent::Type)type, sender),
      m_originalSender(originalSender) {}

  /*! Returns MvcDigestDb::id(). */
  inline static MvcModuleId_t classModuleId() { return MvcDigestDb::id(); }
  /*! Returns classModuleId(). */
  virtual MvcModuleId_t moduleId() const { return classModuleId(); }

  QObject* originalSender() const { return m_originalSender; }

private:
  QObject* m_originalSender;
};


class VClassesEvent : public VDigestDbEvent, public IdSetContainer {
public:
  VClassesEvent( Type type, const IdSet& idSet,
		 MvcBase* sender, QObject* originalSender )
    : VDigestDbEvent(type, sender, originalSender),
      IdSetContainer(idSet) {}
};

class VCollectionsEvent : public VDigestDbEvent, public IdSetContainer {
public:
  VCollectionsEvent( Type type, const IdSet& idSet,
		     MvcBase* sender, QObject* originalSender )
    : VDigestDbEvent(type, sender, originalSender),
      IdSetContainer(idSet) {}
};

class VExperimentsEvent : public VDigestDbEvent, public IdSetContainer {
public:
  VExperimentsEvent( Type type, const IdSet& idSet,
		     MvcBase* sender, QObject* originalSender )
    : VDigestDbEvent(type, sender, originalSender),
      IdSetContainer(idSet) {}
};

class VGesturesEvent : public VDigestDbEvent, public IdSetContainer {
public:
  VGesturesEvent( Type type, const IdSet& idSet,
		  MvcBase* sender, QObject* originalSender )
    : VDigestDbEvent(type, sender, originalSender),
      IdSetContainer(idSet) {}
};

/*! This event exists for both convenience and efficiency purposes. */
class VGesturesChangedClassesEvent : public VGesturesEvent,
				     public AddSetContainer<int>,
				     public RemoveSetContainer<int> {
public:
  VGesturesChangedClassesEvent( const IdSet& idSet,
				const IdSet& addSet,
				const IdSet& removeSet,
				MvcBase* sender, QObject* originalSender )
    : VGesturesEvent(GesturesChangedClasses, idSet, sender, originalSender),
      AddSetContainer<int>(addSet),
      RemoveSetContainer<int>(removeSet) {}
};

/*! This event exists for both convenience and efficiency purposes. */
class VGesturesChangedCollectionsEvent : public VGesturesEvent,
					 public AddSetContainer<int>,
					 public RemoveSetContainer<int> {
public:
  VGesturesChangedCollectionsEvent( const IdSet& idSet,
				    const IdSet& addSet,
				    const IdSet& removeSet,
				    MvcBase* sender, QObject* originalSender )
    : VGesturesEvent(GesturesChangedCollections, idSet, sender, originalSender),
      AddSetContainer<int>(addSet),
      RemoveSetContainer<int>(removeSet) {}
};

class VTrainedRecogsEvent : public VDigestDbEvent, public IdSetContainer {
public:
  VTrainedRecogsEvent( Type type, const IdSet& idSet,
		       MvcBase* sender, QObject* originalSender )
    : VDigestDbEvent(type, sender, originalSender),
      IdSetContainer(idSet) {}
};


#endif //  ! DIGESTDBVIEWEVENTS_H
