/*  -*- c++ -*-  (for Emacs)
 *
 *  settingscontroller.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Wed Aug 31 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "settingscontroller.h"

#include <QCoreApplication>
#include <QDebug> // TODO: remove me

#include "settingsmodel.h"
#include "abstractsettingsview.h"


/*!
 * Constructs a settings model controller with the given \em parent.
 */
SettingsController::SettingsController( QObject* parent )
  : QObject(parent),
    AbstractController(this)
{
}


/*!
 * Constructs a Digest database controller with the given \em parent and
 * \em model.
 */
SettingsController::SettingsController( SettingsModel* model, QObject* parent )
  : QObject(parent),
    AbstractController(this)
{
  setModel( model );
}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * SettingsModel* from AbstractModel*.
 *
 * See also AbstractController::model().
 */
SettingsModel* SettingsController::settingsModel() const {
  return c_settingsModel;
}


/*!
 * Binds a given object \em obj to a setting \em settingsKey, keeping the
 * \em objProperty in sync (in both directions) with the setting whenever the
 * \em objSignal is emitted  or when the setting is modified by something else.
 *
 * Example:
 * \code
 * bind( usernameEdit, "text", SIGNAL(editingFinished()), "username" );
 * \endcode
 *
 * The object's property will be set to reflect the current setting's value
 * before returning.
 *
 * \b Note: The \em objProperty must be read-write (not read-only).
 *
 * \b Note: For safety and convenience, the object's destroyed() signal is
 *          automatically connected to unbind().
 *
 * Asserts that the \em obj, \em objProperty and \em objSignal are all non-null.
 * Also, it's asserted that the settingsModel() is non-null.
 *
 * See also unbind().
 */
void SettingsController::bind( QObject* obj,
			       const char* objProperty,
			       const char* objSignal,
			       const QString& settingsKey )
{
  Q_ASSERT( obj != 0 );
  Q_ASSERT( objProperty != 0 );
  Q_ASSERT( objSignal != 0 );
  Q_ASSERT( c_settingsModel != 0 );
  BindingInfo info  = { objProperty, objSignal, settingsKey };
  m_bindings.insert( obj, info );
  disconnect( obj, objSignal, this, 0 ); // just in case - about to call setProperty
  obj->setProperty( objProperty, c_settingsModel->value(settingsKey) );
  connect( obj, SIGNAL(destroyed(QObject*)), SLOT(unbind(QObject*)) );
  connect( obj, objSignal, SLOT(onBoundObjectSignalled()) );
}


/*!
 * Completely unbinds object \em obj that was previously bound to one or more
 * setting keys.
 *
 * Asserts that the \em obj is non-null.
 *
 * See also bind().
 */
void SettingsController::unbind( QObject* obj )
{
  Q_ASSERT( obj != 0 );
  disconnect( obj, 0, this, 0 );
  m_bindings.remove( obj );
}


/*!
 * Unbinds object \em obj from the the settings key \em settingsKey.
 *
 * Asserts that the \em obj is non-null.
 *
 * See also bind().
 */
void SettingsController::unbind( QObject* obj, const QString& settingsKey )
{
  Q_ASSERT( obj != 0 );
  QMultiHash<QObject*, BindingInfo>::iterator it = m_bindings.find( obj );
  while ( it != m_bindings.end()
	  && it.key() == obj )
    {
      if ( it.value().settingsKey == settingsKey ) {
	disconnect( obj, it.value().objSignal, this, 0 );
	m_bindings.erase( it );
	break;
      }
      ++it;
    }
}


/*!
 * \b Warning: This must only be called by objects that have been bound using
 *             bind(). This requires that the QObject::sender() is legitimate.
 *             Thus, this asserts that the sender() is registered as a bound
 *             object.
 *
 * \b Note: This is currently inefficient for objects with multiple bindings,
 *          as it will update all of its bindings for each signal
 *          (as we don't know which signal triggered the slot).
 */
void SettingsController::onBoundObjectSignalled()
{
  Q_ASSERT( sender() != 0 );
  Q_ASSERT( m_bindings.contains(sender()) );
  QObject* obj = sender();
  QMultiHash<QObject*, BindingInfo>::const_iterator it = m_bindings.find( obj );
  while ( it != m_bindings.end()
	  && it.key() == obj ) {
    QCoreApplication::postEvent( this, // post -> prevent "queue jumping"
				 new CSettingsChangeValueEvent
				 (it.value().settingsKey,
				  obj->property(it.value().objProperty), this) );
    ++it;
  }
}


/*!
 * The controller's event dispatcher.
 *
 * Asserts that the event is non-null.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 */
void SettingsController::dispatchEvent( CEvent* event )
{
  Q_ASSERT( event != 0 );

  AbstractController::dispatchEvent( event );

  if ( event->moduleId() != MvcSettings::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case CSettingsEvent::SettingsChangeValue:
      settingsChangeValueEvent( static_cast<CSettingsChangeValueEvent*>(event) ); break;
    case CSettingsEvent::SettingsRemoveKey:
      settingsRemoveKeyEvent( static_cast<CSettingsRemoveKeyEvent*>(event) ); break;
    case CSettingsEvent::SettingsRemoveAll:
      settingsRemoveAllEvent( static_cast<CSettingsEvent*>(event) ); break;
    case CSettingsEvent::SettingsSyncAll:
      settingsSyncAllEvent( static_cast<CSettingsEvent*>(event) ); break;
    default:
      break;
    }
}



/* Post an event to our views.
 * Any view object that has the same address as the value of "exclude"
 * will be exluded from the updates.
 * Therefore, set "exclude" to 0 if you don't want to exclude anything.
 * It asserts that the view pointer is non-null.
 * BTW: I know, this is very hacky :-)
 */
#define POST_VIEW_EVENT(EVENT,EXCLUDE)	{AbstractViewSetIterator i(views()); while(i.hasNext()) { AbstractView* v=i.next(); Q_ASSERT(v!=0); Q_ASSERT(v->objectPtr()!=0); if (v->objectPtr() != (EXCLUDE)) QCoreApplication::postEvent( v->objectPtr(), new EVENT ); }}


/*!
 * Re-implemented, as to also have the pointer returned by settingModel()
 * updated.
 *
 * Asserts that the \em event is non-null.
 */
void SettingsController::changeModelEvent( CChangeModelEvent* event )
{
  Q_ASSERT( event != 0 );
  AbstractController::changeModelEvent( event );
  c_settingsModel
    = ( (event->model()==0)
	? 0
	: qobject_cast<SettingsModel*>(event->model()->objectPtr()) );
}


/*!
 * Changes the value of the given setting \em event()->key to \em event->value().
 *
 * If a value for the given key doesn't already exist, then it will be added.
 *
 * Objects that have been bound using bind() will be updated to the new value.
 * Bound objects will be updated after the views().
 *
 * Asserts that the \em event is non-null.
 */
void SettingsController::settingsChangeValueEvent( CSettingsChangeValueEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( c_settingsModel != 0 );

  QString key = event->key();

  QCoreApplication::postEvent( c_settingsModel,
			       new MSettingsChangeValueEvent(key, event->value()) );

  POST_VIEW_EVENT( VSettingsValueChangedEvent(key, event->value(), this),
		   event->updateSender() ? 0 : event->sender() );

  QHashIterator<QObject*, BindingInfo> it( m_bindings );
  while ( it.hasNext() ) {
    it.next();
    QObject* obj = it.key();
    Q_ASSERT( obj != 0 );
    // Note: multiple objs may be bound to a single key - don't break on first find
    if ( it.value().settingsKey == key ) {
      /* IMPORTANT:
       * To prevent infinite cyclic updates, we use disconnect() and connect().
       * The obj->blockSignals(...) method was not used, as it not only prevents
       * us from reciving the signal, but everything connected to the obj, which
       * imposes undesirable limitations.
       */
      disconnect( obj, it.value().objSignal, this, 0 );
      obj->setProperty( it.value().objProperty, event->value() );
      connect( obj, it.value().objSignal, SLOT(onBoundObjectSignalled()) );
    }
  }
}


/*!
 * Removes the setting \em event->key() and any sub-settings of it.
 *
 * Objects that have been bound using bind() will be updated to null.
 *
 * Asserts that the \em event is non-null.
 */
void SettingsController::settingsRemoveKeyEvent( CSettingsRemoveKeyEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( c_settingsModel != 0 );

  QCoreApplication::postEvent( c_settingsModel,
			       new MSettingsRemoveKeyEvent(event->key()) );

  POST_VIEW_EVENT( VSettingsKeyRemovedEvent(event->key(), this),
		   event->updateSender() ? 0 : event->sender() );
}


/*!
 * Removes all entries in the primary location.
 *
 * All views will be sent VEvent(VEvent::Reset).
 *
 * Asserts that the \em event is non-null.
 */
void SettingsController::settingsRemoveAllEvent( CSettingsEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( c_settingsModel != 0 );
  QCoreApplication::postEvent( c_settingsModel,
			       new MSettingsEvent(MSettingsEvent::SettingsRemoveAll) );
  POST_VIEW_EVENT( VEvent(VEvent::Reset, this),
		   event->updateSender() ? 0 : event->sender() );
}


/*!
 * Writes any unsaved changes to permanent storage, and reloads any settings that
 * have been changed in the meantime (possibly by another application).
 *
 * All views will be sent VEvent(VEvent::Reset).
 *
 * Asserts that the \em event is non-null.
 */
void SettingsController::settingsSyncAllEvent( CSettingsEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( c_settingsModel != 0 );
  QCoreApplication::postEvent( c_settingsModel,
			       new MSettingsEvent(MSettingsEvent::SettingsSyncAll) );
  POST_VIEW_EVENT( VEvent(VEvent::Reset, this),
		   event->updateSender() ? 0 : event->sender() );
}
