/*  -*- c++ -*-  (for Emacs)
 *
 *  settingscontroller.h
 *  Digest
 * 
 *  Created by Aidan Lane on Wed Aug 31 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef SETTINGSCONTROLLER_H
#define SETTINGSCONTROLLER_H


#include "MvcCore/abstractcontroller.h"

#include <QByteArray>
#include <QMultiHash>
#include <QPointer>
#include <QString>

#include "settingscontrollerevents.h"

class SettingsModel;


/*!
 * \brief The SettingsController class is responsible for controlling the
 *        communication between a SettingsModel model its views.
 *
 * Note that the model's pointer is stored internally as a "guarded pointer".
 * Hence, if the model is deleted, our reference to it will be automatically
 * set back to 0.
 *
 * Warning! We always assert that the model is non-null. Hence, you can't
 * have an instance of a SettingsController without a SettingsModel.
 * This is fine in our case and means that we don't need to constantly check
 * if the diagram is null in the release / deployment version (when assertions
 * are stripped).
 */
class SettingsController : public QObject,
			   public AbstractController {

  Q_OBJECT

public:
  SettingsController( QObject* parent = 0 );
  SettingsController( SettingsModel* model, QObject* parent = 0 );

  SettingsModel* settingsModel() const;

  /*! Returns MvcSettings::id(). */
  inline static MvcModuleId_t classModuleId() { return MvcSettings::id(); }
  /*! Returns classModuleId(). */
  virtual MvcModuleId_t moduleId() const { return classModuleId(); }

  void bind( QObject* obj, const char* objProperty,
	     const char* objSignal, const QString& settingsKey );


private slots:
  void unbind( QObject* obj );
  void unbind( QObject* obj, const QString& settingsKey );
  void onBoundObjectSignalled();


protected:
  inline virtual void customEvent( QEvent* e ) {
    CEvent* ce = dynamic_cast<CEvent*>(e); // slow :-(
    if ( ce != 0 ) dispatchEvent(ce);
  }
  virtual void dispatchEvent( CEvent* );

  virtual void changeModelEvent( CChangeModelEvent* );

  virtual void settingsChangeValueEvent( CSettingsChangeValueEvent* );
  virtual void settingsRemoveKeyEvent( CSettingsRemoveKeyEvent* );
  virtual void settingsRemoveAllEvent( CSettingsEvent* );
  virtual void settingsSyncAllEvent( CSettingsEvent* );


private:
  struct BindingInfo {
    QByteArray  objProperty; // safer than "const char*" for storage
    QByteArray  objSignal;   // ditto.
    QString     settingsKey;
  };

  // Temporary state:
  QMultiHash<QObject*, BindingInfo> m_bindings; // multihash-> * bindings per obj

  // Temporary caches:
  QPointer<SettingsModel> c_settingsModel;
};


#endif // ! SETTINGSCONTROLLER_H
