/*  -*- c++ -*-  (for Emacs)
 *
 *  settingsmodel.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Wed Aug 31 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "settingsmodel.h"

#include <QCoreApplication>

#include "settingscontroller.h"


/*!
 * Constructs a settings model with the given \em parent.
 */
SettingsModel::SettingsModel( const QString& organization,
			      const QString& application,
			      QObject* parent )
  : QObject(parent),
    AbstractModel(this),
    m_settings(new QSettings(organization, application, this))
{}


/*!
 * Constructs a settings model with the given \em parent.
 */
SettingsModel::SettingsModel( QSettings::Scope scope,
			      const QString& organization,
			      const QString& application,
			      QObject* parent )
  : QObject(parent),
    AbstractModel(this),
    m_settings(new QSettings(scope, organization, application, this))
{}


/*!
 * Constructs a settings model with the given \em parent.
 */
SettingsModel::SettingsModel( QSettings::Format format,
			      QSettings::Scope scope,
			      const QString& organization,
			      const QString& application,
			      QObject* parent )
  : QObject(parent),
    AbstractModel(this),
    m_settings(new QSettings(format, scope, organization, application, this))
{}


/*!
 * Constructs a settings model with the given \em parent.
 */
SettingsModel::SettingsModel( const QString& fileName,
			      QSettings::Format format,
			      QObject* parent )
  : QObject(parent),
    AbstractModel(this),
    m_settings(new QSettings(fileName, format, this))
{}


/*!
 * Constructs a settings model with the given \em parent.
 */
SettingsModel::SettingsModel( QObject* parent )
  : QObject(parent),
    AbstractModel(this),
    m_settings(new QSettings(this))
{}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * SettingsController* from AbstractController*.
 *
 * See also AbstractModel::controller().
 */
SettingsController* SettingsModel::settingsController() const {
  return c_settingsController;
}


/*! Returns a non-const version of QSettings for use by sub-classes. */
QSettings* SettingsModel::settings() const {
  return m_settings;
}


/*! Returns a const version of QSettings for reading from. */
const QSettings* SettingsModel::constSettings() const {
  return m_settings;
}


/*!
 * Returns the value for setting \em key.
 * If the setting doesn't exist, returns \em defaultValue.
 *
 * If no default value is specified, a default QVariant is returned.
 */
QVariant SettingsModel::value( const QString& key,
			       const QVariant& defaultValue ) const {
  Q_ASSERT( m_settings != 0 );
  return m_settings->value( key, defaultValue );
}


/*!
 * The model's event dispatcher.
 *
 * Asserts that the event is non-null.
 *
 * \b Warning! It is assumed that the events have their correct type set
 *             (as it uses static casts).
 */
void SettingsModel::dispatchEvent( MEvent* event )
{
  Q_ASSERT( event != 0 );

  AbstractModel::dispatchEvent( event );

  if ( event->moduleId() != MvcSettings::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case MSettingsEvent::SettingsChangeValue:
      settingsChangeValueEvent( static_cast<MSettingsChangeValueEvent*>(event) ); break;
    case MSettingsEvent::SettingsRemoveKey:
      settingsRemoveKeyEvent( static_cast<MSettingsRemoveKeyEvent*>(event) ); break;
    case MSettingsEvent::SettingsRemoveAll:
      settingsRemoveAllEvent( static_cast<MSettingsEvent*>(event) ); break;
    case MSettingsEvent::SettingsSyncAll:
      settingsSyncAllEvent( static_cast<MSettingsEvent*>(event) ); break;
    default:
      break;
    }
}


/* Post an event to our views.
 * Any view object that has the same address as the value of "exclude"
 * will be exluded from the updates.
 * Therefore, set "exclude" to 0 if you don't want to exclude anything.
 * It asserts that the view pointer is non-null.
 * BTW: I know, this is very hacky :-)
 */
#define POST_VIEW_EVENT(EVENT,EXCLUDE)	{AbstractViewSetIterator i(views()); while(i.hasNext()) { AbstractView* v=i.next(); Q_ASSERT(v!=0); Q_ASSERT(v->objectPtr()!=0); if (v->objectPtr() != (EXCLUDE)) QCoreApplication::postEvent( v->objectPtr(), new EVENT ); }}


/*!
 * Re-implemented, as to also have the pointer returned by settingsController()
 * updated.
 *
 * Asserts that the \em event is non-null.
 */
void SettingsModel::changeControllerEvent( MChangeControllerEvent* event )
{
  Q_ASSERT( event != 0 );
  AbstractModel::changeControllerEvent( event );
  c_settingsController
    = ( (event->controller()==0)
	? 0
	: qobject_cast<SettingsController*>(event->controller()->objectPtr()) );
}


/*!
 * Changes the value of the given setting \em event()->key to \em event->value().
 *
 * If a value for the given key doesn't already exist, then it will be added.
 *
 * Asserts that the \em event is non-null.
 */
void SettingsModel::settingsChangeValueEvent( MSettingsChangeValueEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( m_settings != 0 );
  m_settings->setValue( event->key(), event->value() );
}


/*!
 * Removes the setting \em event->key() and any sub-settings of it.
 *
 * Asserts that the \em event is non-null.
 */
void SettingsModel::settingsRemoveKeyEvent( MSettingsRemoveKeyEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( m_settings != 0 );
  m_settings->remove( event->key() );
}


/*!
 * Removes all entries in the primary location.
 */
void SettingsModel::settingsRemoveAllEvent( MSettingsEvent* /*event*/ )
{
  Q_ASSERT( m_settings != 0 );
  m_settings->clear();
}


/*!
 * Writes any unsaved changes to permanent storage, and reloads any settings that
 * have been changed in the meantime (possibly by another application).
 *
 * Asserts that the \em event is non-null.
 */
void SettingsModel::settingsSyncAllEvent( MSettingsEvent* /*event*/ )
{
  Q_ASSERT( m_settings != 0 );
  m_settings->sync();
}
