/*
 *  abstractsettingsview.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Wed Aug 31 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "abstractsettingsview.h"

#include <QCoreApplication>

#include "settingsmodel.h"
#include "settingscontroller.h"


AbstractSettingsView::AbstractSettingsView( QObject* objectPtr )
  : AbstractView(objectPtr)
{
}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * SettingsController* from AbstractController*.
 *
 * See also AbstractView::controller().
 */
SettingsController* AbstractSettingsView::settingsController() const {
  return c_settingsController;
}


/*!
 * Convenience method.
 *
 * Returns a cached guarded pointer that has been dynamically cast to
 * SettingsModel* from AbstractModel*.
 *
 * See also AbstractView::model().
 */
SettingsModel* AbstractSettingsView::settingsModel() const {
  return c_settingsModel;
}


/*!
 * Convenience method.
 *
 * Equivalent to:
 * \code
 * #include <QCoreApplication>
 * #include "MvcSettings/settingscontroller.h"
 * QCoreApplication::postEvent( settingsController(), event );
 * \endcode
 */
void AbstractSettingsView::postSettingsEvent( CEvent* event ) {
  Q_ASSERT( ! c_settingsController.isNull() );
  QCoreApplication::postEvent( c_settingsController, event );
}


void AbstractSettingsView::dispatchEvent( VEvent* event )
{
  Q_ASSERT( event != 0 );

  AbstractView::dispatchEvent( event );

  if ( event->moduleId() != MvcSettings::id() ) return; // prevent event type conflicts

  switch ( event->type() )
    {
    case VSettingsEvent::SettingsValueChanged:
      settingsValueChangedEvent( static_cast<VSettingsValueChangedEvent*>(event) );
      break;
    case VSettingsEvent::SettingsKeyRemoved:
      settingsKeyRemovedEvent( static_cast<VSettingsKeyRemovedEvent*>(event) );
      break;
    default:
      break;
    }
}


/*!
 * Re-implemented, as to also have the pointer returned by settingsController()
 * updated.
 *
 * Asserts that the \em event is non-null.
 */
void AbstractSettingsView::changeControllerEvent( VChangeControllerEvent* event )
{
  Q_ASSERT( event != 0 );
  AbstractView::changeControllerEvent( event );
  c_settingsController
    = ( (event->controller()==0)
	? 0
	: qobject_cast<SettingsController*>(event->controller()->objectPtr()) );

  changeSettingsControllerEvent( event );
}


/*!
 * Re-implemented, as to also have the pointer returned by settingsModel()
 * updated.
 *
 * Asserts that the \em event is non-null.
 */
void AbstractSettingsView::modelChangedEvent( VModelChangedEvent* event )
{
  Q_ASSERT( event != 0 );
  AbstractView::modelChangedEvent( event );
  c_settingsModel
    = ( (event->model()==0)
	? 0
	: qobject_cast<SettingsModel*>(event->model()->objectPtr()) );

  settingsModelChangedEvent( event );
}


void AbstractSettingsView::resetEvent( VEvent* event )
{
  settingsResetEvent( event );
}
