/*  -*- c++ -*-  (for Emacs)
 *
 *  recogniserbrowser.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Tue Jul 21 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "recogniserbrowser.h"

#include <QItemSelectionModel>
#include <QSqlRecord>
#include <QSqlQueryModel>
#include <QStandardItemModel>

#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/digestdbcontrollerevents.h"


/*!
 * \class RecogniserBrowser
 *
 * \brief The RecogniserBrowser class provides a graphical interface for browsing
 *        through, adding and removing trained recogniser records.
 */


RecogniserBrowser::RecogniserBrowser( AbstractController* controller,
				      JavaVM* jvm,
				      QWidget* parent, Qt::WindowFlags flags )
  : GuiDbComponentDialog(controller, parent, flags),
    m_jvm(jvm),
    m_queryModel(0),
    m_selectionModel(0)
{
  /*
   * WARNING: The use of the query model is delayed until we receive VEvent::Reset.
   */
  m_dummyModel = new QStandardItemModel( this );
  ui.setupUi( this );
}


/*!
 * Re-syncs the GUI with the database if digestDbModel() is non-null, as it may
 * have changed for some unknown reason without sending update events to us.
 *
 * Users tend to expect a refresh if a window is hidden and then shown again.
 *
 * It then then passes event onto GuiDbComponentDialog.
 */
void RecogniserBrowser::showEvent( QShowEvent* event )
{
  if ( digestDbModel() ) refreshModel();
  GuiDbComponentDialog::showEvent( event );
}


void RecogniserBrowser::resetEvent( VEvent* )
{
  /*
   * Clean-up
   */
  Q_ASSERT( ui.treeView );
  //ui.treeView->setSelectionModel( 0 );
  ui.treeView->setModel( m_dummyModel ); // can't be null :-(
  ui.treeView->reset();

  delete m_selectionModel;
  delete m_queryModel;
  m_selectionModel = 0;
  m_queryModel = 0;


  /*
   * (Re)build
   */
  if ( digestDbModel() )
    {
      m_queryModel = new QSqlQueryModel( this );
      
      m_selectionModel = new QItemSelectionModel( m_queryModel );
      connect( m_selectionModel,
	       SIGNAL(selectionChanged(const QItemSelection&, const QItemSelection&)),
	       SLOT(onSelectionChanged()) );
      
      Q_ASSERT( ui.treeView );
      ui.treeView->setModel( m_queryModel );
      ui.treeView->setSelectionModel( m_selectionModel );

      refreshModel();
    }
}



void RecogniserBrowser::trainedRecogsEvent( VTrainedRecogsEvent* event )
{
  Q_ASSERT( event );

  /* Ensure that we're in sync with the database.
   * Ignore TrainedRecogAboutToBeRemoved, as refreshModel() only needs to be
   * called when the database has actually changed.
   * Hence, we wait for TrainedRemoved.
   */
  if ( (VDigestDbEvent::Type)event->type()
       != VDigestDbEvent::TrainedRecogsAboutToBeRemoved )
    refreshModel();
}


void RecogniserBrowser::refreshModel()
{
  if ( digestDbModel() )
    {
      // TODO: pre-prepare the query!
      // Note: we need the ID for internal use, but we hide it from the view
      Q_ASSERT( m_queryModel );
      m_queryModel->setQuery( "SELECT id, label, recogniserKey, date "
			      "FROM TrainedRecogniser" );
      
      // Note: Column 0 - the ID is for internal use
      m_queryModel->setHeaderData( 0, Qt::Horizontal, tr("ID") );
      m_queryModel->setHeaderData( 1, Qt::Horizontal, tr("Label") );
      m_queryModel->setHeaderData( 2, Qt::Horizontal, tr("Type") );
      m_queryModel->setHeaderData( 3, Qt::Horizontal, tr("Date") );
      
      // Note: We get strange behaviour if we don't perform the following
      Q_ASSERT( ui.treeView );
      //FIXME: source of many probs: ui.treeView->setColumnHidden( 0, true ); // the ID is for internal use
      ui.treeView->setSelectionModel( m_selectionModel );
      
      onSelectionChanged(); // update selection dependants (info box & remove button)
    }
}


void RecogniserBrowser::on_trainNewButton_clicked()
{
  emit request( "new TrainingAssistant" );
}


void RecogniserBrowser::on_removeButton_clicked()
{
  // TODO: ask the user if they are REALLY sure - because it may take some
  //       time to recreate a trained recogniser!
  // TODO: offer to delete the data file!

  // Delete the selected trained recognisers
  Q_ASSERT( m_selectionModel );
  Q_ASSERT( m_queryModel );
  IdSet idSet;
  foreach ( const QModelIndex& index, m_selectionModel->selectedIndexes() ) {
    bool intOk = false;
    if ( index.column() != 0 ) continue; // only process 1st column of each row
    idSet += m_queryModel->record(index.row()).value(0).toInt(&intOk);
    Q_ASSERT( intOk );
  }
  if ( !idSet.isEmpty() )
    postControllerEvent( new CTrainedRecogsRemoveEvent(idSet, this) );
}


void RecogniserBrowser::on_detailsHtmlWidget_anchorClicked( const QUrl& /*link*/ )
{
  // TODO: don't use the SQL query, use "IDs"
  // TODO: set connection to "local"
  // TODO: handle "multiple selection"
  // TODO: cleanup
  Q_ASSERT( m_selectionModel );
  if ( ! m_selectionModel->selectedIndexes().isEmpty() )
    {
      Q_ASSERT( m_queryModel );
      QString idStr = m_queryModel
	->record(m_selectionModel->selectedIndexes().first().row()).value(0).toString();
      Q_ASSERT( digestDbModel() );
      // Don't use fetchTrainedRecog() - it has to split the trainingSet into a
      // set of integers, just for us to build a string again!
      // TODO: check for SQL errors
      QSqlQuery q( "SELECT trainingSet FROM TrainedRecogniser WHERE id=" + idStr,
		   digestDbModel()->database() );
      if ( q.next() )
	emit request( "show GestureBrowser searchtype=\"sqlwhere\" : "
		      "search=\"ID in (" + q.value(0).toString() + ")" );
    }
  updateDetailsText(); // restore the page & prevent it from trying to open the "link"
}


void RecogniserBrowser::onSelectionChanged()
{
  Q_ASSERT( ui.removeButton );
  Q_ASSERT( m_selectionModel );
  ui.removeButton->setEnabled( ! m_selectionModel->selectedIndexes().isEmpty() );

  updateDetailsText();
}


void RecogniserBrowser::updateDetailsText()
{
  QString detailsText;
  bool intOk = false;

  // TODO: if multiple items are selected, then print "multiple selection" info
  Q_ASSERT( m_selectionModel );
  if ( ! m_selectionModel->selectedIndexes().isEmpty() )
    {
      Q_ASSERT( m_queryModel );
      int id = m_queryModel
	->record(m_selectionModel->selectedIndexes().first().row()).value(0).toInt(&intOk);
      Q_ASSERT( intOk );
      Q_ASSERT( digestDbModel() );
      DTrainedRecogRecord record = digestDbModel()->fetchTrainedRecog( id );
      detailsText = trainedRecogRecordAsHtml( record );
    }
  else
    {
      detailsText = "<html><body><table width=\"100%\">"
	"<tr><td align=\"center\"><font color=\"#AAAAAA\" size=\"+1\"><b>"
	"<br><br>Nothing Selected"
	"</b></font></td></tr>"
	"</table></body></html>";
    }

  Q_ASSERT( ui.detailsHtmlWidget );
  ui.detailsHtmlWidget->setHtml( detailsText );
}


QString RecogniserBrowser::trainedRecogRecordAsHtml( const DTrainedRecogRecord& record ) const
{
  QString str =
    "<html>"
    "<body>"
    "<table cellspacing=\"6\">"
    "<tr><th align=\"right\"><font color=\"#888888\">ID</font></th><td>"
    + QString::number(record.id) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Label</font></th><td>"
    + (record.label.isEmpty() ? tr("(none)") : record.label) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Type</font></th><td>"
    + (record.recogniserKey.isEmpty() ? tr("(none)") : record.recogniserKey) +
    "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">External</font></th><td>"
    + (record.external ? tr("Yes") : tr("No")) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Date</font></th><td>"
    + (record.date.isValid() ? record.date.toString(Qt::LocalDate) : tr("(not set)")) +
    "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Notes</font></th><td>"
    + (record.notes.isEmpty() ? tr("(none)") : record.notes) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Data File</font></th><td>"
    + (record.modelFile.isEmpty() ? tr("(none)") : record.modelFile) +
    "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Features</font></th><td>";
  if ( record.orderedFeatures.isEmpty() )
    str += tr("(none)");
  else {
    QStringList keyList;
    foreach ( const QByteArray& key, record.orderedFeatures )
      keyList += key;
    str += keyList.join( ", " );
  }

  str +=
    "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Training Set</font></th><td>"
    + ( record.trainingSet.isEmpty()
	? tr("(none)")
	: ("<a href=\"" + QString::number(record.id) + "\">"
	   + QString::number(record.trainingSet.size())
	   + " " + tr("gestures") + "</a>") )
    + "</td></tr>"
    "</table>"
    "</body>"
    "</html>";

  return str;
}
