/*  -*- c++ -*-  (for Emacs)
 *
 *  recognisertestpad.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Tue Jul 12 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "recognisertestpad.h"

#include <QListWidgetItem>
#include <QSqlQuery>

#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/digestdbcontroller.h"
#include "MvcSettings/settingscontroller.h"
#include "GestureRecognition/featurefactory.h"
#include "GestureRecognition/recogniserfactory.h"

#include "digestsettings.h"


/*!
 * \class RecogniserTestPad
 *
 * \brief The RecogniserTestPad class provides a graphical interface for testing
 *        recognisers.
 */


RecogniserTestPad::RecogniserTestPad( DigestDbController* digestDbController,
				      SettingsController* settingsController,
				      JavaVM* jvm,
				      QWidget* parent, Qt::WindowFlags flags )
  : GuiDbComponentDialog(digestDbController, parent, flags),
    m_jvm(jvm)
{
  /*
    WARNING: The use of the model is delayed until we receive VEvent::Reset.
  */
  
  m_ui.setupUi( this );

  Q_ASSERT( m_ui.strokesEditor );
  connect( m_ui.strokesEditor, SIGNAL(strokingStarted()),
	   m_ui.resultsHtmlWidget, SLOT(clear()) );
  connect( m_ui.strokesEditor,
	   SIGNAL(strokingFinished(const StrokeList&)),
	   SLOT(recognise(const StrokeList&)) );

  if ( settingsController ) {
    settingsController->bind( m_ui.strokesEditor, "recordHiRes",
			      SIGNAL(recordHiResToggled(bool)),
			      DigestSettings::inputRecordHiResKey );
    settingsController->bind( m_ui.strokesEditor, "multiStrokeTimeout",
			      SIGNAL(multiStrokeTimeoutChanged(double)),
			      DigestSettings::inputMultiStrokeTimeoutKey );
  }

  Q_ASSERT( m_ui.recogListWidget );
  m_ui.recogListWidget->setHeaderText( tr("Trained Recognisers") );
  connect( m_ui.recogListWidget,
	   SIGNAL(itemSelectionChanged()), SLOT(updateInfoText()) );

  Q_ASSERT( m_ui.trainedRecogInfoWidget );
  m_ui.trainedRecogInfoWidget->setHeaderText( tr("Information") );

  updateInfoText();
}


/*!
 * Re-syncs the GUI with the database if digestDbModel() is non-null, as it may
 * have changed for some unknown reason without sending update events to us.
 *
 * Users tend to expect a refresh if a window is hidden and then shown again.
 *
 * It then then passes event onto GuiDbComponentDialog.
 */
void RecogniserTestPad::showEvent( QShowEvent* event )
{
  if ( digestDbModel() ) refreshTrainedRecogList();
  GuiDbComponentDialog::showEvent( event );
}


// TODO: finish me!
void RecogniserTestPad::resetEvent( VEvent* )
{
  /*
   * Clean-up
   */
  Q_ASSERT( m_ui.recogListWidget );
  m_ui.recogListWidget->clear();

  qDeleteAll( m_recognisers.keys() );
  m_recognisers.clear();

  updateInfoText();


  /*
   * (Re)build
   */
  if ( digestDbModel() )
    {
      refreshTrainedRecogList();
    }
}


void RecogniserTestPad::trainedRecogsEvent( VTrainedRecogsEvent* )
{
  refreshTrainedRecogList(); // ensure that we're in sync with the database
}


void RecogniserTestPad::on_recogniseButton_clicked()
{
  Q_ASSERT( m_ui.strokesEditor );
  recognise( m_ui.strokesEditor->strokes() );
}


void RecogniserTestPad::on_enableAllButton_clicked() {
  setAllRecognisersCheckState( Qt::Checked );
}


void RecogniserTestPad::on_enableNoneButton_clicked() {
  setAllRecognisersCheckState( Qt::Unchecked );
}


void RecogniserTestPad::on_clearButton_clicked()
{
  Q_ASSERT( m_ui.strokesEditor );
  m_ui.strokesEditor->setStrokes( StrokeList() ); // also stops timer for us

  Q_ASSERT( m_ui.resultsHtmlWidget );
  m_ui.resultsHtmlWidget->clear();
}


void RecogniserTestPad::recognise( const StrokeList& strokes )
{
  Q_ASSERT( m_ui.strokesEditor );
  m_ui.strokesEditor->stopMultiStrokeTimoutTimer(); // in case we're called first

  // results: recogniser label -> class probabilities
  // Note: using QMultiMap allows for duplicate keys and also keeps them sorted
  QMultiMap<QString, ClassProbabilities> results;

  // TODO: don't rebuild all of the recognisers, just add or remove as needed!
  initRecognisers();

  // Gather all of the results
  QHashIterator<AbstractRecogniser*, QString> it( m_recognisers );
  while ( it.hasNext() ) {
    it.next();
    Q_ASSERT( it.key() );
    results.insert( it.value(), it.key()->classify(strokes) );
  }

  // Update resultsHtmlWidget
  // Important: We're not using a QTableWidget, as that does not allow for copying
  //            the results into another program (via copy&paste or drag-copy).
  Q_ASSERT( m_ui.resultsHtmlWidget );
  m_ui.resultsHtmlWidget->setHtml( recogResultsAsHtml(results) );
}


void RecogniserTestPad::updateInfoText()
{
  QString infoText;
  bool intOk = false;

  // Note: there should only be at most one selected item, so we only look at the first
  Q_ASSERT( m_ui.recogListWidget );
  if ( ! m_ui.recogListWidget->selectedItems().isEmpty() )
    {
      int id = m_ui.recogListWidget
	->selectedItems().first()->data(Qt::UserRole).toInt(&intOk);
      Q_ASSERT( intOk );
      Q_ASSERT( digestDbModel() );
      DTrainedRecogRecord record = digestDbModel()->fetchTrainedRecog( id );
      infoText = trainedRecogRecordAsHtml( record );
    }
  else
    {
      infoText = ( "<html><body><table width=\"100%\">"
		   "<tr><td align=\"center\"><font color=\"#AAAAAA\" size=\"+1\"><b>"
		   "<br><br>Nothing<br>Selected"
		   "</b></font></td></tr>"
		   "</table></body></html>" );
    }

  Q_ASSERT( m_ui.trainedRecogInfoWidget );
  m_ui.trainedRecogInfoWidget->setHtml( infoText );
}


void RecogniserTestPad::refreshTrainedRecogList()
{
  // TODO: remember which where checked between refreshes!!!!!!

  Q_ASSERT( m_ui.recogListWidget );

  // Clear and re-build
  m_ui.recogListWidget->clear();
  // TODO: pre-prepare the QSqlQuery
  QSqlQuery q( "SELECT id,label,ready FROM TrainedRecogniser", database() );
  while ( q.next() )
    {
      QListWidgetItem* item = new QListWidgetItem( m_ui.recogListWidget );
      item->setData( Qt::DisplayRole, q.value(1) ); // label
      item->setData( Qt::UserRole, q.value(0) );    // ID - internal use
      item->setFlags( Qt::ItemIsSelectable
		      | Qt::ItemIsUserCheckable
		      | (q.value(2).toBool() // only enabled if it's ready!
			 ? Qt::ItemIsEnabled
			 : (Qt::ItemFlag)0) ); // yes, this is repeated :-)
      // Note: unlike the "Experiment Assistant", you usually use all m_recognisers.
      item->setCheckState( Qt::Checked ); // need to set default for it to work
    }

#if 0 // -- now that they're all enabled by default, the following isn't needed
  // "Check" the first item that is enabled
  for ( int i=0; i < m_ui.recogListWidget->count(); ++i ) {
    QListWidgetItem* item = m_ui.recogListWidget->item(i);
    Q_ASSERT( item );
    if ( item->flags() & Qt::ItemIsEnabled ) {
      item->setCheckState( Qt::Checked );
      break;
    }
  }
#endif

  // TODO: remove this hack!
  initRecognisers();
}


// TODO: reNAME me!
void RecogniserTestPad::initRecognisers()
{
  // Destroy any existing recognisers!
  qDeleteAll( m_recognisers.keys() );
  m_recognisers.clear();

  // Create the recognisers
  Q_ASSERT( m_ui.recogListWidget );
  int count = m_ui.recogListWidget->count();
  for ( int i=0; i < count; ++i )
    {
      const QListWidgetItem* item = m_ui.recogListWidget->item(i);
      Q_ASSERT( item );
      if ( item->checkState() == Qt::Checked )
	{
	  bool intOk = false;

	  const int id = item->data(Qt::UserRole).toInt(&intOk);
	  Q_ASSERT( intOk );

	  Q_ASSERT( digestDbModel() );
	  DTrainedRecogRecord rec = digestDbModel()->fetchTrainedRecog( id );

	  AbstractRecogniser* recogniser
	    = RecogniserFactory::create( rec.recogniserKey, m_jvm,
					 digestDbModel(), this );
	  if ( recogniser == 0 ) { // TODO: handle me properly!!!! - user may have got it wrong!
	      qDebug("ERROR: Invalid recogniser requested!");
	      // TODO: update the settings?
	      continue;
	    }

	  if ( ! recogniser->loadRecord(rec) ) {
	    qDebug( "recogniser loadRecord failed!" ); // TODO: handle this properly!
	    continue;
	  }

	  m_recognisers.insert( recogniser, rec.label );
	}
    }
}


void RecogniserTestPad::setAllRecognisersCheckState( Qt::CheckState state )
{
  Q_ASSERT( m_ui.recogListWidget );
  int count = m_ui.recogListWidget->count();
  for ( int i=0; i < count; ++i ) {
    Q_ASSERT( m_ui.recogListWidget->item(i) );
    m_ui.recogListWidget->item(i)->setCheckState( state );
  }
}


QString RecogniserTestPad::trainedRecogRecordAsHtml( const DTrainedRecogRecord& record ) const
{
  // Note: There's no need to show the label, as the user must have clicked on a
  //       a item label in the recogListWidget to get this info.
  return
    ( "<html>"
      "<body>"
      "<table cellspacing=\"6\">"
      "<tr><th align=\"right\"><font color=\"#888888\">ID</font></th><td>"
      + QString::number(record.id) + "</td></tr>"
      "<tr><th align=\"right\"><font color=\"#888888\">Type</font></th><td>"
      + (record.recogniserKey.isEmpty() ? tr("(none)") : record.recogniserKey) +
      "</td></tr>"
      "<tr><th align=\"right\"><font color=\"#888888\">External</font></th><td>"
      + (record.external ? tr("Yes") : tr("No")) + "</td></tr>"
      "<tr><th align=\"right\"><font color=\"#888888\">Date</font></th><td>"
      + (record.date.isValid() ? record.date.toString(Qt::LocalDate) : tr("(not set)")) +
      "</td></tr>"
      "<tr><th align=\"right\"><font color=\"#888888\">Notes</font></th><td>"
      + (record.notes.isEmpty() ? tr("(none)") : record.notes) + "</td></tr>"
      "</table>"
      "</body>"
      "</html>" );
}


// Important: We're not using a QTableWidget, as that does not allow for copying
//            the results into another program (via copy&paste or drag-copy).
// results: recogniser label -> class probabilities
// Note: using QMultiMap allows for duplicate keys and also keeps them sorted
QString RecogniserTestPad
::recogResultsAsHtml( const QMultiMap<QString,ClassProbabilities>& results ) const
{
  Q_ASSERT( digestDbModel() );
  QHash<int, QString> classToName = digestDbModel()->fetchClasses();

  QSet<int> usedClassesSet;
  usedClassesSet.reserve( classToName.size() );
  QMapIterator<QString, ClassProbabilities> ri( results );
  while ( ri.hasNext() ) {
    ri.next();
    usedClassesSet += ri.value().keys().toSet();
  }

  if ( usedClassesSet.isEmpty() )
    return ( "<html><body><table width=\"100%\">"
	     "<tr><td align=\"center\"><font color=\"#AAAAAA\"><b>"
	     "<br><br>No Matches Found"
	     "</b></font></td></tr>"
	     "</table></body></html>" );

  QList<int> sortedUsedClassesList = usedClassesSet.toList();
  qSort( sortedUsedClassesList );


  QString str =
    "<html>"
    "<body>"
    "<table cellspacing=\"6\">";

  // Generate header row -> shape class names
  str += "<tr><th></th>"; // start row and skip first column (used for recogniser label)
  foreach ( int classId, sortedUsedClassesList )
    str += ( "<th align=\"center\"><font color=\"#888888\">"
	     + classToName.value(classId) + "</font></th>" );
  str += "</tr>";

  // Generate a confidence result set row for each recogniser
  QMapIterator<QString, ClassProbabilities> si( results );
  while ( si.hasNext() )
    {
      si.next();
      const ClassProbabilities& probs = si.value();

      // Note: Don't store the index, only the value, as two or more classes may
      //       share the maximum probability - need to identify both of them.
      ClassProbabilityT maxProb = 0.0;
      if ( !probs.isEmpty() ) maxProb = * probs.constBegin(); // init
      foreach ( ClassProbabilityT p, probs.values() )
	if ( p > maxProb ) maxProb = p;
     
      str += ( "<tr><th align=\"right\"><font color=\"#888888\">"
	       + si.key() + "</font></th>" );

      // TODO: make precision user adjustable ?
      //       - esp. as as 2dp, two values may SEEM identical, but only one be the max
      // Note: we don't print "%" mark, as it becomes a bit overwhelming!
      foreach ( int classId, sortedUsedClassesList ) {  // use consistent ordering
	ClassProbabilityT p = probs.value( classId );
	str += ( QString("<td>")
		 + (p==maxProb ? "<b>" : "")
		 + QString::number(p * 100.0, 'f', 2)
		 + (p==maxProb ? "</b>" : "")
		 + QString("</td>") );
      }

      str += "</tr>";
    }

  str +=
    "</table>"
    "</html>"
    "</body>";

  return str;
}
