/*  -*- c++ -*-  (for Emacs)
 *
 *  sqlbrowser.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jun 30 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  Based on the Qt Toolkit 4.01 SqlBrowser demonstration application.
 *  Copyright (C) 1992-2005 Trolltech AS. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

// TODO: post events on a query that isn't a selection -> sync other components
// TODO: post events when a table cell is modified -> sync other components

#include "sqlbrowser.h"

#include <QFile>
#include <QFileDialog>
#include <QFileInfo>
#include <QMessageBox>
#include <QSqlError>
#include <QSqlField>
#include <QSqlQueryModel>
#include <QSqlTableModel>
#include <QRegExp>
#include <QTextStream>


/*!
 * \class SqlBrowser
 *
 * \brief The SqlBrowser class provides a generic SQL graphical interface for
 *        browsing tables, executing SQL commands and importing & exporting
 *        records.
 */


SqlBrowser::SqlBrowser( AbstractController* controller,
			QWidget* parent, Qt::WindowFlags flags )
  : GuiDbComponentDialog(controller, parent, flags),
    m_editModel(0),
    m_queryModel(0)
{
  m_ui.setupUi( this );

  Q_ASSERT( m_ui.importButton );
  m_ui.importButton->setEnabled( false ); // disabled until an SQL table is selected

  connect( m_ui.importButton, SIGNAL(clicked(bool)), SLOT(importData()) );
  connect( m_ui.exportButton, SIGNAL(clicked(bool)), SLOT(exportData()) );

  m_editModel = new QSqlTableModel( this );
  m_queryModel = new QSqlQueryModel( this );

   Q_ASSERT( m_ui.tableView );
   m_ui.tableView->setModel( m_queryModel ); // needs to be something

  /* Examples:
   *   1,2
   *   1001,"hello world"
   *   2002 , "hello ""Bob"""
   *   """field1A"",""field1B""", "field2", "field3"
   *
   * This has been successfully tested with CSV export in Microsoft Excel 2004
   * Mac edition.
   */
  c_regexp.setPattern( "\\s*"               // allow for spaces at the beginning
		       "(?:^(?=.)|,)(?:"    /* Must be the start of a line (^) or the next
					       field (,). If at the start, then the field
					       must contain at least one character (?=.)
					       - otherwise we could continue forever reading
					       just the ^, not capturing anything. */
		       "\""                 // open quote
		       "((?:[^\"]|\"\")*)"  /* capture the text that unless in pairs,
					       does not contain double-quotes. */
		       "\""                 // close quote
		       "|"                  // ELSE...
		       "([^\",]*)"          /* capture text that does not contain any
					   double-quotes or commas at all. */
		       ")\\s*" );           // allow for spaces at the end
}


/*!
 * Re-syncs the GUI with the database if digestDbModel() is non-null, as it may
 * have changed for some unknown reason without sending update events to us.
 *
 * Users tend to expect a refresh if a window is hidden and then shown again.
 *
 * It then then passes event onto GuiDbComponentDialog.
 */
void SqlBrowser::showEvent( QShowEvent* event )
{
  if ( digestDbModel() ) {
    refreshTableList();
    refreshQuery();
  }
  GuiDbComponentDialog::showEvent( event );
}


void SqlBrowser::resetEvent( VEvent* ) {
  refreshTableList();
  refreshQuery();
}


void SqlBrowser::classesEvent( VClassesEvent* )
{ refreshQuery(); }

void SqlBrowser::collectionsEvent( VCollectionsEvent* )
{ refreshQuery(); }

void SqlBrowser::experimentsEvent( VExperimentsEvent* )
{ refreshQuery(); }

void SqlBrowser::gesturesEvent( VGesturesEvent* )
{ refreshQuery(); }

void SqlBrowser::trainedRecogsEvent( VTrainedRecogsEvent* )
{ refreshQuery(); }


/*!
 * This will re-execute the current query and call updateNumRecordsLabel().
 * The query could be the result of calling either execUserQuery() or
 * showTable().
 */
void SqlBrowser::refreshQuery()
{
  /*
   * WARNING: For some reason (be it caching or update-only-on-query-change),
   *          we can't store the query as a QSqlQuery and then re-use it for
   *          a refresh. Instead, it must be stored and re-used as a string.
   */
  Q_ASSERT( m_ui.tableView );
  QSqlQueryModel* m = 0;
  if ( m_ui.tableView->model() == m_editModel ) {
    m_editModel->select();
    m = m_editModel;
  }
  else { // query model + catch case
    m_queryModel->setQuery( m_queryString, database() );
    m = m_queryModel;
  }

  Q_ASSERT( m );
  while ( m->canFetchMore() )  // show ALL results
    m->fetchMore();

  updateNumRecordsLabel();
}


void SqlBrowser::execUserQuery()
{
  Q_ASSERT( m_queryModel );

  Q_ASSERT( m_ui.queryEditor );
  m_queryString = m_ui.queryEditor->toPlainText();
  m_queryModel->setQuery( m_queryString, database() );

  Q_ASSERT( m_ui.tableView );
  m_ui.tableView->setModel( m_queryModel );

  while ( m_queryModel->canFetchMore() ) // show ALL results
    m_queryModel->fetchMore();

  QString queryStatus;
  if ( m_queryModel->lastError().type() != QSqlError::NoError )
    queryStatus = ( m_queryModel->lastError().driverText() + "\n"
		    + m_queryModel->lastError().databaseText() );
  else if ( m_queryModel->query().isSelect() )
    queryStatus = tr( "Query OK." );
  else
    queryStatus = ( tr("Query OK, number of affected rows: %1")
		    .arg(m_queryModel->query().numRowsAffected()) );
  setQueryStatus( queryStatus );

  updateNumRecordsLabel(); // dictated by tableView

  // Update the table list, as a table could have been created or dropped.
  refreshTableList();
}


void SqlBrowser::showTable( const QString& table )
{
  // Using table directly -> query no longer in use -> indicate this visually.
  // Similar to how on_submitButton_clicked clears the table selection.
  setQueryStatus( QString() );

  Q_ASSERT( m_editModel );
  m_editModel->setTable( table );
  m_editModel->select();

  Q_ASSERT( m_ui.tableView );
  m_ui.tableView->setModel( m_editModel );

  while ( m_editModel->canFetchMore() ) // show ALL results
    m_editModel->fetchMore();

  updateNumRecordsLabel(); // dictated by tableView
}


void SqlBrowser::setQueryStatus( const QString& message ) {
  Q_ASSERT( m_ui.queryStatusLabel );
  m_ui.queryStatusLabel->setText( message ); 
}


void SqlBrowser::refreshTableList()
{
  Q_ASSERT( m_ui.tableListWidget );

  m_ui.tableListWidget->clear();

  if ( database().isOpen() ) {
    QStringList tables = database().tables();
    tables.removeAll( "sqlite_sequence" ); // TODO: make this hide OPTIONAL (check box or a "hide table" list)
    m_ui.tableListWidget->addItems( tables );
  }

  Q_ASSERT( m_ui.numTablesLabel );
  int n = m_ui.tableListWidget->count();
  m_ui.numTablesLabel->setText( (n==0 ? tr("No") : QString::number(n))
				+ tr(" table")
				+ (n==1 ? "" : "s") );
}


/*!
 * Note: The numRecordsLabel will be updated to reflect the number of items in
 *       tableView. Hence, tableView must be updated before this is called.
 *       This should be intuitive, given that this is exactly what it means
 *       to the user.
 */
void SqlBrowser::updateNumRecordsLabel()
{
  Q_ASSERT( m_ui.tableView );
  Q_ASSERT( m_ui.numRecordsLabel );
  if ( m_ui.tableView->model() ) // test required, as model may not have been set
    {
      int n = m_ui.tableView->model()->rowCount();
      m_ui.numRecordsLabel->setText( (n==0 ? tr("No") : QString::number(n))
				      + tr(" item")
				      + (n==1 ? "" : "s") );
    }
}


void SqlBrowser::on_tableListWidget_itemSelectionChanged()
{
  bool importEnabled = false;

  Q_ASSERT( m_ui.tableListWidget );
  if ( !m_ui.tableListWidget->selectedItems().isEmpty() ) {
      showTable( m_ui.tableListWidget->selectedItems().first()->text() );
      importEnabled = true;
  }

  Q_ASSERT( m_ui.importButton );
  m_ui.importButton->setEnabled( importEnabled ); // to meet importData() demands
}


void SqlBrowser::on_submitButton_clicked()
{
  Q_ASSERT( m_ui.queryEditor );
  m_ui.queryEditor->setFocus();

  // Query will dictate table view, so clear the table selection to indicate this
  Q_ASSERT( m_ui.tableListWidget );
  m_ui.tableListWidget->clearSelection();

  // Update the table view
#if 0 // TODO: re-enable me
  execUserQuery();
#endif
}


void SqlBrowser::on_clearButton_clicked()
{
  setQueryStatus( QString() ); // query no longer in use

  Q_ASSERT( m_ui.queryEditor );
  m_ui.queryEditor->clear();
  m_ui.queryEditor->setFocus();
}


/*!
 * This will import CSV (Comma Separated Values) data from a file into the
 * currently selected table.
 *
 * This method was designed to import data that was exported by exportData().
 * However, the data may be processed/filtered or completely generated by another
 * program or even a human.
 *
 * If there's a problem, this lets SQL database backend complain and this will
 * report the error. After all, this browser is meant to be low-level.
 *
 * This has been successfully tested with CSV export in Microsoft Excel 2004
 * Mac edition.
 *
 * \b Warning: This method requires that an SQL table be selected, thus the user
 *             shouldn't be able to execute this method unless one is. Hence,
 *             no selection requires that the "Import..." button be disabled.
 */
void SqlBrowser::importData()
{
  // TODO: Major code and doc cleanup!

  // Sanity/safety checks
  Q_ASSERT( m_ui.importButton );
  Q_ASSERT( m_ui.importButton->isEnabled() );
  Q_ASSERT( m_ui.tableListWidget );
  Q_ASSERT( !m_ui.tableListWidget->selectedItems().isEmpty() );
  QString table = m_ui.tableListWidget->selectedItems().first()->text();

  /* Note: this should appear as a sheet on Mac OS X, as it's specific to this
   *       browser, which will block until it's closed and will not result in a
   *       new window being opened.
   */
  QString filename = QFileDialog::getOpenFileName( this, tr("Import") );
  QFileInfo fi( filename );

  // Note: the following also covers the case of when the user clicks cancel
  if ( filename.isEmpty() )
    return;

  QFile file( filename );
  if ( ! file.open(QIODevice::ReadOnly | QIODevice::Text) ) {
    // TODO: make the message useful!
    QMessageBox::critical( this, "",
			   tr("Couldn't Open"),
			   QMessageBox::Ok, QMessageBox::NoButton );
    return;
  }

  // QTextStream takes care of converting the 8-bit data stored on disk into a
  // 16-bit Unicode QString
  QTextStream in( &file );

  int  line = 1;
  bool ignoreAllErrors = false;
  while ( !in.atEnd() )
    {
      QStringList list = in.readLine().split('\r');
      if ( list.size() >= 2 ) {
	QString last = list.takeLast();
	for ( int i=last.size()-1; i>=0; --i )
	  file.ungetChar( last.at(i).toLatin1() );
      }

      foreach ( const QString& str, list )
	{
	  /* Note: We CAN'T pre-prepare a query, as different lines of the file may
	   *       have different numbers of columns. The user may not even be aware
	   *       of this. If there's a problem, let the SQL DB complain and we'll
	   *       report the error. After all, this browser is meant to be low-level.
	   */
	  int rxpos = 0;
	  QString qStr = "INSERT INTO " + table + " VALUES (";
	  while ( (rxpos = c_regexp.indexIn(str, rxpos)) != -1 ) {
	    QString value = c_regexp.cap( 1 );                 // try quoted string first
	    if ( value.isEmpty() ) value = c_regexp.cap( 2 );  // ELSE, try un-quoted (see rx)
	    value.replace( "\"\"", "\"" ); // undo the CSV double-quote escaping
	    qStr += formatSqlValue(value) + ",";
	    rxpos += c_regexp.matchedLength();
	  }
	  if ( qStr.right(1) == "," ) qStr.chop( 1 );
	  qStr += ")";

	  QSqlQuery q( qStr, database() );
	  q.exec();
	  if ( q.lastError().type() != QSqlError::NoError
	       && !ignoreAllErrors )
	    {
	      /*
	       * Note: The button to abort is called "Stop", not "Cancel", as any
	       *       records that have already been imported will remain, hence
	       *       the entire task won't be cancelled, it'll be prematurely
	       *       stopped.
	       */
	      QString title;
	      Qt::WindowFlags flags;
#ifdef Q_WS_MAC
	      flags = Qt::Sheet;
#else
	      title = windowTile();
	      flags = Qt::Dialog | Qt::MSWindowsFixedSizeDialogHint;
#endif

	      QMessageBox mb( title,
			      tr("<h4>An error was encountered while importing "
				 "the record on line %1 of \"%2\".</h4>\n\n"
				 "<span style=\"font-size:11pt;\">%3<br>%4</span>")
			      .arg(line)
			      .arg(fi.fileName())
			      .arg(q.lastError().driverText())
			      .arg(q.lastError().databaseText()),
			      QMessageBox::Warning,
			      QMessageBox::Abort | QMessageBox::Escape,
			      QMessageBox::YesAll,
			      QMessageBox::Ignore | QMessageBox::Default,
			      this, flags );
	      mb.setButtonText( QMessageBox::Abort, tr("Stop") );
	      mb.setButtonText( QMessageBox::YesAll, tr("Ignore All") );
	      mb.setMinimumSize( 500, 160 ); // wide and fixed
	      mb.setMaximumSize( 500, 160 ); // ditto.
	      switch ( mb.exec() ) {
	      case QMessageBox::Abort:  return;
	      case QMessageBox::YesAll: ignoreAllErrors=true; break;
	      case QMessageBox::Ignore: break;
	      }
	    }

	  ++line;
	}
    }

  // Show changes
  showTable( table );
}


/*!
 * This "export" method simply dumps the current contents of the records table
 * (be it the full contents of a table or the result of a user-specified SQL query)
 * as an ascii file, where each line represents one record and each field value
 * is quoted (with double quotes - e.g. "value") and separated by a comma
 * (i.e. CSV - Comma Separated Values).
 *
 * Quoting is absolutely critical, as the data that this application uses
 * contains many commas - would would interfere with the field delimiting.
 * On the rare occasion that a field value contains a double-quote character,
 * then it will be replaced with two quotes.
 * For example <em>My name is "Bob"</em> becomes <em>"My name is ""Bob"""</em>.
 * This form of escaping is standard practice and is supported by many
 * applications, including Microsoft Excel.
 *
 * Although this kind of export is primitive, it's very flexible and easy to
 * manipulate, it's human readable and most people are already used to it and
 * finally, virtually all spreadsheets will import this kind of data.
 *
 * This generates a "sane" default file name first. If a table is selected, then
 * the file name will be TABLENAME.csv, otherwise it will be query.csv.
 *
 * The extension ".csv" was chosen over ".txt", as programs such as Microsoft
 * Excel can open files with this extension and know exactly what to do with it,
 * without asking questions about what delimiters it uses and so forth.
 */
void SqlBrowser::exportData()
{
  /* Qt looks after the case of when a user chooses an existing file, asking them
   * for confirmation when they do so.
   */
  Q_ASSERT( m_ui.tableListWidget ); // NOTE: selectedItems() is safer than currentItem() here
  QString filename = ( (m_ui.tableListWidget->selectedItems().isEmpty()
			? tr("query")
			: m_ui.tableListWidget->selectedItems().first()->text())
		       + ".csv" );
  filename = QFileDialog::getSaveFileName( this, tr("Export"), filename );

  // Note: the following also covers the case of when the user clicks cancel
  if ( filename.isEmpty() )
    return;

  QFile file( filename );
  if ( ! file.open(QIODevice::WriteOnly | QIODevice::Text) ) {
    // TODO: make the message useful!
    QMessageBox::critical( this, "",
			   tr("Couldn't Save"),
			   QMessageBox::Ok, QMessageBox::NoButton );
    return;
  }

  QTextStream out( &file );

  Q_ASSERT( m_ui.tableView );
  Q_ASSERT( m_ui.tableView->model() );
  QAbstractItemModel* model = m_ui.tableView->model();
  for ( int j=0; j < model->rowCount(); ++j ) {
    for ( int i=0; i < model->columnCount(); ++i ) {
      QString s = model->data(model->index(j,i)).toString();
      s.replace( "\"", "\"\"" ); // escape double-quote characters
      out <<  "\"" << s << "\"";
      if ( i < model->columnCount()-1 ) out << ","; // not last field -> comma
    }
    if ( j < model->rowCount()-1 ) endl(out); // not last record -> new line
  }

  file.close();
}



QString SqlBrowser::formatSqlValue( QString value )
{
  QVariant var( value );
  bool ok = false;
  return ( ((var.toDouble(&ok) && ok) // encapsulates Int, UInt and Bool
	    || (var.toLongLong(&ok) && ok)
	    || (var.toULongLong(&ok) && ok))
	   ? value // number -> no quoting required
	   : ('\''
	      + value.replace("\'", "\'\'") /* escape single-quotes for SQL,
					       then quote it was a string */
	      + '\'') );
}
