/*  -*- c++ -*-  (for Emacs)
 *
 *  featureslistwidget.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Tue Aug 16 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

// TODO: add ability to say "only allow drags/drop to/from X"
//       -> prevent probs with other instances of TrainingAssistant and apps!

// TODO: allow items to be moved to the very end of the list!

#include "featureslistwidget.h"

#include <QByteArray>
#include <QCursor>
#include <QDebug> // TODO: remove me!
#include <QDrag>
#include <QDragMoveEvent>
#include <QDropEvent>
#include <QMimeData>
#include <QPainter>

#include "GestureRecognition/featurefactory.h"

// TODO: move the following to a globals file!
#define FEATURE_MIME_FORMAT_STR  "text/x-feature-key"


FeaturesListWidget::FeaturesListWidget( QWidget* parent )
  : HeadedListWidget(parent),
    m_drawInsertionLine(false),
    m_insertionLineRow(-1)
{
  m_insertionLinePen = QPen( palette().color(QPalette::Dark) );
  m_insertionLinePen.setWidth( 2 );

  setDragEnabled( true );
  setAcceptDrops( true );
  setDropIndicatorShown( true );
}


void FeaturesListWidget::addFeature( const QByteArray& key ) {
  insertFeature( count(), key );
}


void FeaturesListWidget::insertFeature( int row, const QByteArray& key )
{
  QListWidgetItem* item = new QListWidgetItem( FeatureFactory::title(key) );
  item->setData( FeatureKeyRole, key );
  item->setFlags( Qt::ItemIsEnabled
		  | Qt::ItemIsSelectable
		  | Qt::ItemIsDragEnabled );
  insertItem( row, item );
}


/*! Convenience method. */
QByteArray FeaturesListWidget::itemFeatureKey( QListWidgetItem* v ) {
  return ( v != 0
	   ? v->data(FeatureKeyRole).toByteArray()
	   : QByteArray() );
}


/*! Convenience method. */
QByteArray FeaturesListWidget::rowFeatureKey( int row ) {
  QListWidgetItem* v = item( row );
  return ( v != 0
	   ? v->data(FeatureKeyRole).toByteArray()
	   : QByteArray() );
}


/*!
 * Returns the pen that is used when drawing the insertion line.
 *
 * Defaults to a QPalette::Dark coloured pen with a width (height) of 2 pixels.
 */
const QPen& FeaturesListWidget::insertionLinePen() const {
  return m_insertionLinePen;
}

/*!
 * Sets the pen that is used when drawing the insertion line.
 *
 * Defaults to a QPalette::Dark coloured pen with a width (height) of 2 pixels.
 */
void FeaturesListWidget::setInsertionLinePen( const QPen& pen ) {
  m_insertionLinePen = pen;
  update();
}


void FeaturesListWidget::startDrag( Qt::DropActions /*supportedActions*/ )
{
  QListWidgetItem* item = currentItem();
  if ( item != 0 )
    {
      QByteArray key = item->data(FeatureKeyRole).toByteArray();

      QMimeData* mimeData = new QMimeData;
      mimeData->setData( FEATURE_MIME_FORMAT_STR, key );

      QDrag* drag = new QDrag( this );
      drag->setMimeData( mimeData );

      /* Note: m_dragStartPos WILL be different to
       *       viewport()->mapFromGlobal(QCursor::pos()),
       *       due to the startDragDistance() requirement.
       */
      Q_ASSERT( viewport() != 0 );
      QRect itemRect = visualItemRect( item );
      QPoint localMousePos = m_dragStartPos;
      drag->setPixmap( QPixmap::grabWidget(viewport(), itemRect) );
      drag->setHotSpot( localMousePos - itemRect.topLeft() );

      int itemRow = row( item );
      delete takeItem( itemRow ); // make it disappear -> intuitive (user is holding it)
      if ( drag->start(Qt::MoveAction) != Qt::MoveAction )
	insertFeature( itemRow, key ); // failed -> put it back!
    }
}


void FeaturesListWidget::mousePressEvent( QMouseEvent* event )
{
  Q_ASSERT( event != 0 );
  QListWidget::mousePressEvent( event );
  if ( event->button() == Qt::LeftButton )
    m_dragStartPos = event->pos();
}


void FeaturesListWidget::dragEnterEvent( QDragEnterEvent* event )
{
  Q_ASSERT( event != 0 );
  if ( event->mimeData()->hasFormat(FEATURE_MIME_FORMAT_STR) )
    event->acceptProposedAction();
}


void FeaturesListWidget::dragLeaveEvent( QDragLeaveEvent* /*event*/ )
{
  m_drawInsertionLine = false;
  update();
}


void FeaturesListWidget::dragMoveEvent( QDragMoveEvent* event )
{
  Q_ASSERT( event != 0 );

  if ( event->mimeData()->hasFormat(FEATURE_MIME_FORMAT_STR) )
    {
      QByteArray itemKey = event->mimeData()->data( FEATURE_MIME_FORMAT_STR );

      m_drawInsertionLine = true;
      m_insertionLineRow = indexAt(event->pos()).row();
      update();

      event->setDropAction( Qt::MoveAction );
      event->accept();
    }
  else
    event->ignore();
}


void FeaturesListWidget::dropEvent( QDropEvent* event )
{
  Q_ASSERT( event != 0 );

  if ( event->mimeData()->hasFormat(FEATURE_MIME_FORMAT_STR) )
    {
      insertFeature( indexAt(event->pos()).row(),
		     event->mimeData()->data(FEATURE_MIME_FORMAT_STR) );
      event->setDropAction( Qt::MoveAction );
      event->accept();
    }
  else
    event->ignore();

  m_drawInsertionLine = false;
  update();
}


void FeaturesListWidget::paintEvent( QPaintEvent* event )
{
  HeadedListWidget::paintEvent( event );

  if ( m_drawInsertionLine )
    {
      const QListWidgetItem* itemAtRow = item( m_insertionLineRow );
      if ( itemAtRow != 0 )
	{
	  QRect r = visualItemRect( itemAtRow );
	  QPainter p( viewport() );
	  p.setPen( m_insertionLinePen );
	  p.drawLine( 0, r.y(), width(), r.y() );
	}
    }
}
