/*  -*- c++ -*-  (for Emacs)
 *
 *  trainingassistant.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Tue Jul 21 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "trainingassistant.h"

#include <QCoreApplication>
#include <QDate>
#include <QDebug> // TODO: remove me!
#include <QDir> // TODO: remove me!
#include <QItemSelectionModel>
#include <QSqlQuery> // TODO: remove me!
#include <QStandardItemModel>
#include <QStringList>
#include <QTemporaryFile>

#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/digestdbcontroller.h"
#include "GestureRecognition/abstractrecogniser.h"
#include "GestureRecognition/featurefactory.h"
#include "GestureRecognition/recogniserfactory.h"
#include "DigestGuiCore/categoryitemmodel.h"
#include "DigestGuiCore/gesturequerymodel.h"

// TODO: move the following to digestdatabase.h
#define CLASS_TABLE_STR          "Class"
#define CLASS_TABLE_ID_INDEX     0
#define CLASS_TABLE_LABEL_INDEX  1
// TODO: move the following to digestdatabase.h
// TODO: use the more flexible STRING indices!
#define GESTURE_TABLE_STR        "Gesture"
#define GESTURE_TABLE_ID_STR     "id"
#define GESTURE_TABLE_ID_INDEX   0


// Multiple instances of TrainingAssistant may run at any given time, so we
// should share the following strings.
static bool s_strsInitialized = false;
static QStringList s_pageTitleStrs;
#if 0 // TODO: remove me (when used, user could see layout re-adjusting to it):
static QStringList s_pageDescStrs;
#endif



/*!
 * \class TrainingAssistant
 *
 * \brief The TrainingAssistant class provides a graphical interface for
 *        training new recognisers.
 */


TrainingAssistant::TrainingAssistant( DigestDbController* controller,
				      JavaVM* jvm,
				      QWidget* parent, Qt::WindowFlags flags )
  : GuiDbComponentDialog(controller, parent, flags),
    m_jvm(jvm),
    m_recogniser(0),
    m_currentStep(TrainingAssistant::Introduction),
    m_categoryModel(0),
    m_gestureModel(0)
{
  /*
   * WARNING: The use of the models is delayed until we receive VEvent::Reset.
   */

  if ( ! s_strsInitialized ) initStaticStrs();

  m_dummyModel = new QStandardItemModel( this );

  ui.setupUi( this );
  initWidgets();

  if ( RecogniserFactory::keys().isEmpty() )
    m_currentStep = TrainingAssistant::NoRecognisers;
  else if ( FeatureFactory::keys().isEmpty() )
    m_currentStep = TrainingAssistant::NoFeatures;

  reflectCurrentStep();
}


void TrainingAssistant::initStaticStrs()
{
  Q_ASSERT( ! s_strsInitialized ); // MUST only init once

  s_strsInitialized = true;

  s_pageTitleStrs
    << tr("Introduction")
    << tr("Select Recogniser Type")
    << tr("Select Features")
    << tr("Select Gestures")
    << tr("Final Details")
    << tr("Confirm Details")
    << tr("Training")
    << tr("Finished")
    << tr("Problem")  // no recognisers
    << tr("Problem"); // no features
}


void TrainingAssistant::initWidgets()
{
  /*
   * WARNING: The use of the model is delayed until we receive VEvent::Reset.
   */


  /*
   * Step 1: Select Recogniser Type
   */
  Q_ASSERT( ui.recogTypeCombo );
  foreach ( const QByteArray& key, RecogniserFactory::keys() )
    ui.recogTypeCombo->addItem( RecogniserFactory::title(key), key );

  connect( ui.recogTypeCombo,
	   SIGNAL(activated(int)), SLOT(updateRecogInfoWidget(int)) );

  // Call for an update NOW, as to init the widget to the first item if it exists
  if ( ui.recogTypeCombo->count() > 0 )
    updateRecogInfoWidget( 0 );

  
  /*
   * Step 2: Select Features
   */
  Q_ASSERT( ui.featureListWidget );
  ui.featureListWidget->setHeaderText( tr("Available Features") );

  Q_ASSERT( ui.trainingFeatureListWidget );
  ui.trainingFeatureListWidget->setHeaderText( tr("Features for Training") );
  // Use all featues by default:
  foreach( const QByteArray& key, FeatureFactory::keys() )
    ui.trainingFeatureListWidget->addFeature( key );

  // Select the first item in the list (if there is one), which indirectly calls
  // for its info to be shown.
  if ( ui.trainingFeatureListWidget->count() > 0 )
    ui.trainingFeatureListWidget
      ->setItemSelected( ui.trainingFeatureListWidget->item(0), true );


  /*
   * Step 3: Select Gestures
   */

  // Category view
  // Note: we don't pass in the controller, we're using addChildView()
  Q_ASSERT( m_categoryModel == 0 ); // MUST currently be null!
  m_categoryModel = new CategoryItemModel( this ); // see note above re: controller
  addChildView( m_categoryModel );
  // Note: We don't allow for selection of items, as we wouldn't do anything with
  //       the selection anyway and we don't want the user to think that we do.
  //       Also, we don't allow the labels to be edited, nor are items drop enabled.
  m_categoryModel->setModelFlags( CategoryItemModel::ItemsAreCheckable );
  Q_ASSERT( ui.categoryTreeView );
  ui.categoryTreeView->setHeaderText( tr("Category") );
  ui.categoryTreeView->setModel( m_categoryModel );

  connect( m_categoryModel,
	   SIGNAL(dataChanged(const QModelIndex&, const QModelIndex&)),
	   SLOT(onCategoryModelDataChanged()) );

  // Gesture view
  // Note: we don't pass in the controller, we're using addChildView()
  Q_ASSERT( m_gestureModel == 0 ); // MUST currently be null!
  m_gestureModel = new GestureQueryModel( this ); // see note above re: controller
  addChildView( m_gestureModel );
  Q_ASSERT( ui.gestureListView );
  // Note: These settings are so important, that we don't want to leave them in
  // the .ui file, as they may accidently be modified or reset without noticing.
  ui.gestureListView->setHeaderText( tr("Chosen Gestures") );
  ui.gestureListView->setSelectionMode( QAbstractItemView::NoSelection ); // for viewing purposes only
  ui.gestureListView->setViewMode( QListView::IconMode ); // affects other settings -> call this 1st
  ui.gestureListView->setIconSize( QSize(52,52) ); // TODO: use size from saved settings
  ui.gestureListView->setSpacing( 4 );
  ui.gestureListView->setLayoutMode( QListView::SinglePass ); // scrolls better than Batched
  ui.gestureListView->setResizeMode( QListView::Adjust ); // instant update
  ui.gestureListView->setEditTriggers( QAbstractItemView::NoEditTriggers ); // edit elsewhere
  ui.gestureListView->setUniformItemSizes( true ); // enable optimizations
  ui.gestureListView->setMovement( QListView::Static ); // for viewing purposes only
  ui.gestureListView->setDragEnabled( false ); // for viewing purposes only
  ui.gestureListView->setAcceptDrops( false ); 
}


void TrainingAssistant::resetEvent( VEvent* )
{
  /*
   * Clean-up
   */
  Q_ASSERT( ui.gestureListView );
  ui.gestureListView->setModel( m_dummyModel ); // can't be null :-(
  ui.gestureListView->reset();


  /*
   * (Re)build
   */
  if ( digestDbModel() )
    {
      /*
       * Step 3: Select Gestures
       */
      Q_ASSERT( m_categoryModel );
      m_categoryModel->setData( m_categoryModel->libraryIndex(), // good default:
				Qt::Checked, Qt::CheckStateRole );

      Q_ASSERT( ui.gestureListView );
      ui.gestureListView->setModel( m_gestureModel ); // *** note: set last, after properties
      onCategoryModelDataChanged(); // update the query now
    }
}


void TrainingAssistant::on_goBackButton_clicked() {
  gotoStep( (Step)(m_currentStep-1) );
}


void TrainingAssistant::on_continueButton_clicked() { // a.k.a. "Train" and "Quit"
  gotoStep( m_currentStep >= TrainingAssistant::Finished
	    ? TrainingAssistant::Closed
	    : (TrainingAssistant::Step)(m_currentStep+1) );
}


/*!
 * Updates the featureInfoWidget to show information (the description,
 * in particular) about the currently selected feature.
 *
 * This also makes the item selection mutually exclusive across both lists.
 * Otherwise, the description-to-item mapping could be ambiguous to the user.
 *
 * \em Note: It's asserted that list widget's selection mode is set to
 *           QAbstractItemView::SingleSelection, as it's very important
 *           (This should be set in the Qt Designer .ui file).
 */
void TrainingAssistant::on_featureListWidget_itemSelectionChanged() {
  worker_itemSelectionChanged( ui.featureListWidget, ui.trainingFeatureListWidget );
}


/*!
 * Updates the featureInfoWidget to show information (the description,
 * in particular) about the currently selected feature.
 *
 * This also makes the item selection mutually exclusive across both lists.
 * Otherwise, the description-to-item mapping could be ambiguous to the user.
 *
 * \em Note: It's asserted that list widget's selection mode is set to
 *           QAbstractItemView::SingleSelection, as it's very important
 *           (This should be set in the Qt Designer .ui file).
 */
void TrainingAssistant::on_trainingFeatureListWidget_itemSelectionChanged() {
  worker_itemSelectionChanged( ui.trainingFeatureListWidget, ui.featureListWidget );
}


void TrainingAssistant::worker_itemSelectionChanged( FeaturesListWidget* wCurrent,
						     FeaturesListWidget* wNotCurrent )
{
  Q_ASSERT( wCurrent );
  Q_ASSERT( wNotCurrent );
  Q_ASSERT( wCurrent->selectionMode() == QAbstractItemView::SingleSelection );

  // Note: we must prevent the other itemSelectionChanged() from being called.
  wNotCurrent->blockSignals( true );
  foreach ( QListWidgetItem* item, wNotCurrent->selectedItems() )
    wNotCurrent->setItemSelected( item, false );
  wNotCurrent->blockSignals( false );

  // Note: The only selected item may have been de-selected - i.e. no selected items
  if ( ! wCurrent->selectedItems().isEmpty() ) {
    QListWidgetItem* item = wCurrent->selectedItems().first();
    Q_ASSERT( item );
    updateFeatureInfoWidget( wCurrent->itemFeatureKey(item) );
  }
}


void TrainingAssistant::gotoStep( TrainingAssistant::Step step )
{
  bool callTrain = false;

  m_currentStep = step;
  Q_ASSERT( m_currentStep >= TrainingAssistant::Introduction ); // sanity check
  Q_ASSERT( m_currentStep <= TrainingAssistant::Closed );       // ditto.
 
  // Perform tasks that are required for the current step
  switch ( m_currentStep )
    {
    case TrainingAssistant::SelectGestures:
      // TODO: remove the need for this hack!
      //       - currently required to get the icons to appear in a grid from first view
      ui.gestureListView->setViewMode( QListView::ListMode );
      ui.gestureListView->setViewMode( QListView::IconMode );
      break;

    case TrainingAssistant::FinalDetails:
      // Generate a default label for the recogniser
      // TODO: ensure that the label is unique!
      Q_ASSERT( ui.recogLabelEdit );
      Q_ASSERT( ui.recogTypeCombo );
      if ( ! ui.recogLabelEdit->isModified() )
	ui.recogLabelEdit->setText( ui.recogTypeCombo->currentText() ); // use title
      break;

    case TrainingAssistant::ConfirmDetails:
      gatherDetails();
      Q_ASSERT( ui.detailsHtmlWidget );
      ui.detailsHtmlWidget->setHtml( detailsAsHtml() );
      break;

    case TrainingAssistant::Training:
      // Disable the continue button until we've not only trained the recogniser
      // but also have renamed it's file and its' READY for use - see trainedRecogEvent().
      Q_ASSERT( ui.continueButton );
      ui.continueButton->setEnabled( false );
      callTrain = true;
      break;

    case TrainingAssistant::Finished:
    case TrainingAssistant::NoRecognisers:
    case TrainingAssistant::NoFeatures:
      Q_ASSERT( ui.continueButton );
      ui.continueButton->setEnabled( true );
      break;

    case TrainingAssistant::Closed:
      close();
      return; // required - can't step any further!
    
    default:
      break;
    }

  reflectCurrentStep();

  /* This is a bit hacky, but it allows the user to see the Training page while
   * training is in progress, thus making the progress bar visible.
   */
  if ( callTrain ) train();
}


void TrainingAssistant::reflectCurrentStep()
{
  /* WARNING:
   * We don't set the description directly here, instead we have a different
   * label widget on each page of the stepWidgetStack.
   * Otherwise, the user would see the layout being re-adjusted to fit it (yuck!).
   */

  setUpdatesEnabled( false ); // start of many changes

  Q_ASSERT( ui.stepTitleLabel );
  Q_ASSERT( ui.stepWidgetStack );
  ui.stepTitleLabel->setText( s_pageTitleStrs.at(m_currentStep) );
  ui.stepWidgetStack->setCurrentIndex( m_currentStep );

  Q_ASSERT( ui.goBackButton );
  ui.goBackButton->setEnabled( m_currentStep > TrainingAssistant::Introduction
			       && m_currentStep < TrainingAssistant::Training ); // past point of no return

  Q_ASSERT( ui.continueButton );
  ui.continueButton->setText( (m_currentStep >= TrainingAssistant::Finished)
				 ? tr("&Close")
				 : ((m_currentStep >= TrainingAssistant::ConfirmDetails)
				    ? tr("&Train")
				    : tr("&Continue")) );				 

  setUpdatesEnabled( true ); // end of many changes
}


void TrainingAssistant::updateRecogInfoWidget( int recogTypeComboIndex )
{
  Q_ASSERT( ui.recogTypeCombo );
  Q_ASSERT( ui.recogInfoWidget );
  QByteArray key( ui.recogTypeCombo->itemData(recogTypeComboIndex).toByteArray() );
  ui.recogInfoWidget->setHtml( RecogniserFactory::description(key) );
}


void TrainingAssistant::updateFeatureInfoWidget( const QByteArray& key )
{
  Q_ASSERT( ui.featureInfoWidget );
  ui.featureInfoWidget->setHtml( FeatureFactory::description(key) );
}


void TrainingAssistant::gatherDetails()
{
  // Step 1. Recogniser Type
  Q_ASSERT( ui.recogTypeCombo );
  m_recogTypeKey = ui.recogTypeCombo->itemData(ui.recogTypeCombo->currentIndex()).toByteArray();

  // Step 2. Features
  Q_ASSERT( ui.trainingFeatureListWidget );
  m_featureKeys.clear(); // VERY IMPORTANT, as we may be called multiple times (back&forward)
  int numFeatures = ui.trainingFeatureListWidget->count();
  for ( int i=0; i < numFeatures; ++i ) {
    const QListWidgetItem* item = ui.trainingFeatureListWidget->item(i);
    Q_ASSERT( item );
    m_featureKeys += item->data(FeaturesListWidget::FeatureKeyRole).toByteArray();
  }

  // Step 3. Gestures
  Q_ASSERT( m_categoryModel );
  m_gestureIds.clear(); // VERY IMPORTANT, as we may be called multiple times
  m_gestureQuery = m_categoryModel->gestureQueryForCheckedIndexes(); // be safe -> update it!
  QSqlQuery q( m_gestureQuery );
  while ( q.next() ) {
    bool intOk = false;
    m_gestureIds += q.value(0).toInt(&intOk);
    Q_ASSERT( intOk );
  }

  // Step 4. Final Details
  Q_ASSERT( ui.recogLabelEdit );
  Q_ASSERT( ui.recogNotesEdit );
  m_recogLabelStr = ui.recogLabelEdit->text();
  m_recogNotesStr = ui.recogNotesEdit->toPlainText(); // as HTML will cause issues
}


void TrainingAssistant::train()
{
  int id = -1;
  QString tmpModelFilename;
  DTrainedRecogRecord rec;

  /* Generate a temporary filename for the recogniser's training data.
   * Note: We rename it to something better that's based on the trained
   *       recogniser's ID (once we get it) in trainedRecogEvent().
   */
  // NOTE: QTemporaryFile should look after the tricky prevention of a race condition.
  // TODO: OPTIMISE ME!!!
  // Note: The filename needs to have a file extension for
  //       AbstractRecogniserTrainer::outputPath() to work.
  QTemporaryFile tf( QDir::homePath() + "/Library/Digest/tmp.XXXXXX" );
  tf.setAutoRemove( false ); // keep it as a marked placeholder -> prevent race condition
  if ( ! tf.open() ) {
    qDebug() << "failed to create temp file"; // TODO: write proper message box code
    return;
  }
  tmpModelFilename = tf.fileName();
  tf.close();
  // Get name of the file, excluding the path.
  QStringList fileNameStrs = tmpModelFilename.split( "/", QString::SkipEmptyParts );
  if ( ! fileNameStrs.isEmpty() )
    tmpModelFilename = fileNameStrs.last();
  else {
    Q_ASSERT( ! "panic!" );  // TODO: write proper error message!
  }
    

  // Remove any existing recogniser!
  delete m_recogniser;
  m_recogniser = 0; // let's be safe :-)

  m_recogniser
    = RecogniserFactory::create( m_recogTypeKey, m_jvm, digestDbModel(), this );
  Q_ASSERT( m_recogniser );

  Q_ASSERT( ui.trainingProgressBar );
  ui.trainingProgressBar->setMinimum( 0 );
  ui.trainingProgressBar->setMaximum( m_gestureIds.size() ); // TODO: use fixed num like ExperimentAssistant

  connect( m_recogniser,
	   SIGNAL(trainingProgressed(int)),
           SLOT(onTrainingProgressed(int)) );

  /* Note:
   * Our onTrainingProgressed() method makes calls to
   * QCoreApplication::processEvents() during training, as to keep the GUI
   * responsive and the progress bar in sync.
   */
  // TODO: update trainingFailed when calling train()
  QHash<QString, QVariant> params = m_recogniser->defaultParams(); // TODO: remove me!
  m_recogniser->train( m_gestureIds, m_featureKeys,
		       QDir::homePath() + "/Library/Digest/" + tmpModelFilename, // TODO: put elsewhere on Linux
		       params );

  rec.id = id; // updating a record -> must provide the id
  rec.label = m_recogLabelStr;
  rec.recogniserKey = m_recogTypeKey;
  rec.external = false; // created by app or the app -> internal!
  rec.date = QDate::currentDate();
  rec.notes = m_recogNotesStr;
  rec.ready = false; // VERY important - not ready until we've renamed the file - see trainedRecogEvent
  rec.modelFile = tmpModelFilename;
  rec.orderedFeatures = m_featureKeys;
  rec.trainingSet = m_gestureIds;

  // Continues in TrainingAssistant::trainedRecogEvent() ...
  postControllerEvent( new CTrainedRecogAddEvent(rec, this) );

  // TODO: create timeout, just in-case we don't ever get to trainedRecogEvent()
}


// TODO: doc me:
// On TrainedRecogUpdated and recogniser is ready, calls gotoStep(Finished);
void TrainingAssistant::trainedRecogsEvent( VTrainedRecogsEvent* event )
{
  // TODO: cleanup!!!

  Q_ASSERT( event );

  if ( event->idSet().isEmpty() ) return;
  int id = * event->idSet().begin(); // hacky

  if ( event->originalSender() == this )
    {
      VDigestDbEvent::Type type = (VDigestDbEvent::Type)event->type();

      Q_ASSERT( digestDbModel() );
      DTrainedRecogRecord rec = digestDbModel()->fetchTrainedRecog( id );

      if ( type == VDigestDbEvent::TrainedRecogsUpdated
	   && rec.ready ) {
	gotoStep( Finished );
      }
      else if ( type == VDigestDbEvent::TrainedRecogsAdded )
	{
	  // If a trained recogniser was ADDED (not updated) by us, then rename
	  // its filename to one based on its ID and kind, making it easier to find.
	  
	  Q_ASSERT( ! rec.ready ); // sanity check - it can't be ready yet!
	  
	  // Generate a unique filename - TODO: ensure that it is unique, altering it if it already exists!
	  QString betterFileBase
	    = rec.recogniserKey + "_recogdata" + QString::number(rec.id);
      
	  QString fileExt = ".model";
	  QString dirStr = QDir::homePath() + "/Library/Digest/"; // TODO: fixme!

	  // Handle the case where the proposed filename already exists
	  if ( QFile::exists(dirStr+betterFileBase+fileExt) )
	    {
	      QString stub = dirStr + betterFileBase + "_";
	      bool success = false;
	      for ( uint32_t i=2; i < (uint32_t)-1; ++i ) {
		if ( !QFile::exists(stub+QString::number(i)+fileExt) ) {
		  betterFileBase += "_" + QString::number(i);
		  success = true;
		  break;
		}
	      }
	      if ( ! success ) { // wow!
		betterFileBase = rec.modelFile; // (safely) throw arms in air
		fileExt = QString(); // hacky!
	      }
	    }

	  // WARNING!!!
	  // No one should be using the trained recogniser yet!!!
	  // Otherwise we would could easily run into a race condition to the file!
	  QFile::rename( dirStr+rec.modelFile,
			 dirStr+betterFileBase+fileExt );

	  rec.modelFile = betterFileBase+fileExt; // change it now, we finished with the old name
	  rec.ready = true; // VERY important - see the warning above!

	  postControllerEvent( new CTrainedRecogUpdateEvent(rec, this) );
	}
    }
}


void TrainingAssistant::onTrainingProgressed( int progress )
{
  Q_ASSERT( ui.trainingProgressBar );
  ui.trainingProgressBar->setValue( progress );

  /* In order to keep the UI responsive and properly updated during training,
   * we have Qt process any pending events after processing blocks of 10 gestures
   * (and also at the end - which may not be a multiple of 10).
   */
  if ( progress == m_gestureIds.size()
       || progress % 10 == 0 )
    QCoreApplication::processEvents(); // don't call with time - it seems to use all of it!
}


void TrainingAssistant::onCategoryModelDataChanged()
{
  Q_ASSERT( m_categoryModel );
  if ( m_gestureModel ) { // testing proved that we MUST check this
    m_gestureQuery = m_categoryModel->gestureQueryForCheckedIndexes();
    m_gestureModel->setQuery( m_gestureQuery );

    // Fetch ALL gestures - otherwise there are many issues to handle.
    while ( m_gestureModel->canFetchMore() )
      m_gestureModel->fetchMore();
  }
}


QString TrainingAssistant::detailsAsHtml() const
{
  QString str
    = "<html>"
    "<body>"
    "<table cellspacing=\"6\">"
    "<tr><th align=\"right\"><font color=\"#888888\">Label</font></th><td>"
    + (m_recogLabelStr.isEmpty() ? tr("(none)") : m_recogLabelStr) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Notes</font></th><td>"
    + (m_recogNotesStr.isEmpty() ? tr("(none)") : m_recogNotesStr) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Type</font></th><td>"
    + QString(m_recogTypeKey) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Feature Set</font></th><td>";

  if ( m_featureKeys.isEmpty() )
    str += tr("(none)");
  else {
    str += tr("%1 Features - ").arg(m_featureKeys.count());
    foreach ( const QByteArray& k, m_featureKeys )
      str += QString(k) + ", ";
    if ( str.right(2) == ", " ) str.chop(2);
  }

  str
    += "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Training Set</font></th><td>";

  if ( m_gestureIds.isEmpty() )
    str += tr("(none)");
  else {
    str += tr("%1 Gestures - ").arg(m_gestureIds.count());
    foreach ( int id, m_gestureIds )
      str += QString::number(id) + ", ";
    if ( str.right(2) == ", " ) str.chop(2);
  }

  str
    += "</td></tr>"
    "</body>"
    "</html>";

  return str;
}
