/*  -*- c++ -*-  (for Emacs)
 *
 *  headedscrollarea.ipp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Aug 04 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <QPainter>
#include <QPaintEvent>
#include <QPixmap>
#include <QScrollBar>


#define DEFAULT_HEADER_TEXT       QString()
#define DEFAULT_HEADER_FONT_SIZE  11
#define DEFAULT_HEADER_HEIGHT     17

#define FIXED_GAP_HEIGHT 1


template <class extendT>
HeadedScrollArea<extendT>::HeadedScrollArea( QWidget* parent )
  : extendT(parent),
    m_headerText(DEFAULT_HEADER_TEXT)
{
  init();
}


template <class extendT>
HeadedScrollArea<extendT>::HeadedScrollArea( const QString& headerText,
					     QWidget* parent )
  : extendT(parent),
    m_headerText(headerText)
{
  init();
}


template <class extendT>
void HeadedScrollArea<extendT>::init()
{
  extendT::setLineWidth( 1 ); // this makes it look better!

  m_headerFont = extendT::font();
  m_headerFont.setPointSize( DEFAULT_HEADER_FONT_SIZE );
  m_headerHeight = DEFAULT_HEADER_HEIGHT;

  m_leftPix = QPixmap( ":/images/HeaderLeft.png" );
  m_midPix = QPixmap( ":/images/HeaderMid.png" );
  m_rightPix = QPixmap( ":/images/HeaderRight.png" );

  extendT::setViewportMargins( 0, m_headerHeight + FIXED_GAP_HEIGHT, 0, 0 );
}


template <class extendT>
void HeadedScrollArea<extendT>::setHeaderText( const QString& text ) {
  m_headerText = text;
  extendT::update();
}


template <class extendT>
void HeadedScrollArea<extendT>::setHeaderFont( const QFont& font ) {
  m_headerFont = font;
  extendT::update();
}


/*!
 * Set the height of the header (in pixels). Defaults to 17 [pixels].
 */
template <class extendT>
void HeadedScrollArea<extendT>::setHeaderHeight( int height ) {
  m_headerHeight = height;
  extendT::setViewportMargins( 0, m_headerHeight + FIXED_GAP_HEIGHT, 0, 0 );
}


template <class extendT>
bool HeadedScrollArea<extendT>::event( QEvent* event )
{
  Q_ASSERT( event != 0 );

  /*
   * QAbstractScrollArea's event() method watches for QEvent::Paint
   * and sends it directly to QFrame, not its own paintEvent(), as
   * that has been re-mapped to receive viewport events.
   * Hence, we let the frame be painted and then paint the header!
   */
  if ( event->type() == QEvent::Paint ) {
    extendT::event( event );  // paint frame
    paintHeader();
    return true;
  }

  return extendT::event( event );
}


template <class extendT>
void HeadedScrollArea<extendT>::paintHeader()
{
  QPainter p( this );

  int fw = extendT::frameWidth();
  int w = extendT::width();

  int lw = 1; // left width
  int rw = 1; // right width

  QRect contentsRect( fw+lw, fw, w-(2*fw)-rw, m_headerHeight );

  p.drawPixmap( fw, fw, lw, m_headerHeight, m_leftPix );

  // Note: fillRect would not work, as it uses a global texture origin
  p.drawTiledPixmap( contentsRect, m_midPix );

  p.drawPixmap( w-fw-rw, fw, rw, m_headerHeight, m_rightPix );

  // Note: AlignHCenter (horiz only) looks better than AlignCenter
  p.setFont( m_headerFont );
  p.drawText( contentsRect, Qt::AlignHCenter, m_headerText );

  // Draw a thin gap between the header and the viewport
  p.setPen( extendT::palette().window().color() );
  p.drawRect( fw, fw+m_headerHeight+1, w-(2*fw), FIXED_GAP_HEIGHT );
}
