/*  -*- c++ -*-  (for Emacs)
 *
 *  qwpdoubleslider.cpp
 * 
 *  Created by Aidan Lane on Wed Feb 01 2006.
 *  Copyright (c) 2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "qwpdoubleslider.h"

#include <QApplication>
#include <QFocusEvent>

#define SLIDER_MIN    0
#define SLIDER_MAX    10000000 /* If the number is greater than this, you get problems:
				  1. sporadic control with the mouse scroll wheel
				  2. sometimes the slider would wrap
				  3. crash on the Mac with Qt 4.1 when drawing the ticks
			       */
#define SLIDER_RANGE  (SLIDER_MAX - SLIDER_MIN)


/*!
  \class QwpDoubleSlider

  \brief The QwpDoubleSlider widget provides a double-precision floating-point
         vertical or horizontal slider.

  Reasons for using an internal QSlider, rather than drawing the slider using the
  style and looking after the control ourself:
    1. There are lots of control issues, which differ across platforms - lots of work.
    2. The styles cheat a bit, as they check if a widget is actually a QSlider before
       they perfrom some operations.
*/


QwpDoubleSlider::QwpDoubleSlider( QWidget* parent )
  : QWidget(parent)
{
  init();
}

QwpDoubleSlider::QwpDoubleSlider( Qt::Orientation orientation, QWidget* parent )
  : QWidget(parent)
{
  init();
  setOrientation( orientation );
}


void QwpDoubleSlider::init()
{
  // Note: Don't use a layout to set the size of the slider - it's simple to
  //       resize and we don't want to show the layout in Qt Designer.
  m_slider = new QSlider( this );
  m_slider->resize( size() );

  setFocusPolicy( m_slider->focusPolicy() );
  setFocusProxy( m_slider );

  // Init with known and sane defaults
  m_value           = 0.0;
  m_min             = 0.0;
  m_max             = 99.99; // to match QDoubleSpinBox
  m_singleStep      = 1.0;
  m_pageStep        = 10.0;
  m_sliderPosition  = m_value;
  m_tickInterval    = 0.0;

  m_slider->setRange( SLIDER_MIN, SLIDER_MAX );

  m_slider->blockSignals( true );
  m_slider->setValue( doubleToSliderValue(m_value) );
  m_slider->setSingleStep( doubleToSliderValue(m_singleStep) );
  m_slider->setPageStep( doubleToSliderValue(m_pageStep) );
  // Note: We MUST set the tickInterval after the step widths, because if the
  //       interval is 0 (the default), then one of the step widths will be used.
  m_slider->setTickInterval( doubleToSliderValue(m_tickInterval) );
  m_slider->blockSignals( false );

  // Note: rangeChanged isn't needed, as the internal slider's range never changes
  connect( m_slider, SIGNAL(actionTriggered(int)), SIGNAL(actionTriggered(int)) );
  connect( m_slider, SIGNAL(sliderMoved(int)), SLOT(onSliderMoved(int)) );
  connect( m_slider, SIGNAL(sliderPressed()), SIGNAL(sliderPressed()) );
  connect( m_slider, SIGNAL(sliderReleased()), SIGNAL(sliderReleased()) );
  connect( m_slider, SIGNAL(valueChanged(int)), SLOT(onSliderValueChanged(int)) );
}


double QwpDoubleSlider::value() const {
  return m_value;  // more accurate than the slider's copy
}

void QwpDoubleSlider::setValue( double value )
{
  // Note: We block m_slider's valueChanged() (which calls onSliderValueChanged())
  //       and isssue our own, as the double may change without the integer changing.
  Q_ASSERT( m_slider );
  if ( value == m_value ) return;
  m_slider->blockSignals( true );
  m_slider->setValue( doubleToSliderValue(m_value = value) );
  m_slider->blockSignals( false );
  emit valueChanged( value );
}


double QwpDoubleSlider::minimum() const {
  return m_min;
}

double QwpDoubleSlider::maximum() const {
  return m_max;
}

void QwpDoubleSlider::setMinimum( double min ) {
  setRange( min, m_max );
}

void QwpDoubleSlider::setMaximum( double max ) {
  setRange( m_min, max );
}

/*!
  Sets the slider's minimum to \em min and its maximum to \em max.
  
  If \em max is smaller than \em min, \em min becomes the only legal value.
*/
void QwpDoubleSlider::setRange( double min, double max )
{
  // Note: m_slider's range remains constant
  m_min = min;
  m_max = ( min > max
	    ? min
	    : max );
  emit rangeChanged( m_min, m_max );
}


Qt::Orientation QwpDoubleSlider::orientation() const
{ Q_ASSERT(m_slider); return m_slider->orientation(); }

void QwpDoubleSlider::setOrientation( Qt::Orientation orientation )
{ Q_ASSERT(m_slider); m_slider->setOrientation(orientation); }


bool QwpDoubleSlider::hasTracking() const
{ Q_ASSERT(m_slider); return m_slider->hasTracking(); }

void QwpDoubleSlider::setTracking( bool enable )
{ Q_ASSERT(m_slider); m_slider->setTracking(enable); }


bool QwpDoubleSlider::invertedAppearance() const
{ Q_ASSERT(m_slider); return m_slider->invertedAppearance(); }

void QwpDoubleSlider::setInvertedAppearance( bool enable )
{ Q_ASSERT(m_slider); m_slider->setInvertedAppearance(enable); }


bool QwpDoubleSlider::invertedControls() const
{ Q_ASSERT(m_slider); return m_slider->invertedControls(); }

void QwpDoubleSlider::setInvertedControls( bool enable )
{ Q_ASSERT(m_slider); m_slider->setInvertedControls(enable); }


bool QwpDoubleSlider::isSliderDown() const
{ Q_ASSERT(m_slider); return m_slider->isSliderDown(); }

void QwpDoubleSlider::setSliderDown( bool enable )
{ Q_ASSERT(m_slider); m_slider->setSliderDown(enable); }


double QwpDoubleSlider::singleStep() const {
  return m_singleStep;  // more accurate than the slider's copy
}

void QwpDoubleSlider::setSingleStep( double step ) {
  Q_ASSERT( m_slider );
  m_slider->setSingleStep( doubleToSliderValue(m_singleStep = step) );
}


double QwpDoubleSlider::sliderPosition() const {
  return m_sliderPosition;  // more accurate than the slider's copy
}

void QwpDoubleSlider::setSliderPosition( double position )
{
  // Note: We block m_slider's sliderMoved() (which calls onSliderMoved()) and
  //       issue our own, as the double may change without the integer changing.
  Q_ASSERT( m_slider );
  if ( position == m_sliderPosition ) return;
  m_slider->blockSignals( true );
  m_slider->setSliderPosition( doubleToSliderValue(m_sliderPosition = position) );
  m_slider->blockSignals( false );
  emit sliderMoved( position );
}


double QwpDoubleSlider::pageStep() const {
  return m_pageStep;  // more accurate than the slider's copy
}

void QwpDoubleSlider::setPageStep( double step ) {
  Q_ASSERT( m_slider );
  m_slider->setPageStep( doubleToSliderValue(m_pageStep = step) );
}


QSlider::TickPosition QwpDoubleSlider::tickPosition() const
{ Q_ASSERT(m_slider); return m_slider->tickPosition(); }

void QwpDoubleSlider::setTickPosition( QSlider::TickPosition position )
{ Q_ASSERT(m_slider); m_slider->setTickPosition(position); }


double QwpDoubleSlider::tickInterval() const {
  return m_tickInterval;  // more accurate than the slider's copy
}

void QwpDoubleSlider::setTickInterval( double interval ) {
  Q_ASSERT( m_slider );
  m_slider->setTickInterval( doubleToSliderValue(m_tickInterval = interval) );
}


void QwpDoubleSlider::triggerAction( QSlider::SliderAction action ) {
  Q_ASSERT( m_slider );
  m_slider->triggerAction( action );
}


QSlider* QwpDoubleSlider::slider() const {
  return m_slider;
}


double QwpDoubleSlider::sliderValueToDouble( int value ) const {
  return (value - SLIDER_MIN) / (double)SLIDER_RANGE * (m_max - m_min) + m_min;
}


int QwpDoubleSlider::doubleToSliderValue( double value ) const {
  return (int) ( (value - m_min) / (m_max - m_min)
		 * (double)SLIDER_RANGE + (double)SLIDER_MIN );
}


void QwpDoubleSlider::onSliderMoved( int value ) {
  emit sliderMoved( m_sliderPosition = sliderValueToDouble(value) );
}

void QwpDoubleSlider::onSliderValueChanged( int value ) {
  emit valueChanged( m_value = sliderValueToDouble(value) );
}


QSize QwpDoubleSlider::sizeHint() const {
  Q_ASSERT( m_slider );
  return m_slider->sizeHint();
}

void QwpDoubleSlider::resizeEvent( QResizeEvent* event ) {
  Q_ASSERT( m_slider );
  QWidget::resizeEvent( event );
  m_slider->resize( size() );
}
