/*  -*- c++ -*-  (for Emacs)
 *
 *  searchwidget.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Aug 04 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "searchwidget.h"

#include <QHBoxLayout>
#include <QLineEdit>
#include <QMacStyle>
#include <QMenu>
#include <QPainter>

#include "simpleicontoolbutton.h"

#define WIDGET_HEIGHT 22
#define BUTTON_WIDTH  22


SearchWidget::SearchWidget( QWidget* parent )
  : QWidget(parent),
    m_searchTriggers(SearchWidget::AllTriggers)
{
  QHBoxLayout* layout = new QHBoxLayout( this );
  layout->setMargin( 0 );
  layout->setSpacing( 0 );
  
  m_leftCapPixmap = QPixmap( ":/images/SearchLeft.png" );
  m_middleTilePixmap = QPixmap( ":/images/SearchMiddle.png" );
  m_rightCapPixmap = QPixmap( ":/images/SearchRight.png" );

  m_searchButton = new SimpleIconToolButton( this );
  m_searchButton->setIcon( QIcon(":/images/SearchPopup.png") );
  m_searchButton->setPopupMode( QToolButton::InstantPopup );

  m_lineEdit = new QLineEdit( this );
  m_lineEdit->setFrame( false );
  m_lineEdit->setMinimumHeight( WIDGET_HEIGHT ); // to suit the given pixmaps
  m_lineEdit->setMaximumHeight( WIDGET_HEIGHT ); // ditto.
  QPalette editPal = m_lineEdit->palette();
  editPal.setBrush( QPalette::Base, QBrush(Qt::NoBrush) ); // we'll draw the base
  m_lineEdit->setPalette( editPal );
  connect( m_lineEdit,
	   SIGNAL(returnPressed()), SLOT(onReturnPressed()) );
  connect ( m_lineEdit,
	    SIGNAL(textEdited(const QString&)),
	    SLOT(onTextEdited(const QString&)) );

  m_clearButton = new SimpleIconToolButton( this );
  m_clearButton->setToolTip( tr("Erase search text.") );
  QIcon clearIcon( QIcon(":/images/SearchClearNormal.png") );
  clearIcon.addPixmap( QPixmap(":/images/SearchClearPressed.png"), QIcon::Active );
  m_clearButton->setIcon( clearIcon );
  connect( m_clearButton,
	   SIGNAL(clicked()), SLOT(clear()) );

  layout->addSpacing( 6 ); // make room for 6px of left cap past search button
  layout->addWidget( m_searchButton );
  layout->addWidget( m_lineEdit );
  layout->addWidget( m_clearButton );
  layout->addSpacing( 4 ); // make room for 4px of right cap past clear button

#ifdef Q_WS_MAC
  QMacStyle::setFocusRectPolicy( m_lineEdit, QMacStyle::FocusDisabled );
#endif

  m_clearButton->hide(); // hidden until the user enters some text
}


QMenu* SearchWidget::menu() const
{
  Q_ASSERT( m_searchButton );
  return m_searchButton->menu();
}


/*!
 * The menu may be set to null (0) to remove any existing menu.
 */
void SearchWidget::setMenu( QMenu* menu )
{
  Q_ASSERT( m_searchButton );

  // Disconnect from the old menu (if there was one).
  // We're being safe by calling disconnect unconditionally.
  if ( m_searchButton->menu() )
    m_searchButton->menu()->disconnect( this );

  m_searchButton->setMenu( menu );

  // Try to connect to the the new menu (if there is one).
  // When user changes search type, a new search will begin (standard in MacOSX).
  if ( m_searchTriggers & SearchWidget::OnMenuActionTriggered
       && m_searchButton->menu() )
    connect( m_searchButton->menu(),
	     SIGNAL(triggered(QAction*)), SLOT(search()) );
}


/*! Defaults to SearchWidget::AllTriggers. */
SearchWidget::SearchTriggers SearchWidget::searchTriggers() const {
  return m_searchTriggers;
}


void SearchWidget::setSearchTriggers( SearchTriggers triggers )
{
  m_searchTriggers = triggers;

  // Note: onReturnPressed() and onTextEdited() handle some of the triggers.

  // Calling setMenu updates the connections to them menu based on the triggers.
  Q_ASSERT( m_searchButton );
  setMenu( m_searchButton->menu() );
}


void SearchWidget::search()
{
  Q_ASSERT( m_lineEdit );
  emit beginSearch( m_lineEdit->text() );
}


void SearchWidget::clear()
{
  Q_ASSERT( m_clearButton );
  m_clearButton->hide();

  Q_ASSERT( m_lineEdit );
  m_lineEdit->clear();

  emit endSearch();
}


/*!
 * \b Note: If the \em text is empty, then clear() will be called instead.
 */
void SearchWidget::setText( const QString& text, bool startSearch )
{
  if ( text.isEmpty() ) {
    clear();
  }
  else {
    Q_ASSERT( m_clearButton );
    Q_ASSERT( m_lineEdit );
    m_clearButton->setVisible( ! text.isEmpty() );
    m_lineEdit->setText( text );
    if ( startSearch ) search();
  }
}


void SearchWidget::onReturnPressed()
{
  if ( m_searchTriggers & SearchWidget::OnReturnPressed )
    search();
}


void SearchWidget::onTextEdited( const QString& text )
{
  Q_ASSERT( m_clearButton );
  m_clearButton->setVisible( ! text.isEmpty() );

  if ( m_searchTriggers & SearchWidget::OnTextEdited )
    search();
}


void SearchWidget::paintEvent( QPaintEvent* )
{
  Q_ASSERT( m_lineEdit );
  const QRect& editRect   = m_lineEdit->geometry();

  // TODO: cleanup & optimise me!
  QPainter p( this );
  p.drawPixmap( 0, editRect.top(), m_leftCapPixmap );
  p.drawTiledPixmap( m_leftCapPixmap.width(), editRect.top(),
		     width()-m_leftCapPixmap.width()-m_rightCapPixmap.width(),
		     WIDGET_HEIGHT,
		     m_middleTilePixmap );
  p.drawPixmap( width()-m_rightCapPixmap.width(), editRect.top(),
		m_rightCapPixmap );
}


QLineEdit* SearchWidget::lineEdit() const
{ return m_lineEdit; }

SimpleIconToolButton* SearchWidget::searchButton() const
{ return m_searchButton; }

SimpleIconToolButton* SearchWidget::clearButton() const
{ return m_clearButton; }
