/*  -*- c++ -*-  (for Emacs)
 *
 *  simpleiconbutton.h
 *
 *  Created by Aidan Lane on Thu Apr 14 2005.
 *  Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */


#ifndef SIMPLEICONBUTTON_H
#define SIMPLEICONBUTTON_H


#include <QStylePainter>

#define MAX_SIMPLEICONBUTTON_WIDTH   1024
#define MAX_SIMPLEICONBUTTON_HEIGHT  1024


/*!
 * \brief The SimpleIconButton class wraps around an existing Qt button to provide
 *        a button that only draws its icon (i.e. without a frame).
 *
 * Type \em T must be a derivative of QAbstractButton, such as QPushButton or
 * QToolButton.
 */
template <class T>
class SimpleIconButton : public T {

public:
  SimpleIconButton( QWidget* parent = 0 ) : T(parent) {
    T::setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed ); // we respect the icon, which as a fixed size (layout uses sizeHint)
  }

  virtual ~SimpleIconButton() {}

  const QIcon& backgroundIcon() const { return m_backgroundIcon; }

  void setBackgroundIcon( const QIcon& icon ) {
    m_backgroundIcon = icon;
    T::update();
  }


  /*!
   * We use the size of the icon's normal-off pixmap.
   */
  virtual QSize sizeHint() const
  {
    // From the Qt4.0.0 Doc for actualSize(): "The result might be smaller than requested, but never larger."
    return ( T::icon().actualSize(QSize(MAX_SIMPLEICONBUTTON_WIDTH,
					MAX_SIMPLEICONBUTTON_HEIGHT))
	     .expandedTo(m_backgroundIcon.actualSize(QSize(MAX_SIMPLEICONBUTTON_WIDTH,
							   MAX_SIMPLEICONBUTTON_HEIGHT))) );
  }


protected:
  // TODO: optimise me! possibly introduce caching...
  virtual void paintEvent( QPaintEvent* )
  {
    QStylePainter p( this ); // assume we're using the screen -> no error checking

    // Get the appropriate pixmap from the icon
    // WRANING! DON'T use icon().paint(), as it won't actually scale the icon,
    //          it just clips it. Also, it seems like it doesn't draw until
    //          the entire app has loaded and is idling (very noticeable).
    // TODO: cache the pixmaps!

    int w = T::width();
    int h = T::height();

    QIcon::Mode mode = QIcon::Disabled;
    if ( T::isEnabled() )
      mode = T::isDown() ? QIcon::Active : QIcon::Normal;
    QPixmap fgPixmap = T::icon().pixmap( QSize(w, h),                   // TODO: cleanup!
					 mode,
					 T::isChecked() ? QIcon::On : QIcon::Off );

    QPixmap bgPixmap = m_backgroundIcon.pixmap( QSize(w, h),                   // TODO: cleanup!
						mode,
						T::isChecked() ? QIcon::On : QIcon::Off );

    // Draw the icon pixmaps centred within us
    // TODO: cache positions!
    p.drawPixmap( (w - bgPixmap.width()) / 2,
		  (h - bgPixmap.height()) / 2,
		  bgPixmap );
    p.drawPixmap( (w - fgPixmap.width()) / 2,
		  (h - fgPixmap.height()) / 2,
		  fgPixmap );

    // Draw the text last, on-top of the icons
    p.drawItemText( QRect(0, 0, w, h), Qt::AlignCenter | Qt::TextShowMnemonic,
		    T::palette(), T::isEnabled(), T::text() );
  }


private:
  QIcon m_backgroundIcon;
};


#endif // ! SIMPLEICONBUTTON_H
