/*  -*- c++ -*-  (for Emacs)
 *
 *  extrafeatures.cpp
 *  Digest
 * 
 *  Created by Adrian Bickerstaffe on 10/5/2006.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <algorithm>
#include <iostream>
#include <cassert>
#include <vector>
#include <cmath>

#include "extrafeatures.h"
#include "../SvmRecogniser/librecognizer/common.h"

using namespace std;

typedef StlFeatureResultT FeatureCalcT;

// maximum angle three points can make and still be considered a potential cusp
#define MAX_CUSP_ANGLE			30
// minimum angle the points surrounding a cusp must make for the cusp to
// be confirmed
#define MIN_CUSP_CONFIRM		170

///////////////////////////////

// Default constructor sets both coordinate variables to zero.
hull_vertex::hull_vertex(void)
{
	x = 0; y = 0;
}

// Constructor to set coordinate variables to parameter values.
hull_vertex::hull_vertex(const float & the_x, const float & the_y)
{
	x = the_x;
	y = the_y;
}

// Function used for sorting convex hull vertices according to their
// x-coordinates.
bool vertex_x_compare(const hull_vertex & a, const hull_vertex & b)
{
	return a.x < b.x;
}

// Function used for sorting convex hull vertices according to their
// x-coordinates.
bool vertex_y_compare(const hull_vertex & a, const hull_vertex & b)
{
	return a.y < b.y;
}

////////////////////

// Destructor to clean up array memory.
convex_hull::~convex_hull(void)
{
	if(hull_vertices != NULL)
		delete[] hull_vertices;
	if(stroke_vertices != NULL)
		delete[] stroke_vertices;
}

// Constructor takes a stroke parameter and creates a convex hull around
// this stroke.
convex_hull::convex_hull(const StlStroke & the_stroke)
{
	int i;
	StlStroke::const_iterator iter;
	hull_vertices = NULL; stroke_vertices = NULL;
	
	cout << "Stroke size: " << the_stroke.size() << endl;
	// create the array of convex hull vertices (max. required is the_stroke.size())
	hull_vertices = new(nothrow) hull_vertex[the_stroke.size()];
	// create the array to store stroke points
	stroke_vertices = new(nothrow) hull_vertex[the_stroke.size()];
	// check memory was successfully allocated 
	if(hull_vertices == NULL || stroke_vertices == NULL)
	{
		cout << "Error:- could not allocate vertex memory during convex hull creation!"
			 << endl;
		return;
	}
	
	// for each point in the stroke
	for(i = 0, iter = the_stroke.begin(); iter != the_stroke.end(); iter++, i++)
	{	
		// add the point (in the form of a hull vertex) to the array
		hull_vertex new_vertex( (*iter).x, (*iter).y );
		stroke_vertices[i] = new_vertex;
		cout << "x: " << (*iter).x << ", y: " << (*iter).y << endl;
	}
	// sort the stroke vertices by x-coords and then y-coords
	sort(stroke_vertices, stroke_vertices + the_stroke.size(), vertex_x_compare);
	sort(stroke_vertices, stroke_vertices + the_stroke.size(), vertex_y_compare);
	// create the convex hull and the length of the hull	
	num_vertices = create_hull(stroke_vertices, the_stroke.size(), hull_vertices);
	cout << "Number of hull vertices: " << num_vertices << endl << endl;
//  !!!!! FIXME !!!!!!!
//	perimeter_length = calc_perimeter_length();
//	area = calc_area();
	perimeter_length = 1;
	area = 1;
}

// Function to create a convex hull around stroke_vertices.
// Andrew's monotone chain 2D convex hull algorithm, see:
// http://softsurfer.com/Archive/algorithm_0109/algorithm_0109.htm
// Input:  P[] = an array of 2D points 
//               presorted by increasing x- and y-coordinates
//         n = the number of points in P[]
// Output: H[] = an array of the convex hull vertices (max is n)
// Return: the number of points in H[]
int convex_hull::create_hull(hull_vertex *stroke_vertices, int n,
							 hull_vertex *hull_vertices)							  
{
    // the output array hull_vertices[] will be used as the stack
    int bot=0, top=(-1);  // indices for bottom and top of the stack
    int i;                // array scan index

    // get the indices of points with min x-coord and min|max y-coord
    int minmin = 0, minmax;
    float xmin = stroke_vertices[0].x;
    for (i=1; i<n; i++)
        if (stroke_vertices[i].x != xmin) break;
    minmax = i-1;
	// degenerate case: all x-coords == xmin
    if (minmax == n-1) {
        hull_vertices[++top] = stroke_vertices[minmin];
		// a nontrivial segment
        if (stroke_vertices[minmax].y != stroke_vertices[minmin].y)
            hull_vertices[++top] = stroke_vertices[minmax];
		// add polygon endpoint
        hull_vertices[++top] = stroke_vertices[minmin];
        return top+1;
    }

    // get the indices of points with max x-coord and min|max y-coord
    int maxmin, maxmax = n-1;
    float xmax = stroke_vertices[n-1].x;
    for (i=n-2; i>=0; i--)
        if (stroke_vertices[i].x != xmax) break;
    maxmin = i+1;

    // compute the lower hull on the stack hull_vertices
	// push minmin point onto stack
    hull_vertices[++top] = stroke_vertices[minmin];
    i = minmax;
    while (++i <= maxmin)
    {
        // the lower line joins stroke_vertices[minmin] with stroke_vertices[maxmin]
        if (is_left(stroke_vertices[minmin], stroke_vertices[maxmin], stroke_vertices[i]) >= 0 &&
			i < maxmin)
            continue;          // ignore stroke_vertices[i] above or on the lower line		
		// there are at least 2 points on the stack
        while (top > 0)       
        {
            // test if stroke_vertices[i] is left of the line at the stack top
            if (is_left( hull_vertices[top-1], hull_vertices[top], stroke_vertices[i]) > 0)
                break;         // stroke_vertices[i] is a new hull vertex
            else
                top--;         // pop top point off stack
        }
		// push stroke_vertices[i] onto stack
        hull_vertices[++top] = stroke_vertices[i];
    }

    // next, compute the upper hull on the stack hull_vertices above the bottom hull
    if(maxmax != maxmin)      // if distinct xmax points
        hull_vertices[++top] = stroke_vertices[maxmax];  // push maxmax point onto stack
	// the bottom point of the upper hull stack
    bot = top;                 
    i = maxmin;
    while (--i >= minmax)
    {
        // the upper line joins stroke_vertices[maxmax] with stroke_vertices[minmax]
        if(is_left(stroke_vertices[maxmax], stroke_vertices[minmax], stroke_vertices[i]) >= 0 &&
			i > minmax)
            continue;          // ignore stroke_vertices[i] below or on the upper line
		// at least 2 points on the upper stack
        while (top > bot)
        {
            // test if stroke_vertices[i] is left of the line at the stack top
            if(is_left(hull_vertices[top-1], hull_vertices[top], stroke_vertices[i]) > 0)
                break;         // stroke_vertices[i] is a new hull vertex
            else
                top--;         // pop top point off stack
        }
		// push stroke_vertices[i] onto stack
        hull_vertices[++top] = stroke_vertices[i];
    }
	// push joining endpoint onto stack
    if (minmax != minmin)
        hull_vertices[++top] = stroke_vertices[minmin];

    return top + 1;
}

// Function tests if a point is Left|On|Right of an infinite line.
// Input:  three points P0, P1, and P2
// Return: > 0 for P2 left of the line through P0 and P1
//         = 0 for P2 on the line
//         < 0 for P2 right of the line
float convex_hull::is_left(hull_vertex P0, hull_vertex P1, hull_vertex P2)
{
	return (P1.x - P0.x) * (P2.y - P0.y) - (P2.x - P0.x) * (P1.y - P0.y);
}

// Function returns the Euclidean distance travelled around the perimeter of
// the convex hull.
double convex_hull::calc_perimeter_length(void)
{
	int i;
	double length = 0.0;
	hull_vertex prev_vertex = hull_vertices[0], curr_vertex;
	// for each edeg
	for(i = 1; i < num_vertices; i++)
	{
		curr_vertex = hull_vertices[i];
		// calculate the length of the edge and it to the accumulated
		// length result
		length += hypot(curr_vertex.x - prev_vertex.x, 
						curr_vertex.y - prev_vertex.y);
						
		prev_vertex = curr_vertex;
	}
	
	return length;
}

// Function calculates the area of the convex hull. This source code is taken
// from http://astronomy.swin.edu.au/~pbourke/geometry/polyarea/.
double convex_hull::calc_area(void)
{
   int i, j;
   double result = 0.0;

   for(i = 0; i < num_vertices; i++)
   {
		j = (i + 1) % num_vertices;
		result += hull_vertices[i].x * hull_vertices[j].y;
		result -= hull_vertices[i].y * hull_vertices[j].x;
	}

   result /= 2.0;
   return (result < 0 ? -result : result);
}

// Function returns the length of the convex hull's perimeter.
double convex_hull::get_perimeter_length(void) const
{
	return perimeter_length;
}

// Function returns the area of the convex hull.
double convex_hull::get_area(void) const
{
	return area;
}

// Function to output details about the convex hull.
void convex_hull::print(void) const
{
	int i;
	
	for(i = 0; i < num_vertices; i++)
		cout << i + 1 << ":\t" << hull_vertices[i].x << ", " 
			 << hull_vertices[i].y << endl;
	cout << endl;
	
	cout << "Perimeter length: " << perimeter_length << endl
		 << "Area: " << area << endl << endl;
}

///////////////////////////////


StlFeatureResultT relative_length::classCalcValue(const StlStroke & stroke, bool* ok)
{
	*ok = true;
	return calc_relative_length(stroke);
}

const std::string & relative_length::classDescription(void)
{
	static const string str("Calculate the relative length of a single stroke.  "
	"The relative distance is the ratio of the stroke length to the distance between "
	"the first and last points of the stroke."
	"<br><br>Adrian Bickerstaffe, 10/02/2006");
	
	return str;
}

///////////////

StlFeatureResultT general_density::classCalcValue(const StlStroke & stroke, bool* ok)
{
	// run length of the stroke measured as Euclidean distance
	double stroke_length = calc_length(stroke);
	// create a bounding box around the stroke
	bounding_box bbox(stroke);
	double diagonal_length = bbox.diagonal_length();
	
	// avoid divide by zero error when calculating the ratio
	if(diagonal_length != 0)
	{
		*ok = true;
		return stroke_length / diagonal_length;
	}
	else
	{
		*ok = false;
		return 0;
	}
}

const std::string & general_density::classDescription(void)
{
	static const string str("Calculate the density of a stroke.  Density is defined "
	" here as the ratio of stroke length to the diagonal length of the bounding box."
	"<br><br>Adrian Bickerstaffe, 10/02/2006");
	
	return str;
}

///////////////

StlFeatureResultT density_spread::classCalcValue(const StlStroke & stroke, bool* ok)
{
	bounding_box bbox(stroke);
	*ok = true;
	return bbox.classify_point_density(stroke);
}

const std::string & density_spread::classDescription(void)
{
	static const string str("Point density is quantized into three categories: 1) even "
	"distribution of points, 2) more points in the upper section than the lower section, "
	"3) more points in the lower section than the upper section."
	"<br><br>Adrian Bickerstaffe, 10/02/2006");
	
	return str;
}

///////////////

StlFeatureResultT abs_coord_change::classCalcValue(const StlStroke & stroke, bool* ok)
{
	// absolute delta_x and delta_y
	double abs_delta_x = 0.0, abs_delta_y = 0.0;
	StlStroke::const_iterator iter = stroke.begin();
	StlStrokePoint prev_point = *iter, curr_point;

	// for each stroke point
	for(iter++; iter != stroke.end(); iter++)
	{
		curr_point = *iter;	
		// accumulate the absolute difference in each direction	
		abs_delta_x += abs(curr_point.x - prev_point.x);
		abs_delta_y += abs(curr_point.y - prev_point.y);
		prev_point = curr_point;
	}
	
	// avoid divide by zero error when calculating the ratio
	if(abs_delta_x != 0)
	{
		*ok = true;
		return abs_delta_y / abs_delta_x;
	}
	else
	{
		*ok = false;
		return 0;
	}
}

const std::string & abs_coord_change::classDescription(void)
{
	static const string str("Calculate the ratio sum(delta_y * delta_y) : "
	"sum(delta_x * delta_x)"
	"<br><br>Adrian Bickerstaffe, 10/02/2006");
	
	return str;
}

///////////////

StlFeatureResultT path_ratio::classCalcValue(const StlStroke & stroke, bool* ok)
{
	// calculate the length of the stroke
	double stroke_length = calc_length(stroke);
	// create a convex hull around the stroke
	convex_hull stroke_hull(stroke);
	
	// avoid divide by zero error when calculating the ratio
	if(stroke_length != 0)
	{
		*ok = true;
		return stroke_hull.get_perimeter_length() / stroke_length;
	}
	else
	{
		*ok = false;
		return 0;
	}
}

const std::string & path_ratio::classDescription(void)
{
	static const string str("Calculate the ratio of the convex hull perimeter "
	"length to the stroke length."
	"<br><br>Adrian Bickerstaffe, 10/02/2006");
	
	return str;
}

///////////////

StlFeatureResultT area_ratio::classCalcValue(const StlStroke & stroke, bool* ok)
{
	// create a bounding box around the stroke
	bounding_box bbox(stroke);
	double bbox_area = bbox.area();
	// create a convex hull around the stroke
	convex_hull stroke_hull(stroke);
	
	// avoid divide by zero error when calculating the ratio
	if(bbox_area != 0)
	{
		*ok = true;
		return stroke_hull.get_area() / bbox_area;
	}
	else
	{
		*ok = false;
		return 0;
	}
}

const std::string & area_ratio::classDescription(void)
{
	static const string str("Calculate the ratio of the convex hull area to the "
	"area enclosed by the bounding box."
	"<br><br>Adrian Bickerstaffe, 10/02/2006");
	
	return str;
}

///////////////
 
StlFeatureResultT start_quadrant::classCalcValue(const StlStroke & stroke, bool* ok)
{
	bounding_box bbox(stroke);
	*ok = true;
	return bbox.find_quadrant(stroke.front());
}

const std::string & start_quadrant::classDescription(void)
{
	static const string str("The quadrant of the bounding box which contains the "
	"first point of a stroke."
	"<br><br>Adrian Bickerstaffe, 10/02/2006");
	
	return str;
}

///////////////

StlFeatureResultT end_quadrant::classCalcValue(const StlStroke & stroke, bool* ok)
{
	bounding_box bbox(stroke);
	*ok = true;
	return bbox.find_quadrant(stroke.back());
}

const std::string & end_quadrant::classDescription(void)
{
	static const string str("The quadrant of the bounding box which contains the "
	"last point of a stroke."
	"<br><br>Adrian Bickerstaffe, 10/02/2006");
	
	return str;
}

///////////////

StlFeatureResultT num_cusps::classCalcValue(const StlStroke & stroke, bool* ok)
{
	// no cusps detected initially
	int num_cusps = 0;
	StlStroke::const_iterator iter = stroke.begin();
	// point neighbourhood for a potential cusp
	StlStrokePoint p0, p1, p2, p3, p4;
	// must have at least five points to detect a cusp
	if(stroke.size() < 5)
		return 0;
	// initialise the set of points to those leading the stroke
	p0 = *iter;	iter++;
	p1 = *iter; iter++;
	p2 = *iter; iter++;
	p3 = *iter;
	// for every position in the stroke
	for(iter++; iter != stroke.end(); iter++)
	{
		p4 = *iter;
		// calculate the angle between the points
		double curr_angle = inside_angle(p0, p1, p2);
		// check for a potential cusp
		if(curr_angle <= MAX_CUSP_ANGLE)
		{
			// if the surrounding points confirm a cusp, record it
			if(inside_angle(p0, p1, p2) >= MIN_CUSP_CONFIRM &&
			   inside_angle(p2, p3, p4) >= MIN_CUSP_CONFIRM)
			   num_cusps++;
		}
		
		// move points along by one position
		p0 = p1; p1 = p2; p2 = p3; p3 = p4;
	}
	
	return num_cusps;
}

const std::string & num_cusps::classDescription(void)
{
	static const string str("Calculate the number of cusps present in "
	"a given stroke.  A cusp is loosely defined as a point at which the local line"
	" segments entering and exiting the point have roughly the same angle of incidence."
	"<br><br>Adrian Bickerstaffe, 10/02/2006");
	
	return str;
}