/*  -*- c++ -*-  (for Emacs)
 *
 *  linearrecogniser.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Mon Jul 11 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "linearrecogniser.h"

#include <QByteArray>
#include <QFile>
#include <QStringList>
#include <QTextStream>


LinearRecogniser::LinearRecogniser( JavaVM* jvm )
  : AbstractRecogniser(jvm)
{}


bool LinearRecogniser::readModelFile( const QString& fileName )
{
  QFile file( fileName );
  if ( ! file.open(QIODevice::ReadOnly | QIODevice::Text) )
    return false;

  QTextStream in( &file );
  while ( ! in.atEnd() )
    {
      bool ok = false;

      const QString line = in.readLine();
      if ( line.isEmpty() ) continue;

      const QStringList keyAndValue = line.split( "=", QString::SkipEmptyParts );
      if ( keyAndValue.size() != 2 ) return false;

      int key = keyAndValue.at(0).toInt(&ok);
      if ( !ok ) return false;

      // TODO: optimise me! - it's not good to append to vectors!
      QVector<WeightT> weights;
      const QStringList weightStrs = keyAndValue.at(1).split(",", QString::SkipEmptyParts);
      foreach( const QString& wStr, weightStrs ) {
	weights.append( wStr.toDouble(&ok) );
	if ( !ok ) return false;
      }

      m_classWeights.insert( key, weights );
  }

  file.close();

  return true;
}



ClassProbabilities LinearRecogniser::classifyGestureImp( const DGestureRecord& /*gesture*/,
						     const QVector<FeatureResultT>& featureVec )
{
  ClassProbabilities results;
  ClassProbabilityT resultsSum = 0.0;
  ClassProbabilityT negValue = 0.0; // init this to 0 -> no neg value
  int numFeatures = featureVec.size();

  // Step 1 - Fill the results hash with our internal scores
  QHashIterator< int, QVector<WeightT> > c(m_classWeights);
  while ( c.hasNext() )
    {
      c.next();

      const QVector<WeightT>& w = c.value();
      ClassProbabilityT v = w.at(0);
      for ( int i=0; i < numFeatures; ++i )
	v += w.at(i+1) * featureVec.at(i); // note: using w_1 to w_F -> w.at(i+1)

      results.insert( c.key(), v );
      resultsSum += v;

      if ( v < negValue ) negValue = v;
    }

  // Step 2 - Turn the internal scores in the results hash in to confience values
  resultsSum -= negValue * (ClassProbabilityT)results.size();
  if ( resultsSum != 0 ) { // TODO: use fabs() ?
    MutableClassProbabilitiesIterator r(results);
    while ( r.hasNext() ) {
      r.next();
      r.setValue( (r.value()-negValue) / resultsSum );
    }
  }

  return results;
}
