/*  -*- c++ -*-  (for Emacs)
 *
 *  stlrecogniserinterface.h
 *  Digest
 * 
 *  Created by Aidan Lane on Tue Jan 17 2006.
 *  Copyright (c) 2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef STLRECOGNISERINTERFACE_H
#define STLRECOGNISERINTERFACE_H


#include <algorithm> // for std::copy
#include <list>
#include <map>
#include <set>
#include <string>

#include "stlrecognitioncommon.h"


#ifndef DECLARE_STL_RECOGNISER_KEY
#define DECLARE_STL_RECOGNISER_KEY(KEY)				\
  static const std::string& classKey() {			\
    static const std::string str(KEY);				\
    return str;							\
  }								\
  virtual const std::string& key() const { return classKey(); }
#endif

#ifndef DECLARE_STL_RECOGNISER_TITLE
#define DECLARE_STL_RECOGNISER_TITLE(TITLE)				\
  static const std::string& classTitle() {				\
    static const std::string str(TITLE);				\
    return str;								\
  }									\
  virtual const std::string& title() const { return classTitle(); }
#endif

#ifndef DECLARE_STL_RECOGNISER_DESCRIPTION
#define DECLARE_STL_RECOGNISER_DESCRIPTION(DESCRIPTION)			\
  static const std::string& classDescription() {			\
    static const std::string str(DESCRIPTION);				\
    return str;								\
  }									\
  virtual const std::string& description() const { return classDescription(); }
#endif



/*!
  \brief The StlRecogniserInterface class is an abstract recogniser base class
         that is well suited to plugins, as it does not require Qt, only the
         C++ STL.
 
  Note that this class does not call "using namespace std", as that may not be
  wanted by the recogniser implementer. Hence, in the places where STL containers
  are used, they are prefixed with "std::".
 */
class StlRecogniserInterface {

public:
  virtual ~StlRecogniserInterface() {}

  virtual const std::string& key() const = 0;
  virtual const std::string& title() const = 0;
  virtual const std::string& description() const = 0;

  virtual const std::map<std::string, std::string>& defaultParams() const = 0;

  virtual bool initTraining( const std::list<std::string>& featureKeys,
			     const std::map<std::string, std::string>& params ) = 0;
  virtual bool examineSample( const StlFeatureVec& featureVec,
			      const std::set<int>& classes ) = 0;
  virtual bool finaliseTraining() = 0;

  virtual bool writeModelFile( const std::string& fileName ) = 0;
  virtual bool readModelFile( const std::string& fileName ) = 0;

  virtual StlStroke flatten( const StlStrokeList& strokes ) {
    StlStroke stroke;
    StlStrokeList::const_iterator it = strokes.begin();
    while ( it != strokes.end() ) {
      std::copy( (*it).begin(), (*it).end(), back_inserter(stroke) );
      ++it;
    }
    return stroke;
  }

  virtual StlClassProbabilities classify( const StlFeatureVec& featureVec ) = 0;
};


/*!
  \fn StlStroke StlRecogniserInterface::flatten( const StlStrokeList& strokes )

  This is used to flatten/link/join/merge a (potentially) multi-stroke gesture
  before feature extraction is performed (and thus examination or classification).

  If any filtering needs to be performed on the stroke(s), then it should be done
  by this method. This includes smoothing the stroke(s) or even reducing the
  number of points in them.

  This default implementation simply joins all of the strokes together.
  It does not join strokes together based on the location of their start and end
  points.

  \b Note: This is non-const, as the recogniser may need to examine the
           \em stroke's structure and record information about it, ready
           for use by either classify() or examineSample().
*/


#endif  // ! STLRECOGNISERINTERFACE_H
