/*  -*- c++ -*-  (for Emacs)
 *
 *  svm_recogniser.ipp
 *  Digest
 * 
 *  Created by Adrian Bickerstaffe on Wed Jan 18 20		06.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.	
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef SVM_RECOGNISER_IPP
#define SVM_RECOGNISER_IPP

#include <cmath>
#include <cassert>

#include "common.h"	

using namespace std;

// minimum direction change at the start/end of a stroke to be
// considered a hooklet
#define MIN_HOOK_ANGLE			60
// minimum number of points a stroke must have for hook detected to
// be performed
#define MIN_HOOK_POINTS			20
// minumum number of pixels between stroke points before being classed
// as ``jitter''
#define JITTER_DISTANCE			3.0
// uniform Euclidean separation between points of the equidistant
// resampled stroke
#define EQUIDISTANT_THRESHOLD	5.0
// reversal threshold [0, 1] (0.37 taken from E-Chalk experimental setup)
#define	REVERSAL_DELTA			0.37
// number of "in-recogniser" features
#define NUM_ADDED_FEATURES		0

// remove one in every SUBSAMPLE_RATE points in a stroke so that Dunnart
// and GestureLab stroke resolution is more closely matched
#define SUBSAMPLE_RATE	5
// empirical estimate of the time between GestureLab samples (in milliseconds)
#define GESTURELAB_TIME_BETWEEN_SAMPLES 9

///// Template class implementations /////

// Default constructor opens a file to store debugging information.
template<class T>
svm_recogniser<T>::svm_recogniser(void)
{
#ifdef DEBUG_RECOGNIZER	
	gesture_num = 1;
	debug_file.open("DunnartDebug.txt");
	// could not create output file
	if(!debug_file)
	{
		cout << "Error:- couldnt not create output file for recognizer debugging!"
			 << endl;
		exit(1);
	}
#endif
}

// Destructor closes the debug file if one had been opened for writing.
template<class T>
svm_recogniser<T>::~svm_recogniser(void)
{
#ifdef DEBUG_RECOGNIZER	
	debug_file.close();
#endif
}

// Function returns the default parameters of the recogniser.
template<class T>
const map<string, string> & svm_recogniser<T>::defaultParams(void) const
{
	// return an empty map since no parameters are applicable
	static map<string, string> p;
	return p;
}

// Function to initialise the recogniser prior to training.
template<class T>
bool svm_recogniser<T>::initTraining(const list<string> & featureKeys,
									 const map<string, string> & params)
{
	// tell the dataset how many features per sample to expect
	training_dataset.set_num_features(featureKeys.size() + NUM_ADDED_FEATURES);
	return true;
}

// Function to examine a single sample during training.  This function
// simply adds the sample to the training dataset.
template<class T>
bool svm_recogniser<T>::examineSample(const StlFeatureVec & featureVec,
									  const set<int> & classes)
{
	StlFeatureVec updated_features = add_features(featureVec);
	print_features(updated_features, cout);
	
	set<int>::const_iterator iter;		
	// add the sample to each class it belongs
	for(iter = classes.begin(); iter != classes.end(); iter++)
		training_dataset.add_sample(*iter, updated_features);

	return true;
}

// Function to finalise the training process once all samples have been
// recorded.  This function simply trains the SVM model using the dataset
// setup by examineSample().
template<class T>
bool svm_recogniser<T>::finaliseTraining(void)
{
	// scale the features of the dataset
	training_dataset.scale();
	training_dataset.print();
//	training_dataset.save("/Users/adrian/saved_dataset.txt");

	// train the model using the scaled dataset
	svm_model.train(training_dataset);
	
	cout << "SVM model details:" << endl;
	svm_model.print();
	
	return true;
}

// Function to classify a sample using a trained SVM model.
template<class T>
StlClassProbabilities svm_recogniser<T>::classify(
										 const StlFeatureVec & featureVec)
{
	// add any in-recogniser features to the sample
	StlFeatureVec updated_features = add_features(featureVec);

	// predict the probability of the sample belonging to each
	// of the target classes
	StlClassProbabilities classification_probs;
	svm_model.classify_with_leaf(updated_features, T::unscaled,
								 classification_probs);
	
	print_classification_probs(classification_probs, cout);
	return classification_probs;
}

// Function to classify a sample using a trained SVM model.  This version of
// the function returns a single most likely prediction.
template<class T>
int svm_recogniser<T>::classifyWithArray(const float * stroke_data, int stroke_data_size)
{
	// reconstruct the strokes from an array of point data
	StlStrokeList strokes = arrayToStrokes(stroke_data, stroke_data_size);
//	print_gesture(strokes, cout);
	// post-process the stroke
	StlStroke flattened_stroke = flatten(strokes);
	// extract default features from the processed stroke
	StlFeatureVec features = extract_features(flattened_stroke);
	// add any in-recogniser features to the sample
	StlFeatureVec updated_features = add_features(features);
//	print_features(updated_features, cout);

	// predict the probability of the sample belonging to each
	// of the target classes
	return static_cast<int>(svm_model.classify_sample(updated_features, T::unscaled));
}

// Function to classify a sample using a trained SVM model.  This version of
// the function returns an array of class Ids and prediction probabilities.
template<class T>
float * svm_recogniser<T>::classifyProbWithArray(const float * stroke_data, int stroke_data_size,
												 int * map_array_size)
{
	// reconstruct the strokes from an array of point data
	StlStrokeList strokes = arrayToStrokes(stroke_data, stroke_data_size);
	print_gesture(strokes, cout);
	// post-process the stroke
	StlStroke flattened_stroke = flatten(strokes);
	// extract default features from the processed stroke
	StlFeatureVec features = extract_features(flattened_stroke);
	// add any in-recogniser features to the sample
	StlFeatureVec updated_features = add_features(features);
	print_features(updated_features, cout);

	// predict the probability of the sample belonging to each
	// of the target classes
	StlClassProbabilities prob_map;
	svm_model.classify_with_leaf(updated_features, T::unscaled, prob_map);
	// convert the probability map into a 1D array and return this array
	return probMapToArray(prob_map, map_array_size);
}

// Function to convert a serialized version of strokes (i.e. 1D array of data) into a
// StlStrokeList object.  
template<class T>
StlStrokeList svm_recogniser<T>::arrayToStrokes(const float * stroke_data,
												int stroke_data_size)
{
	int curr_pos = 0;
	// the converted stroke
	StlStrokeList strokes;

	// while there is more stroke data to convert
	while(curr_pos < stroke_data_size)
	{
		// convert the next stroke and store it in the stroke list
		strokes.push_back(arrayToStroke(curr_pos, stroke_data, stroke_data_size));
		// skip over the end-of-stroke delimiter
		curr_pos++;
	}
	
	return strokes;
}

// Function convert a serialized version of a single stroke (i.e. 1D array of data) into
// am StlStroke object. This function assumes that stroke points are serialized with
// attribute order (x, y, milliTime) and that strokes are delimited by -1.
template<class T>
StlStroke svm_recogniser<T>::arrayToStroke(int & curr_pos, const float * stroke_data,
										   int stroke_data_size)
{
	StlStroke stroke;

	// for every serialized point (x, y, milliTime)
	while(stroke_data[curr_pos] != -1 && curr_pos < stroke_data_size)
	{
		// reconstruct an StlPoint
		StlStrokePoint curr_point;
		curr_point.x = stroke_data[curr_pos];
		curr_point.y = stroke_data[curr_pos + 1];
		curr_point.milliTime = stroke_data[curr_pos + 2];
		// add the point to the end of the stroke
		stroke.push_back(curr_point);
		// move to the next triplet of point information
		curr_pos += 3;
	}

	return stroke;
}

// Function to convert a map of class predictions and prediction probabilities into a
// serialized array form.
template<class T>
float * svm_recogniser<T>::probMapToArray(const StlClassProbabilities & probMap, int * map_array_size)
{
	int index = 0;
	// allocate the serialized version of the map
	float * serializedMap = new float[probMap.size() * 2];
	StlClassProbabilities::const_iterator iter;
	// each entry in the map has two parts: the class ID and the prediction probability
	*map_array_size = probMap.size() * 2;
		
	// for every class prediction
	for(iter = probMap.begin(); iter != probMap.end(); iter++)
	{
		// set the array element values
		serializedMap[index] = static_cast<float>(iter->first);
		serializedMap[index + 1] = static_cast<float>(iter->second);
		index += 2;
	}
	
	return serializedMap;
}

// Function returns a feature vector which is formed using the built-in features
// and added the in-recogniser features.
template<class T>
StlFeatureVec svm_recogniser<T>::add_features(const StlFeatureVec & old_features)
{
	// create a new feature vector using the in-built features as a basis
	StlFeatureVec updated_features = old_features;
#if 0
	// add the in-recogniser features
	updated_features.push_back(static_cast<double>(f_num_strokes));		// 1
	updated_features.push_back(f_endpoint_distance);					// 2
	updated_features.push_back(f_endpoint_angle);						// 3
	updated_features.push_back(f_penup_time);							// 4
#endif
	return updated_features;	
}

// Function to write the classifier model to file.
template<class T>
bool svm_recogniser<T>::writeModelFile(const string & filePath)
{
	return svm_model.save(filePath);
}

// Function to read the classifier model from file.
template<class T>
bool svm_recogniser<T>::readModelFile(const string & filePath)
{
	bool result = svm_model.load(filePath);
	svm_model.print();
	return result;
}

// Function to flatten a multi-stroke gesture.
template<class T>
StlStroke svm_recogniser<T>::flatten(const StlStrokeList & strokes)
{
	int total_points = 0;
	// the flattened stroke
	StlStroke flattened_stroke;
	// the list of processed strokes
	StlStrokeList processed_strokelist = strokes;
	StlStrokeList::const_iterator iter;

//	cout << "Unflattened stroke..." << endl;
//	print_gesture(strokes);
	
#ifdef ESTIMATE_GESTURELAB_RESOLUTION
	estimate_sampling_rate(strokes);
#endif

#if 0
	// for each stroke, get a reversed stroke if necessary
	for(iter = strokes.begin(); iter != strokes.end(); iter++)
	{
		StlStroke preprocessed_stroke = reverse(*iter, REVERSAL_DELTA);
		processed_strokelist.push_back(preprocessed_stroke);
	}
	// re-order the strokes after the reversal stage
	processed_strokelist = reorder(processed_strokelist);
#endif

	// preprocess re-ordered strokes and combine into one stroke
	for(iter = processed_strokelist.begin(); iter != processed_strokelist.end(); iter++)
	{
		// must have at least one point in the stroke for
		// preprocessing to work
		assert((*iter).size() > 0);
	
		StlStroke preprocessed_stroke = *iter;

//		preprocessed_stroke = resolution_resample(preprocessed_stroke);
//		cout << "Resolution resampled stroke with rate: " 
//			 << GESTURELAB_TIME_BETWEEN_SAMPLES << endl;
//		print_stroke(preprocessed_stroke);

		preprocessed_stroke = remove_duplicates(preprocessed_stroke);
		preprocessed_stroke = smooth(preprocessed_stroke);
		preprocessed_stroke = remove_jitter(preprocessed_stroke);
		preprocessed_stroke = equidistant_resample(preprocessed_stroke);
		// TODO: debug hooklet removal
//		preprocessed_stroke = remove_hooklet(preprocessed_stroke);

		// copy the preprocessed stroke points into the flattened stroke
		copy(preprocessed_stroke.begin(), preprocessed_stroke.end(), 
			 back_inserter(flattened_stroke));
			 
		total_points += (*iter).size();
	}

#if 0
	f_num_strokes = strokes.size();
	f_endpoint_distance = average_endpoint_distance(strokes);
	f_endpoint_angle = average_endpoint_angle(strokes);
	f_penup_time = portion_pen_up(strokes);
#endif

#if 0
	cout << "Average speed: " 
		 << total_points / (processed_strokelist.back().back().milliTime / 1000.0)
		 << " points per second." << endl;
#endif

//	cout << "Flattened stroke..." << endl;
//	print_stroke(flattened_stroke);

#ifdef DEBUG_RECOGNIZER
	//// this code writes tab-delimited debug information to file ////

	// extract default features from the processed stroke
	StlFeatureVec raw_features = extract_features(flattened_stroke);
	// scale the features since this is what the recognizer
	// uses for training and testing classification
	StlFeatureVec scaled_features = svm_model.scale_sample(raw_features);
	StlFeatureVec::const_iterator feature_iter;
	// classify the flattened strokes
	StlClassProbabilities class_probs = classify(raw_features);
	
	debug_file << gesture_num << "\t";	
	// output the recognizer prediction
	debug_file << find_mostlikely_class(class_probs) << "\t";
	// output each of the feature values
	for(feature_iter = scaled_features.begin(); feature_iter != scaled_features.end();
		feature_iter++)
		debug_file << *feature_iter << "\t";
	debug_file << endl;
	
	// one more gesture has been processed
	gesture_num++;
#endif

    return flattened_stroke;
}

#ifdef ESTIMATE_GESTURELAB_RESOLUTION
// Function estimates the median time between samples over a number of strokes.
template<class T>
void svm_recogniser<T>::estimate_sampling_rate(const StlStrokeList & strokes)
{
	vector<int> times_between_points;
	StlStrokeList::const_iterator strokes_iter;
	// for each stroke in the gesture
	for(strokes_iter = strokes.begin(); strokes_iter != strokes.end(); strokes_iter++)
	{
		// get the current stroke
		StlStroke curr_stroke = *strokes_iter;
		StlStroke::const_iterator stroke_iter = curr_stroke.begin();
		// set the first point
		StlStrokePoint prev_point = *stroke_iter;
		// and move onto the second as a starting point for iteration
		for(stroke_iter++; stroke_iter != curr_stroke.end(); stroke_iter++)
		{
			times_between_points.push_back((*stroke_iter).milliTime - prev_point.milliTime);
			prev_point = *stroke_iter;
		}
	}
	// sort the vector of times elapsed
	sort(times_between_points.begin(), times_between_points.end());
	// output the mid-point of the sorted vector
	if(times_between_points.size() > 0)
		cout << "*-* Median sampling time over " << times_between_points.size() << " points:\t" 
			 << times_between_points[times_between_points.size() / 2] << endl;
}
#endif

#ifdef DEBUG_RECOGNIZER
// Given a set of predicition probabilities, this function returns the class ID which has the
// highest probability.
template<class T>
int svm_recogniser<T>::find_mostlikely_class(const StlClassProbabilities & class_probs)
{
	int mostlikely_class_id = 0;
	StlClassProbabilityT mostlikely_class_prob = 0.0;
	
	// only search through the probability map if it has been setup by 
	// classification
	if(class_probs.size() > 0)
	{
		StlClassProbabilities::const_iterator iter = class_probs.begin();
		// assume the first class has the highest probability
		mostlikely_class_id = iter->first;
		mostlikely_class_prob = iter->second;
		// linearly search through the other classes
		for(iter++; iter != class_probs.end(); iter++)
		{
			// adjust result if a new highest probability is found
			if(iter->second > mostlikely_class_prob)
			{
				mostlikely_class_id = iter->first;
				mostlikely_class_prob = iter->second;
			}
		}
	}
	
	return mostlikely_class_id;
}
#endif

// Function to erase every one in every SUBSAMPLE_RATE points of a gesture.  This function is used
// to get more closely matched stroke resolution between GestureLab and Dunnart.
template<class T>
StlStroke svm_recogniser<T>::sub_sample(const StlStroke & stroke)
{
	// points processed thus far
	int point_num = 1;
	StlStroke subsampled_stroke;
	StlStroke::const_iterator iter;
	
	// for every point in the current stroke
	for(iter = stroke.begin(); iter != stroke.end(); iter++, point_num++)
	{
		// remove the current point if appropriate
		if(point_num % SUBSAMPLE_RATE != 0)
			subsampled_stroke.push_back(*iter);
	}
	
	print_stroke(subsampled_stroke, cout);
	return subsampled_stroke;
}

// Function to reverse a single stroke if necessary.
template<class T>
StlStroke svm_recogniser<T>::reverse(const StlStroke & stroke,
									 const long double & delta)
{
	// the resultant stroke (assume the stroke will not be reversed)
	StlStroke result = stroke;
	// the type of stroke is used to determine whether it must be reversed
	StrokeType type;
	long double R_x, R_y, diagonal_length;
	
	// create a bounding box around the stroke
	bounding_box bbox(stroke);
	
	// find the bound box's diagonal length
	diagonal_length = bbox.diagonal_length();
	// find the ratio of (last - first point) over the bounding box diagonal
	R_x = (stroke.back().x - stroke.front().x) / diagonal_length;
	R_y = (stroke.back().y - stroke.front().y) / diagonal_length;
	
//	cout << "R_x = " << R_x << ", R_y = " << R_y << endl;
	
	// find the stroke type using these ratios and a parameter threshold delta
	type = find_stroke_type(R_x, R_y, delta);
	
	// if the stroke should be reversed, reverse it
	if( (type == horizontal && stroke.back().x < stroke.front().x) ||
		(type == vertical   && stroke.back().y < stroke.front().y) ||
		(type == diagonal   && stroke.back().y < stroke.front().y) )
	{
//		cout << "Reversing this stroke." << endl << endl;
		result = reverse_points(stroke);
	}

	return result;
}

// Function to determine a stroke's type given two indicative ratios and a 
// threshold parameter delta.
template<class T>
typename svm_recogniser<T>::StrokeType svm_recogniser<T>::find_stroke_type(
														  const long double & R_x, 
														  const long double & R_y,
														  const long double & delta)
{
	// the stroke is closed
	if(R_x < delta && R_y < delta)
		return closed;
	// the stroke is horizontal
	else if(R_x >= delta && R_y < delta)
		return horizontal;
	// the stroke is vertical
	else if(R_x < delta && R_y >= delta)
		return vertical;
	// the stroke is diagonal
	else if(R_x >= delta && R_y >= delta)
		return diagonal;
	// default stroke is closed, but shouldnt reach here
	else
		return closed;
}

// Function to reverse the coordinates of points in a stroke whilst
// maintaining the millitime data.
template<class T>
StlStroke svm_recogniser<T>::reverse_points(const StlStroke & stroke)
{
	int i;
	// the resultant stroke
	StlStroke result = stroke;
	// iterators point to the start and end of the stroke
	StlStroke::iterator front_iter, back_iter;
	
	// must have at least two points to do the reversal
	if(stroke.size() < 2)
		return result;
		
	// for each point to swap
	for(i = 0, front_iter = result.begin(), back_iter = result.end(); 
		i < result.size() / 2;
		i++, front_iter++, back_iter--)
	{
		// swap the coordinates and pressure details but keep
		// timing information in order
		swap_coordinates(*front_iter, *back_iter);
	}

	return result;
}

// Function to swap all details of twp stroke points _except_ the
// timing information.  This is useful when reversing a stroke.
template<class T>
void svm_recogniser<T>::swap_coordinates(StlStrokePoint & first_point,
										 StlStrokePoint & second_point)
{
	// store a temporary copy of the first stroke
	StlStrokePoint temp_point = first_point;
	// copy the second stroke's details into the first stroke
	first_point.x = second_point.x;
	first_point.y = second_point.y;
	first_point.pressure = second_point.pressure;
	// copy the first stroke's details into the second stroke
	second_point.x = temp_point.x;
	second_point.y = temp_point.y;
	second_point.pressure = temp_point.pressure;
}

// Function to re-order a list of strokes according to the angle between
// the upper segment of the bounding box and the segment limited by the upper
// left corner of the bounding box and the last point of the stroke.
template<class T>
StlStrokeList svm_recogniser<T>::reorder(const StlStrokeList & strokes)
{
	// the re-ordered list of strokes
	StlStrokeList result;
	StlStrokeList::const_iterator list_iter;
	// map each stroke to the angle it forms with the bounding box top-edge
	map<long double, StlStroke> stroke_map;
	map<long double, StlStroke>::const_iterator map_iter;
	
	// dont bother attempting to reorder less than two strokes
	if(strokes.size() < 2)
		return strokes;

	// for each stroke
	for(list_iter = strokes.begin(); list_iter != strokes.end(); list_iter++)
	{
		// create a bounding box and calculate the top-edge angle
		bounding_box bbox(*list_iter);
		// storing in a map means that entries are sorted in ascending order
		// according to the top-edge angle
		stroke_map[bbox.top_edge_angle(*list_iter)] = *list_iter;
	}
	
	// add the strokes to a new stroke list in order
	for(map_iter = stroke_map.begin(); map_iter != stroke_map.end(); map_iter++)
		result.push_back(map_iter->second);
	
	return result;
}

// Function to resize a stroke to unit dimensions.
template<class T>
StlStroke svm_recogniser<T>::unit_resize(const StlStroke & the_stroke)
{
	double r, mean_x, mean_y, cumul_r = 0.0;
	// the points of the resized stroke
	StlStrokePoint new_point;
	// the resultant stroke
	StlStroke resized_stroke;
	StlStroke::const_iterator iter;
	// get the average x and y values of the original stroke
	find_coordinate_means(the_stroke, mean_x, mean_y);
	// calculate the sum of squared differences
	for(iter = the_stroke.begin(); iter != the_stroke.end(); iter++)
		cumul_r += (mean_x - (*iter).x) * (mean_x - (*iter).x) +
				   (mean_y - (*iter).y) * (mean_y - (*iter).y);
	// calculate r by normalizing over the number of points
	r = sqrt(cumul_r / the_stroke.size());
	// for each point in the original stroke
	for(iter = the_stroke.begin(); iter != the_stroke.end(); iter++)
	{
		// create a new point and add it to the new stroke
		new_point.x = (mean_x - (*iter).x) / r;
		new_point.y = (mean_y - (*iter).y) / r;
		new_point.milliTime = (*iter).milliTime;
		resized_stroke.push_back(new_point);
	}

	return resized_stroke;
}

// Function to remove duplicate adjacent points in a stroke.
template<class T>
StlStroke svm_recogniser<T>::remove_duplicates(const StlStroke & stroke)
{
	// the resultant stroke without duplicates
	StlStroke result;
	// the previous point in the stroke
	StlStrokePoint prev_point;
	StlStroke::const_iterator iter = stroke.begin();
	long int iteration = 1;
	// store the first point in the stroke
	result.push_back(*iter);
	prev_point = *iter;
	
	// for each remaining point in the stroke
	for(iter++; iter != stroke.end(); iter++, iteration++)
	{
		// if the current point's coordinates differ from the previous
		// point's coordinates, add the current point to the new stroke
		if( (*iter).x != prev_point.x || (*iter).y != prev_point.y)
			result.push_back(*iter);
			
		// upate the previous point prior to the next iteration
		prev_point = *iter;
	}
	
	return result;
}

// Function to remove hooks (if present) from the beginning and end
// of given stroke.
template<class T>
StlStroke svm_recogniser<T>::remove_hooklet(const StlStroke & stroke)
{
	StlStroke processed_stroke;
	StlStroke::const_iterator head_endpoint, tail_startpoint;
	bool start_hook = false, end_hook = false;
	
	if(stroke.size() < MIN_HOOK_POINTS)
		return stroke;
		
	set_head_tail(stroke, head_endpoint, tail_startpoint);
	
	// -> check the start of the stroke first
	// check whether a direction-change type hook is present @ stroke start
	if(direction_change_hook(stroke, stroke.begin()))
		start_hook = true;
	// -> repeat the hooklet detection at the end of the stroke
	// check whether a direction-change type hook is present @ stroke end
	if(direction_change_hook(stroke, tail_startpoint))
		end_hook = true;

	// hooklet at the start of the stroke but not the end
	if(start_hook && !end_hook)
		copy(head_endpoint, stroke.end(), back_inserter(processed_stroke));	
	// hooklet at the start and end of the stroke
	else if(start_hook && end_hook)
		copy(head_endpoint, tail_startpoint, back_inserter(processed_stroke));
	// hooklet at the end of the stroke but not at the start
	else if(!start_hook && end_hook)
		copy(stroke.begin(), tail_startpoint, back_inserter(processed_stroke));
	// no hooklet at the start and no hooklet at the end
	else
		return stroke;

	return processed_stroke;
}

// Function to set the limit of the region in which a leading and trailing
// hook may occur.
template<class T>
void svm_recogniser<T>::set_head_tail(const StlStroke & the_stroke,
									  StlStroke::const_iterator & head_endpoint,
									  StlStroke::const_iterator & tail_startpoint)
{
	int i;
	// start the leading region at the start of the stroke
	head_endpoint = the_stroke.begin();
	// start the training region to the point that is one past the last point
	// of the stroke (and subsequently work backwards)
	tail_startpoint = the_stroke.end();
	// advance to the end of the leading region
	for(i = 0; i < 4; i++)
		head_endpoint++;
	// work backwards from the end (5 steps not 4 steps since the_stroke.end() returns
	// an iterator which is one point *past* the last stroke point)
	for(i = 0; i < 5; i++)
		tail_startpoint--;
}

// Function to detect a hook based on a large change in direction which occurs
// over a small distance and timespan.
template<class T>
bool svm_recogniser<T>::direction_change_hook(const StlStroke & stroke,
											  const StlStroke::const_iterator & start_iter)
{
	double angle = 0.0;
	// assume no hook is present and check for hook existance
	bool hook_detected = false;
	// five points in the hook region
	StlStrokePoint p1, p2, p3, p4, p5;	
	StlStroke::const_iterator iter = start_iter;
	
	// set the local neighbourhood of points
	p1 = *iter; iter++; p2 = *iter; iter++;
	p3 = *iter; iter++; p4 = *iter; iter++;
	p5 = *iter;
	// calculate the accumulative angle
	angle = (180 - inside_angle(p1, p2, p3)) +
			(180 - inside_angle(p2, p3, p4)) +
			(180 - inside_angle(p3, p4, p5));
	// if the angle is large enough to indicate a hook
	if(angle > MIN_HOOK_ANGLE)
		hook_detected = true;
	
	return hook_detected;
}

// Function to remove jitter from a given stroke.  Jitter occurs when an input
// point occurs within JITTER_DISTANCE pixels of the previous input point.
template<class T>
StlStroke svm_recogniser<T>::remove_jitter(const StlStroke & stroke)
{
	// the resultant stroke without jitter
	StlStroke processed_stroke;
	StlStroke::const_iterator iter = stroke.begin();
	// current and previous stroke points
	StlStrokePoint prev_point;
	
	// if there are less than two points in the stroke, return an
	// unaltered stroke
	if(stroke.size() < 2)
		return stroke;
	
	// extract the first point in the stroke and store it in the
	// new stroke
	prev_point = *iter;
	processed_stroke.push_back(prev_point);
	
	// for each remaining point in the stroke
	for(iter++; iter != stroke.end(); iter++)
	{
		// extract the current point
		StlStrokePoint curr_point = *iter;
		// calculate the distance between the current and
		// previous points
		double curr_distance = hypot(curr_point.x - prev_point.x, 
									 curr_point.y - prev_point.y);
				 
		// if the distance is excedes the jitter threshold 
		if(curr_distance > JITTER_DISTANCE)
		{
			// store the current point and update the previous point
			// for the next iteration
			processed_stroke.push_back(curr_point);
			prev_point = curr_point;
		}
	}
	
	return processed_stroke;
}

// Function to smooth a single stroke by averaging each point with its previous
// and next neighbours.  The smoothing is performed using weights (0.25, 0.5, 0.25)
// for stroke points at indices (i-1, i, i+1).
template<class T>
StlStroke svm_recogniser<T>::smooth(const StlStroke & stroke)
{
	// the resultant stroke
	StlStroke result;
	// the new, previous, current and next points in the stroke
	StlStrokePoint prev_point, curr_point, next_point;
	StlStroke::const_iterator next_iter, iter = stroke.begin();
	long int iteration = 1;
	
	// must have at least three points in the stroke for this
	// function to work
	if(stroke.size() < 3)
		return stroke;

	// assume an ``imaginary'' first point with the same attribute values as
	// the actual first point
	prev_point = *iter;
	next_iter = iter;
	
	// for each remaining point in the stroke
	for(next_iter++; iter != stroke.end(); iter++, next_iter++, iteration++)
	{
		// extract the current point
		curr_point = *iter;
		
		// if we're not at the end of the stroke, extract the next point too
		if(next_iter != stroke.end())
			next_point = *next_iter;
		// else assume the next point is the same as the current point
		else
			next_point = curr_point;
		
		// smooth the x-coordinate
		curr_point.x = (0.25 * prev_point.x) + (0.5 * curr_point.x)  + 
					   (0.25 * next_point.x);
		// smooth the y-coordinate
		curr_point.y = (0.25 * prev_point.y) + (0.5 * curr_point.y)  + 
					   (0.25 * next_point.y);		
		// smooth the pressure value
		curr_point.pressure = (0.25 * prev_point.pressure) + 
							  ( 0.5 * curr_point.pressure) + 
							  (0.25 * next_point.pressure);
		// add the smoothed point to the new stroke
		result.push_back(curr_point);
		
		// upate the previous point prior to the next iteration
		prev_point = curr_point;
	}

	return result;
}

// Function to resample a gesture so that points are equally spaced using the
// Euclidean distance measure.
template<class T>
StlStroke svm_recogniser<T>::equidistant_resample(const StlStroke & stroke)
{
	// the resultant stroke
	StlStroke resampled_stroke;
	// move to the start of the stroke
	StlStroke::const_iterator iter = stroke.begin();	

	// start the process by storing the first point and moving to the next
	// point in the original stroke
	resampled_stroke.push_back(*iter);
	iter++;
	
	// for each remaining point in the stroke
	while(iter != stroke.end())
	{
		// calculate the Euclidean distance to the previous point
		double curr_distance = hypot((*iter).x - resampled_stroke.back().x,
									 (*iter).y - resampled_stroke.back().y);

		// if we have travelled far enough to add a new point to the stroke
		if(curr_distance > EQUIDISTANT_THRESHOLD)
		{
			// get the interpolated point and add it to the stroke
			StlStrokePoint interpolated_point = interpolate_by_distance(
												resampled_stroke.back(),
												*iter,
												EQUIDISTANT_THRESHOLD);

			resampled_stroke.push_back(interpolated_point);
		}
		// else we have not covered enough distance to add an interpolated point
		// so move to the next point in the original stroke
		else
			iter++;
	}
	
	return resampled_stroke;
}

// Function to resample a gesture so that points are equally spaced according to
// the average sampling rate of GestureLab.
template<class T>
StlStroke svm_recogniser<T>::resolution_resample(const StlStroke & stroke)
{
	// the resultant stroke
	StlStroke resampled_stroke;
	// move to the start of the stroke
	StlStroke::const_iterator iter = stroke.begin();	

	// start the process by storing the first point and moving to the next
	// point in the original stroke
	resampled_stroke.push_back(*iter);
	iter++;
	
	// for each remaining point in the stroke
	while(iter != stroke.end())
	{
		// calculate the time elapsed since the previous point
		double curr_elapsed_time = (*iter).milliTime - resampled_stroke.back().milliTime;

		// if enough time has elapsed to add a new point to the stroke
		if(curr_elapsed_time > GESTURELAB_TIME_BETWEEN_SAMPLES)
		{
			// get the interpolated point and add it to the stroke
			StlStrokePoint interpolated_point = interpolate_by_time(
												resampled_stroke.back(),
												*iter,
												GESTURELAB_TIME_BETWEEN_SAMPLES);

			resampled_stroke.push_back(interpolated_point);
		}
		// else not enough time has elapsed to add an interpolated point,
		// so move to the next point in the original stroke
		else
			iter++;
	}
	
	return resampled_stroke;
}

// Function creates a new StlPoint which lies on the straight line between first_point
// and second_point at Euclidean ``distance'' from the first point.
template<class T>
StlStrokePoint svm_recogniser<T>::interpolate_by_distance(const StlStrokePoint & first_point,
														  const StlStrokePoint & second_point,
														  double distance)
{
	StlStrokePoint resultant_point;
	
//	cout << "Required distance along the long edge: " << distance << endl;
	
	// calculate the angle between the two points
	double theta = atan2(second_point.y - first_point.y,
						 second_point.x -  first_point.x);
	// calculate the Euclidean distance between the two points
	double long_edge = hypot(first_point.x - second_point.x,
							 first_point.y - second_point.y);
	// calculate the time elapsed (in milliseconds) between the two points
	double time_elapsed = second_point.milliTime - first_point.milliTime;
	
	// calculate the offset from the starting point along the x-axis
	float x_offset = distance * cos(theta);
	// calculate the offset from the starting point along the y-axis
	float y_offset = distance * sin(theta);
	// set the new point's coordinates
	resultant_point.x = first_point.x + x_offset;
	resultant_point.y = first_point.y + y_offset;
	// avoid potential divide by zero before linearly interpolating the 
	// millitime value of the new point
	if(long_edge > 0)
		resultant_point.milliTime = lrint(first_point.milliTime + 
										  (distance / long_edge) * time_elapsed);
	// set default pressure (unused for now)
	resultant_point.pressure = 1;
	
	return resultant_point; 
}

// Function creates a new StlPoint which lies on the straight line between first_point
// and second_point at Euclidean ``distance'' from the first point.
template<class T>
StlStrokePoint svm_recogniser<T>::interpolate_by_time(const StlStrokePoint & first_point,
													  const StlStrokePoint & second_point,
													  double target_time_elapsed)
{
	StlStrokePoint resultant_point;
	
//	cout << "First point: " << first_point.x << ", " << first_point.y << ", " << first_point.milliTime << endl;
//	cout << "Second point: " << second_point.x << ", " << second_point.y << ", " << second_point.milliTime << endl;
	
	// calculate the angle between the two points
	double theta = atan2(second_point.y - first_point.y,
						 second_point.x -  first_point.x);
	// calculate the Euclidean distance between the two points
	double long_edge = hypot(first_point.x - second_point.x,
							 first_point.y - second_point.y);
//	cout << "Long edge: " << long_edge << endl;
								 	
	// calculate the time elapsed (in milliseconds) between the two points
	double time_elapsed = second_point.milliTime - first_point.milliTime;
//	cout << "Time elapsed: " << time_elapsed << endl;
	
	// calculate the ratio of target elapsed time to actual elapsed time
	double time_ratio;
	// avoid divide by zero which might occur due to jitter
	if(time_elapsed > 0)
		time_ratio = target_time_elapsed / time_elapsed;
	else
		time_ratio = 1.0;
		
//	cout << "Time ratio: " << time_ratio << endl;
	
	// calculate the offset from the starting point along the x-axis
	float x_offset = (time_ratio * long_edge) * cos(theta);
//	cout << "x offset: " << x_offset << endl;
	
	// calculate the offset from the starting point along the y-axis
	float y_offset = (time_ratio * long_edge) * sin(theta);
//	cout << "y offset: " << y_offset << endl;
	
	// set the new point's coordinates
	resultant_point.x = first_point.x + x_offset;
	resultant_point.y = first_point.y + y_offset;
	// set the new point's time which should be uniform in the resultant stroke
	resultant_point.milliTime = first_point.milliTime + target_time_elapsed;
	// set default pressure (unused for now)
	resultant_point.pressure = 1;
	
//	cout << resultant_point.x << ", " << resultant_point.y << ", " << resultant_point.milliTime << endl;
	
	return resultant_point; 
}

// Function to calculate the average distance between the last point of one stroke and
// the first point of the next stroke.  For each pair of strokes, this distance is
// normalised using the average relative distance of each stroke.
template<class T>
double svm_recogniser<T>::average_endpoint_distance(const StlStrokeList & strokes)
{
	// not multi-stroke so dont procede
	if(strokes.size() == 1)
		return 0;
	// multi-stroke gesture
	else
	{
		StlStroke curr_stroke, prev_stroke;
		StlStrokeList::const_iterator iter = strokes.begin();
		double result = 0.0;	
		
		prev_stroke = *iter;
		// for each pair of "adjacent" strokes
		for(iter++; iter != strokes.end(); iter++)
		{
			double distance, average_rel_distance;
			
			curr_stroke = *iter;
			// calculate the distance between the first point of the current stroke
			// and the last point of the previous stroke
			distance = hypot(curr_stroke.front().x - prev_stroke.back().x,
							 curr_stroke.front().y - prev_stroke.back().y);
			// calculate the average relative distance of the two strokes
			average_rel_distance = (calc_relative_length(prev_stroke) + 
								    calc_relative_length(curr_stroke)) / 2.0; 
			// normalise the distance between the start and end points
			result += distance / average_rel_distance;
			prev_stroke = curr_stroke;
		}
		// return the average
		return result / strokes.size();
	}
}

// Function to calculate the average angle between the first point of the current
// stroke and the last point of the previous stroke.
template<class T>
double svm_recogniser<T>::average_endpoint_angle(const StlStrokeList & strokes)
{							
	// not multi-stroke so dont procede
	if(strokes.size() == 1)
		return 0;
	// multi-stroke gesture
	else
	{
		StlStroke curr_stroke, prev_stroke;
		StlStrokeList::const_iterator iter = strokes.begin();
		double result = 0.0;	
		
		prev_stroke = *iter;
		// for each pair of "adjacent" strokes
		for(iter++; iter != strokes.end(); iter++)
		{
			curr_stroke = *iter;
			// accumulate the angle between the first point of the current stroke
			// and the last point of the previous stroke
			result += atan2(curr_stroke.front().y - prev_stroke.back().y,
							curr_stroke.front().x - prev_stroke.back().x);
							
			prev_stroke = curr_stroke;
		}
		// return the average angle between endpoints
		return result / strokes.size();
	}
}

// Function to calculate the ratio of pen-up time to the time taken to draw all
// strokes in the StrokeList.
template<class T>
double svm_recogniser<T>::portion_pen_up(const StlStrokeList & strokes)
{								
	// not multi-stroke so dont procede
	if(strokes.size() == 1)
		return 0;
	// multi-stroke gesture
	else
	{
		// portion of pen-up time, total time elapsed for the gesture
		double portion_uptime = 0.0, total_time = strokes.back().back().milliTime;
		StlStrokeList::const_iterator iter = strokes.begin();
		StlStroke curr_stroke, prev_stroke = *iter;
	
		// for each pair of "adjacent" strokes
		for(iter++; iter != strokes.end(); iter++)
		{
			curr_stroke = *iter;
			// accumulate the pen-up time between strokes
			portion_uptime += (curr_stroke.front().milliTime - 
							   prev_stroke.back().milliTime);
	
			prev_stroke = curr_stroke;
		}
		// check for divide by zero before returning the ratio
		if(total_time != 0)
			return portion_uptime / total_time;
		else
			return 0;
	}
}

#endif  // !SVM_RECOGNISER_IPP
